<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\Employee;
use App\Models\SalaryComponent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class SalaryComponentController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = SalaryComponent::where('company_id', $companyId);

        // Apply filters if provided
        if ($request->has('type') && $request->type != '') {
            $query->where('type', $request->type);
        }

        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('name_en', 'like', "%{$search}%")
                    ->orWhere('code', 'like', "%{$search}%");
            });
        }

        $components = $query->orderBy('type')
            ->orderBy('name')
            ->paginate(10);

        return view('hrm.salary_components.index', compact('components'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('hrm.salary_components.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'code' => 'nullable|string|max:50',
            'type' => 'required|in:allowance,deduction',
            'is_taxable' => 'boolean',
            'is_fixed' => 'boolean',
            'default_amount' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');

        $component = new SalaryComponent();
        $component->company_id = $companyId;
        $component->name = $request->name;
        $component->name_en = $request->name_en;
        $component->code = $request->code;
        $component->type = $request->type;
        $component->is_taxable = $request->has('is_taxable');
        $component->is_fixed = $request->has('is_fixed');
        $component->default_amount = $request->default_amount ?? 0;
        $component->description = $request->description;
        $component->status = $request->status;
        $component->created_by = Auth::id();
        $component->save();

        return redirect()->route('hrm.salary_components.index')
            ->with('success', 'Salary component created successfully.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $component = SalaryComponent::findOrFail($id);

        // Check if component belongs to the current company
        $companyId = session('company_id');
        if ($component->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Get employees using this component
        $employees = $component->employees()->paginate(10);

        return view('hrm.salary_components.show', compact('component', 'employees'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $companyId = $request->session()->get('company_id');
        $component = SalaryComponent::findOrFail($id);

        // Check if component belongs to the current company
        if ($component->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        return view('hrm.salary_components.edit', compact('component'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $component = SalaryComponent::findOrFail($id);

        // Check if component belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($component->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'code' => 'nullable|string|max:50',
            'type' => 'required|in:allowance,deduction',
            'is_taxable' => 'boolean',
            'is_fixed' => 'boolean',
            'default_amount' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $component->name = $request->name;
        $component->name_en = $request->name_en;
        $component->code = $request->code;
        $component->type = $request->type;
        $component->is_taxable = $request->has('is_taxable');
        $component->is_fixed = $request->has('is_fixed');
        $component->default_amount = $request->default_amount ?? 0;
        $component->description = $request->description;
        $component->status = $request->status;
        $component->updated_by = Auth::id();
        $component->save();

        return redirect()->route('hrm.salary_components.index')
            ->with('success', 'Salary component updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        $component = SalaryComponent::findOrFail($id);

        // Check if component belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($component->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Check if component is being used in any salary records
        if ($component->salaries()->exists()) {
            return redirect()->route('hrm.salary_components.index')
                ->with('error', 'Cannot delete component as it is being used in salary records.');
        }

        // Check if component is assigned to any employees
        if ($component->employees()->exists()) {
            return redirect()->route('hrm.salary_components.index')
                ->with('error', 'Cannot delete component as it is assigned to employees.');
        }

        $component->delete();

        return redirect()->route('hrm.salary_components.index')
            ->with('success', 'Salary component deleted successfully.');
    }

    /**
     * Assign component to multiple employees.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function assignToEmployees(Request $request)
    {
        $companyId = $request->session()->get('company_id');

        // Get all active components
        $components = SalaryComponent::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('type')
            ->orderBy('name')
            ->get();

        // Get all active employees
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('name')
            ->get();

        // Get all departments for filter
        $departments = Department::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('name')
            ->get();

        return view('hrm.salary_components.assign', compact('components', 'employees', 'departments'));
    }

    /**
     * Process component assignment to employees.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function processAssignment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'component_id' => 'required|exists:salary_components,id',
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:employees,id',
            'amount' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');
        $componentId = $request->component_id;
        $employeeIds = $request->employee_ids;
        $amount = $request->amount;

        // Check if component belongs to the company
        $component = SalaryComponent::findOrFail($componentId);
        if ($component->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        $successCount = 0;
        $skipCount = 0;

        foreach ($employeeIds as $employeeId) {
            // Check if employee belongs to the company
            $employee = Employee::findOrFail($employeeId);
            if ($employee->company_id != $companyId) {
                continue; // Skip this employee
            }

            // Check if component is already assigned to this employee
            $exists = $employee->salaryComponents()
                ->where('salary_component_id', $componentId)
                ->exists();

            if ($exists) {
                // Update the existing assignment
                $employee->salaryComponents()->updateExistingPivot($componentId, [
                    'amount' => $amount,
                    'updated_by' => Auth::id(),
                ]);
            } else {
                // Create a new assignment
                $employee->salaryComponents()->attach($componentId, [
                    'amount' => $amount,
                    'created_by' => Auth::id(),
                ]);

                $successCount++;
            }
        }

        $message = "Component assigned to {$successCount} employees successfully.";
        if ($skipCount > 0) {
            $message .= " {$skipCount} employees were updated.";
        }

        return redirect()->route('hrm.salary_components.show', $componentId)
            ->with('success', $message);
    }

    /**
     * Remove component assignment from an employee.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $componentId
     * @param  int  $employeeId
     * @return \Illuminate\Http\Response
     */
    public function removeAssignment(Request $request, $componentId, $employeeId)
    {
        $companyId = $request->session()->get('company_id');

        // Check if component belongs to the company
        $component = SalaryComponent::findOrFail($componentId);
        if ($component->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Check if employee belongs to the company
        $employee = Employee::findOrFail($employeeId);
        if ($employee->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Remove the assignment
        $employee->salaryComponents()->detach($componentId);

        return redirect()->route('hrm.salary_components.show', $componentId)
            ->with('success', 'Component removed from employee successfully.');
    }
}
