<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Employee;
use App\Models\Payroll;
use App\Models\PayrollItem;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PayrollController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = Payroll::where('company_id', $companyId);

        // Apply filters if provided
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('month') && $request->month != '') {
            $query->where('payroll_period', 'like', "%{$request->month}%");
        }

        if ($request->has('year') && $request->year != '') {
            $query->where('payroll_period', 'like', "%{$request->year}%");
        }

        $payrolls = $query->orderBy('created_at', 'desc')->paginate(10);

        return view('hrm.payroll.index', compact('payrolls'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $companyId = $request->session()->get('company_id');

        // Get current month and year for default values
        $currentMonth = date('m');
        $currentYear = date('Y');

        // Get list of months for dropdown
        $months = [
            '01' => 'January',
            '02' => 'February',
            '03' => 'March',
            '04' => 'April',
            '05' => 'May',
            '06' => 'June',
            '07' => 'July',
            '08' => 'August',
            '09' => 'September',
            '10' => 'October',
            '11' => 'November',
            '12' => 'December',
        ];

        // Get list of years (current year and 5 years back)
        $years = [];
        for ($i = 0; $i <= 5; $i++) {
            $year = date('Y') - $i;
            $years[$year] = $year;
        }

        return view('hrm.payroll.create', compact('months', 'years', 'currentMonth', 'currentYear'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'month' => 'required|string',
            'year' => 'required|string',
            'payment_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');

        // Format month name
        $monthName = date('F', mktime(0, 0, 0, $request->month, 10));
        $payrollPeriod = $monthName . ' ' . $request->year;

        // Calculate start and end dates for the month
        $startDate = Carbon::createFromDate($request->year, $request->month, 1)->startOfMonth();
        $endDate = Carbon::createFromDate($request->year, $request->month, 1)->endOfMonth();

        // Check if payroll already exists for this period
        $existingPayroll = Payroll::where('company_id', $companyId)
            ->where('payroll_period', $payrollPeriod)
            ->first();

        if ($existingPayroll) {
            return redirect()->back()
                ->with('error', 'Payroll already exists for this period.')
                ->withInput();
        }

        // Create new payroll
        $payroll = new Payroll();
        $payroll->company_id = $companyId;
        $payroll->payroll_period = $payrollPeriod;
        $payroll->start_date = $startDate;
        $payroll->end_date = $endDate;
        $payroll->payment_date = $request->payment_date;
        $payroll->status = 'draft';
        $payroll->notes = $request->notes;
        $payroll->created_by = Auth::id();
        $payroll->save();

        return redirect()->route('hrm.payroll.show', $payroll->id)
            ->with('success', 'Payroll created successfully. You can now add employees to this payroll.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $payroll = Payroll::with(['payrollItems.employee'])->findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = session('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        return view('hrm.payroll.show', compact('payroll'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $companyId = $request->session()->get('company_id');
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only draft payrolls can be edited
        if ($payroll->status != 'draft') {
            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Only draft payrolls can be edited.');
        }

        return view('hrm.payroll.edit', compact('payroll'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only draft payrolls can be updated
        if ($payroll->status != 'draft') {
            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Only draft payrolls can be updated.');
        }

        $validator = Validator::make($request->all(), [
            'payment_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $payroll->payment_date = $request->payment_date;
        $payroll->notes = $request->notes;
        $payroll->updated_by = Auth::id();
        $payroll->save();

        return redirect()->route('hrm.payroll.show', $id)
            ->with('success', 'Payroll updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only draft payrolls can be deleted
        if ($payroll->status != 'draft') {
            return redirect()->route('hrm.payroll.index')
                ->with('error', 'Only draft payrolls can be deleted.');
        }

        // Delete all payroll items first
        $payroll->payrollItems()->delete();

        // Delete the payroll
        $payroll->delete();

        return redirect()->route('hrm.payroll.index')
            ->with('success', 'Payroll deleted successfully.');
    }

    /**
     * Generate payroll items for all employees.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function generatePayroll(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only draft payrolls can be generated
        if ($payroll->status != 'draft') {
            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Only draft payrolls can be generated.');
        }

        // Get all active employees
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Begin transaction
        DB::beginTransaction();

        try {
            // Clear existing payroll items
            $payroll->payrollItems()->delete();

            $totalBasicSalary = 0;
            $totalAllowances = 0;
            $totalDeductions = 0;
            $totalNetSalary = 0;

            foreach ($employees as $employee) {
                // Get attendance data for the payroll period
                $attendanceData = $this->getAttendanceData($employee->id, $payroll->start_date, $payroll->end_date);

                // Calculate salary components
                $basicSalary = $employee->basic_salary;

                // Calculate allowances (example: housing, transportation, etc.)
                $allowances = [
                    'housing' => $employee->housing_allowance ?? 0,
                    'transportation' => $employee->transportation_allowance ?? 0,
                    'phone' => $employee->phone_allowance ?? 0,
                    'other' => $employee->other_allowance ?? 0,
                ];
                $totalAllowanceAmount = array_sum($allowances);

                // Calculate deductions
                $deductions = [
                    'tax' => ($basicSalary * 0.05), // Example: 5% tax
                    'insurance' => ($basicSalary * 0.02), // Example: 2% insurance
                    'late' => ($attendanceData['late_minutes'] * 0.5), // Example: 0.5 per minute late
                    'absence' => ($attendanceData['absent_days'] * ($basicSalary / 30)), // Deduct per day absent
                ];
                $totalDeductionAmount = array_sum($deductions);

                // Calculate overtime
                $overtimeAmount = $attendanceData['overtime_hours'] * ($basicSalary / 176) * 1.5; // Assuming 176 working hours per month and 1.5x overtime rate

                // Calculate net salary
                $netSalary = $basicSalary + $totalAllowanceAmount + $overtimeAmount - $totalDeductionAmount;

                // Create payroll item
                $payrollItem = new PayrollItem();
                $payrollItem->payroll_id = $payroll->id;
                $payrollItem->employee_id = $employee->id;
                $payrollItem->basic_salary = $basicSalary;
                $payrollItem->total_allowances = $totalAllowanceAmount;
                $payrollItem->total_deductions = $totalDeductionAmount;
                $payrollItem->net_salary = $netSalary;
                $payrollItem->allowances = $allowances;
                $payrollItem->deductions = $deductions;
                $payrollItem->working_days = $attendanceData['working_days'];
                $payrollItem->leave_days = $attendanceData['leave_days'];
                $payrollItem->absent_days = $attendanceData['absent_days'];
                $payrollItem->overtime_hours = $attendanceData['overtime_hours'];
                $payrollItem->overtime_amount = $overtimeAmount;
                $payrollItem->late_minutes = $attendanceData['late_minutes'];
                $payrollItem->late_deduction = $deductions['late'];
                $payrollItem->save();

                // Add to totals
                $totalBasicSalary += $basicSalary;
                $totalAllowances += $totalAllowanceAmount;
                $totalDeductions += $totalDeductionAmount;
                $totalNetSalary += $netSalary;
            }

            // Update payroll totals
            $payroll->total_basic_salary = $totalBasicSalary;
            $payroll->total_allowances = $totalAllowances;
            $payroll->total_deductions = $totalDeductions;
            $payroll->total_net_salary = $totalNetSalary;
            $payroll->status = 'processing';
            $payroll->updated_by = Auth::id();
            $payroll->save();

            DB::commit();

            return redirect()->route('hrm.payroll.show', $id)
                ->with('success', 'Payroll generated successfully for ' . $employees->count() . ' employees.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Error generating payroll: ' . $e->getMessage());
        }
    }

    /**
     * Get attendance data for an employee within a date range.
     *
     * @param  int  $employeeId
     * @param  \Carbon\Carbon  $startDate
     * @param  \Carbon\Carbon  $endDate
     * @return array
     */
    private function getAttendanceData($employeeId, $startDate, $endDate)
    {
        // Get all attendance records for the employee within the date range
        $attendanceRecords = Attendance::where('employee_id', $employeeId)
            ->whereBetween('date', [$startDate, $endDate])
            ->get();

        // Calculate working days (excluding weekends)
        $workingDays = 0;
        $currentDate = $startDate->copy();
        while ($currentDate <= $endDate) {
            // Skip weekends (Friday and Saturday in some countries, or Saturday and Sunday)
            if ($currentDate->dayOfWeek !== Carbon::FRIDAY && $currentDate->dayOfWeek !== Carbon::SATURDAY) {
                $workingDays++;
            }
            $currentDate->addDay();
        }

        // Count present days, leave days, absent days
        $presentDays = $attendanceRecords->where('status', 'present')->count();
        $lateDays = $attendanceRecords->where('status', 'late')->count();
        $halfDays = $attendanceRecords->where('status', 'half_day')->count();
        $leaveDays = $attendanceRecords->where('status', 'on_leave')->count();

        // Calculate absent days (working days - present days - leave days)
        $absentDays = $workingDays - $presentDays - $lateDays - $halfDays - $leaveDays;
        if ($absentDays < 0) $absentDays = 0;

        // Calculate total late minutes
        $lateMinutes = $attendanceRecords->sum('late_minutes');

        // Calculate total overtime hours
        $overtimeHours = $attendanceRecords->sum('overtime_minutes') / 60;

        return [
            'working_days' => $workingDays,
            'present_days' => $presentDays + $lateDays + ($halfDays * 0.5),
            'leave_days' => $leaveDays,
            'absent_days' => $absentDays,
            'late_minutes' => $lateMinutes,
            'overtime_hours' => $overtimeHours,
        ];
    }

    /**
     * Approve a payroll.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only processing payrolls can be approved
        if ($payroll->status != 'processing') {
            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Only processing payrolls can be approved.');
        }

        $payroll->status = 'approved';
        $payroll->approved_by = Auth::id();
        $payroll->approved_at = now();
        $payroll->save();

        return redirect()->route('hrm.payroll.show', $id)
            ->with('success', 'Payroll approved successfully.');
    }

    /**
     * Mark a payroll as paid.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsPaid(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only approved payrolls can be marked as paid
        if ($payroll->status != 'approved') {
            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Only approved payrolls can be marked as paid.');
        }

        $payroll->status = 'paid';
        $payroll->updated_by = Auth::id();
        $payroll->save();

        return redirect()->route('hrm.payroll.show', $id)
            ->with('success', 'Payroll marked as paid successfully.');
    }

    /**
     * Cancel a payroll.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function cancel(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only draft or processing payrolls can be cancelled
        if (!in_array($payroll->status, ['draft', 'processing'])) {
            return redirect()->route('hrm.payroll.show', $id)
                ->with('error', 'Only draft or processing payrolls can be cancelled.');
        }

        $payroll->status = 'cancelled';
        $payroll->updated_by = Auth::id();
        $payroll->save();

        return redirect()->route('hrm.payroll.show', $id)
            ->with('success', 'Payroll cancelled successfully.');
    }

    /**
     * Export payroll to PDF.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function exportPdf(Request $request, $id)
    {
        $payroll = Payroll::with(['payrollItems.employee', 'company'])->findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Generate PDF (implementation depends on your PDF library)
        // Example using Laravel-DOMPDF
        $pdf = PDF::loadView('hrm.payroll.pdf', compact('payroll'));

        return $pdf->download('payroll-' . $payroll->payroll_period . '.pdf');
    }

    /**
     * Export payroll to Excel.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function exportExcel(Request $request, $id)
    {
        $payroll = Payroll::with(['payrollItems.employee'])->findOrFail($id);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Generate Excel (implementation depends on your Excel library)
        // Example using Laravel-Excel
        return Excel::download(new PayrollExport($payroll), 'payroll-' . $payroll->payroll_period . '.xlsx');
    }

    /**
     * Show employee payslip.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $payrollId
     * @param  int  $employeeId
     * @return \Illuminate\Http\Response
     */
    public function showPayslip(Request $request, $payrollId, $employeeId)
    {
        $payroll = Payroll::findOrFail($payrollId);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        $payrollItem = PayrollItem::with('employee')
            ->where('payroll_id', $payrollId)
            ->where('employee_id', $employeeId)
            ->firstOrFail();

        return view('hrm.payroll.payslip', compact('payroll', 'payrollItem'));
    }

    /**
     * Download employee payslip as PDF.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $payrollId
     * @param  int  $employeeId
     * @return \Illuminate\Http\Response
     */
    public function downloadPayslip(Request $request, $payrollId, $employeeId)
    {
        $payroll = Payroll::with('company')->findOrFail($payrollId);

        // Check if payroll belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($payroll->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        $payrollItem = PayrollItem::with('employee')
            ->where('payroll_id', $payrollId)
            ->where('employee_id', $employeeId)
            ->firstOrFail();

        // Generate PDF (implementation depends on your PDF library)
        // Example using Laravel-DOMPDF
        $pdf = PDF::loadView('hrm.payroll.payslip_pdf', compact('payroll', 'payrollItem'));

        return $pdf->download('payslip-' . $payroll->payroll_period . '-' . $payrollItem->employee->name . '.pdf');
    }
}
