<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Loan;
use App\Models\HRM\LoanInstallment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class LoanInstallmentController extends Controller
{
    /**
     * عرض قائمة أقساط القروض
     */
    public function index(Request $request)
    {
        $companyId = session('company_id');

        $installments = LoanInstallment::with(['employee', 'loan'])
            ->where('company_id', $companyId)
            ->when($request->employee_id, function ($query, $employeeId) {
                return $query->where('employee_id', $employeeId);
            })
            ->when($request->loan_id, function ($query, $loanId) {
                return $query->where('loan_id', $loanId);
            })
            ->when($request->status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($request->due_date_from, function ($query, $dueDateFrom) {
                return $query->where('due_date', '>=', $dueDateFrom);
            })
            ->when($request->due_date_to, function ($query, $dueDateTo) {
                return $query->where('due_date', '<=', $dueDateTo);
            })
            ->orderBy('due_date')
            ->paginate(15);

        $loans = Loan::where('company_id', $companyId)
            ->whereIn('status', ['active', 'completed'])
            ->orderBy('start_date', 'desc')
            ->get();

        $statuses = LoanInstallment::STATUSES;

        return view('hrm.loan_installments.index', compact('installments', 'loans', 'statuses'));
    }

    /**
     * عرض تفاصيل قسط القرض
     */
    public function show(LoanInstallment $installment)
    {
        $companyId = session('company_id');

        if ($installment->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القسط');
        }

        $installment->load(['employee', 'loan', 'payrollItem']);

        return view('hrm.loan_installments.show', compact('installment'));
    }

    /**
     * عرض نموذج تعديل قسط القرض
     */
    public function edit(LoanInstallment $installment)
    {
        $companyId = session('company_id');

        if ($installment->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القسط');
        }

        if ($installment->status != 'pending') {
            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('error', 'لا يمكن تعديل القسط بعد دفعه أو إلغائه');
        }

        return view('hrm.loan_installments.edit', compact('installment'));
    }

    /**
     * تحديث قسط القرض
     */
    public function update(Request $request, LoanInstallment $installment)
    {
        $companyId = session('company_id');

        if ($installment->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القسط');
        }

        if ($installment->status != 'pending') {
            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('error', 'لا يمكن تعديل القسط بعد دفعه أو إلغائه');
        }

        $request->validate([
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $installment->amount = $request->amount;
            $installment->due_date = $request->due_date;
            $installment->notes = $request->notes;
            $installment->updated_by = Auth::id();
            $installment->save();

            // تحديث إجمالي مبلغ القرض إذا تغير مبلغ القسط
            $loan = $installment->loan;
            $totalAmount = LoanInstallment::where('loan_id', $loan->id)->sum('amount');
            $loan->total_amount = $totalAmount;
            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('success', 'تم تحديث قسط القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث قسط القرض: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * تسجيل دفع قسط القرض
     */
    public function markAsPaid(Request $request, LoanInstallment $installment)
    {
        $companyId = session('company_id');

        if ($installment->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القسط');
        }

        if ($installment->status != 'pending' && $installment->status != 'overdue') {
            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('error', 'لا يمكن تسجيل دفع القسط في حالته الحالية');
        }

        $request->validate([
            'payment_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $installment->status = 'paid';
            $installment->payment_date = $request->payment_date;
            $installment->notes = $request->notes;
            $installment->updated_by = Auth::id();
            $installment->save();

            // تحديث عدد الأقساط المتبقية في القرض
            $loan = $installment->loan;
            $remainingInstallments = LoanInstallment::where('loan_id', $loan->id)
                ->where('status', '!=', 'paid')
                ->where('status', '!=', 'cancelled')
                ->count();

            $loan->remaining_installments = $remainingInstallments;

            // إذا لم تعد هناك أقساط متبقية، قم بتحديث حالة القرض إلى مكتمل
            if ($remainingInstallments == 0 && $loan->status == 'active') {
                $loan->status = 'completed';
            }

            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('success', 'تم تسجيل دفع قسط القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تسجيل دفع قسط القرض: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة قسط القرض إلى متأخر
     */
    public function markAsOverdue(LoanInstallment $installment)
    {
        $companyId = session('company_id');

        if ($installment->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القسط');
        }

        if ($installment->status != 'pending') {
            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('error', 'لا يمكن تحديث حالة القسط إلى متأخر في حالته الحالية');
        }

        try {
            DB::beginTransaction();

            $installment->status = 'overdue';
            $installment->updated_by = Auth::id();
            $installment->save();

            DB::commit();

            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('success', 'تم تحديث حالة قسط القرض إلى متأخر بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث حالة قسط القرض: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء قسط القرض
     */
    public function cancel(Request $request, LoanInstallment $installment)
    {
        $companyId = session('company_id');

        if ($installment->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القسط');
        }

        if ($installment->status == 'paid' || $installment->status == 'cancelled') {
            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('error', 'لا يمكن إلغاء القسط في حالته الحالية');
        }

        $request->validate([
            'cancellation_reason' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            $installment->status = 'cancelled';
            $installment->notes = $request->cancellation_reason;
            $installment->updated_by = Auth::id();
            $installment->save();

            // تحديث عدد الأقساط المتبقية في القرض
            $loan = $installment->loan;
            $remainingInstallments = LoanInstallment::where('loan_id', $loan->id)
                ->where('status', '!=', 'paid')
                ->where('status', '!=', 'cancelled')
                ->count();

            $loan->remaining_installments = $remainingInstallments;

            // إذا لم تعد هناك أقساط متبقية، قم بتحديث حالة القرض إلى مكتمل
            if ($remainingInstallments == 0 && $loan->status == 'active') {
                $loan->status = 'completed';
            }

            // تحديث إجمالي مبلغ القرض
            $totalAmount = LoanInstallment::where('loan_id', $loan->id)
                ->where('status', '!=', 'cancelled')
                ->sum('amount');

            $loan->total_amount = $totalAmount;
            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loan-installments.show', $installment)
                ->with('success', 'تم إلغاء قسط القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إلغاء قسط القرض: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة جميع الأقساط المتأخرة
     */
    public function updateOverdueInstallments()
    {
        $companyId = session('company_id');

        try {
            DB::beginTransaction();

            // تحديث حالة الأقساط المتأخرة
            $overdueInstallments = LoanInstallment::where('company_id', $companyId)
                ->where('status', 'pending')
                ->where('due_date', '<', now()->startOfDay())
                ->get();

            foreach ($overdueInstallments as $installment) {
                $installment->status = 'overdue';
                $installment->updated_by = Auth::id();
                $installment->save();
            }

            DB::commit();

            return redirect()->route('hrm.loan-installments.index')
                ->with('success', 'تم تحديث حالة الأقساط المتأخرة بنجاح. عدد الأقساط المتأخرة: ' . $overdueInstallments->count());
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث حالة الأقساط المتأخرة: ' . $e->getMessage());
        }
    }
}
