<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Loan;
use App\Models\HRM\LoanInstallment;
use App\Models\HRM\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class LoanController extends Controller
{
    /**
     * عرض قائمة القروض
     */
    public function index(Request $request)
    {
        $companyId = session('company_id');

        $loans = Loan::with(['employee', 'approver'])
            ->where('company_id', $companyId)
            ->when($request->employee_id, function ($query, $employeeId) {
                return $query->where('employee_id', $employeeId);
            })
            ->when($request->loan_type, function ($query, $loanType) {
                return $query->where('loan_type', $loanType);
            })
            ->when($request->status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($request->start_date_from, function ($query, $startDateFrom) {
                return $query->where('start_date', '>=', $startDateFrom);
            })
            ->when($request->start_date_to, function ($query, $startDateTo) {
                return $query->where('start_date', '<=', $startDateTo);
            })
            ->orderBy('start_date', 'desc')
            ->paginate(15);

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('first_name')
            ->get();

        $loanTypes = Loan::LOAN_TYPES;
        $statuses = Loan::STATUSES;

        return view('hrm.loans.index', compact('loans', 'employees', 'loanTypes', 'statuses'));
    }

    /**
     * عرض نموذج إنشاء قرض جديد
     */
    public function create()
    {
        $companyId = session('company_id');

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('first_name')
            ->get();

        $loanTypes = Loan::LOAN_TYPES;

        return view('hrm.loans.create', compact('employees', 'loanTypes'));
    }

    /**
     * حفظ قرض جديد
     */
    public function store(Request $request)
    {
        $companyId = session('company_id');

        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'loan_type' => 'required|in:' . implode(',', array_keys(Loan::LOAN_TYPES)),
            'amount' => 'required|numeric|min:0',
            'interest_rate' => 'required|numeric|min:0|max:100',
            'installment_count' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'description' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // حساب مبلغ الفائدة والمبلغ الإجمالي
            $amount = $request->amount;
            $interestRate = $request->interest_rate;
            $interestAmount = $amount * ($interestRate / 100);
            $totalAmount = $amount + $interestAmount;

            // حساب مبلغ القسط وتاريخ الانتهاء
            $installmentCount = $request->installment_count;
            $installmentAmount = round($totalAmount / $installmentCount, 2);
            $startDate = Carbon::parse($request->start_date);
            $endDate = (clone $startDate)->addMonths($installmentCount - 1);

            $loan = new Loan();
            $loan->company_id = $companyId;
            $loan->employee_id = $request->employee_id;
            $loan->loan_type = $request->loan_type;
            $loan->amount = $amount;
            $loan->interest_rate = $interestRate;
            $loan->interest_amount = $interestAmount;
            $loan->total_amount = $totalAmount;
            $loan->installment_amount = $installmentAmount;
            $loan->installment_count = $installmentCount;
            $loan->remaining_installments = $installmentCount;
            $loan->start_date = $startDate;
            $loan->end_date = $endDate;
            $loan->status = 'pending';
            $loan->description = $request->description;
            $loan->notes = $request->notes;
            $loan->created_by = Auth::id();
            $loan->save();

            // إنشاء الأقساط
            for ($i = 0; $i < $installmentCount; $i++) {
                $installmentDate = (clone $startDate)->addMonths($i);

                $installment = new LoanInstallment();
                $installment->company_id = $companyId;
                $installment->loan_id = $loan->id;
                $installment->employee_id = $request->employee_id;
                $installment->installment_number = $i + 1;
                $installment->amount = $installmentAmount;
                $installment->due_date = $installmentDate;
                $installment->status = 'pending';
                $installment->created_by = Auth::id();
                $installment->save();
            }

            DB::commit();

            return redirect()->route('hrm.loans.index')
                ->with('success', 'تم إنشاء القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء القرض: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * عرض تفاصيل القرض
     */
    public function show(Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        $loan->load(['employee', 'approver', 'creator']);

        $installments = LoanInstallment::where('loan_id', $loan->id)
            ->orderBy('installment_number')
            ->get();

        return view('hrm.loans.show', compact('loan', 'installments'));
    }

    /**
     * عرض نموذج تعديل القرض
     */
    public function edit(Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'pending') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'لا يمكن تعديل القرض بعد الموافقة عليه أو رفضه');
        }

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('first_name')
            ->get();

        $loanTypes = Loan::LOAN_TYPES;

        return view('hrm.loans.edit', compact('loan', 'employees', 'loanTypes'));
    }

    /**
     * تحديث القرض
     */
    public function update(Request $request, Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'pending') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'لا يمكن تعديل القرض بعد الموافقة عليه أو رفضه');
        }

        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'loan_type' => 'required|in:' . implode(',', array_keys(Loan::LOAN_TYPES)),
            'amount' => 'required|numeric|min:0',
            'interest_rate' => 'required|numeric|min:0|max:100',
            'installment_count' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'description' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // حساب مبلغ الفائدة والمبلغ الإجمالي
            $amount = $request->amount;
            $interestRate = $request->interest_rate;
            $interestAmount = $amount * ($interestRate / 100);
            $totalAmount = $amount + $interestAmount;

            // حساب مبلغ القسط وتاريخ الانتهاء
            $installmentCount = $request->installment_count;
            $installmentAmount = round($totalAmount / $installmentCount, 2);
            $startDate = Carbon::parse($request->start_date);
            $endDate = (clone $startDate)->addMonths($installmentCount - 1);

            $loan->employee_id = $request->employee_id;
            $loan->loan_type = $request->loan_type;
            $loan->amount = $amount;
            $loan->interest_rate = $interestRate;
            $loan->interest_amount = $interestAmount;
            $loan->total_amount = $totalAmount;
            $loan->installment_amount = $installmentAmount;
            $loan->installment_count = $installmentCount;
            $loan->remaining_installments = $installmentCount;
            $loan->start_date = $startDate;
            $loan->end_date = $endDate;
            $loan->description = $request->description;
            $loan->notes = $request->notes;
            $loan->updated_by = Auth::id();
            $loan->save();

            // حذف الأقساط القديمة
            LoanInstallment::where('loan_id', $loan->id)->delete();

            // إنشاء الأقساط الجديدة
            for ($i = 0; $i < $installmentCount; $i++) {
                $installmentDate = (clone $startDate)->addMonths($i);

                $installment = new LoanInstallment();
                $installment->company_id = $companyId;
                $installment->loan_id = $loan->id;
                $installment->employee_id = $request->employee_id;
                $installment->installment_number = $i + 1;
                $installment->amount = $installmentAmount;
                $installment->due_date = $installmentDate;
                $installment->status = 'pending';
                $installment->created_by = Auth::id();
                $installment->save();
            }

            DB::commit();

            return redirect()->route('hrm.loans.show', $loan)
                ->with('success', 'تم تحديث القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث القرض: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * حذف القرض
     */
    public function destroy(Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'pending') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'لا يمكن حذف القرض بعد الموافقة عليه أو رفضه');
        }

        try {
            DB::beginTransaction();

            // حذف الأقساط أولاً
            LoanInstallment::where('loan_id', $loan->id)->delete();

            // ثم حذف القرض
            $loan->delete();

            DB::commit();

            return redirect()->route('hrm.loans.index')
                ->with('success', 'تم حذف القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف القرض: ' . $e->getMessage());
        }
    }

    /**
     * الموافقة على القرض
     */
    public function approve(Request $request, Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'pending') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'تم بالفعل اتخاذ إجراء على هذا القرض');
        }

        try {
            DB::beginTransaction();

            $loan->status = 'approved';
            $loan->approved_by = Auth::id();
            $loan->approved_at = now();
            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loans.show', $loan)
                ->with('success', 'تمت الموافقة على القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء الموافقة على القرض: ' . $e->getMessage());
        }
    }

    /**
     * رفض القرض
     */
    public function reject(Request $request, Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'pending') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'تم بالفعل اتخاذ إجراء على هذا القرض');
        }

        $request->validate([
            'rejection_reason' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            $loan->status = 'rejected';
            $loan->notes = $request->rejection_reason;
            $loan->approved_by = Auth::id();
            $loan->approved_at = now();
            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loans.show', $loan)
                ->with('success', 'تم رفض القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء رفض القرض: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة القرض إلى نشط
     */
    public function activate(Request $request, Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'approved') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'يجب الموافقة على القرض أولاً قبل تنشيطه');
        }

        try {
            DB::beginTransaction();

            $loan->status = 'active';
            $loan->updated_by = Auth::id();
            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loans.show', $loan)
                ->with('success', 'تم تنشيط القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تنشيط القرض: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة القرض إلى مكتمل
     */
    public function complete(Request $request, Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if ($loan->status != 'active') {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'يجب أن يكون القرض نشطًا قبل تحديده كمكتمل');
        }

        // التحقق من أن جميع الأقساط مدفوعة
        $pendingInstallments = LoanInstallment::where('loan_id', $loan->id)
            ->where('status', '!=', 'paid')
            ->count();

        if ($pendingInstallments > 0) {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'لا يمكن إكمال القرض حتى يتم دفع جميع الأقساط');
        }

        try {
            DB::beginTransaction();

            $loan->status = 'completed';
            $loan->remaining_installments = 0;
            $loan->updated_by = Auth::id();
            $loan->save();

            DB::commit();

            return redirect()->route('hrm.loans.show', $loan)
                ->with('success', 'تم إكمال القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إكمال القرض: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء القرض
     */
    public function cancel(Request $request, Loan $loan)
    {
        $companyId = session('company_id');

        if ($loan->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا القرض');
        }

        if (!in_array($loan->status, ['approved', 'active'])) {
            return redirect()->route('hrm.loans.show', $loan)
                ->with('error', 'لا يمكن إلغاء القرض في حالته الحالية');
        }

        $request->validate([
            'cancellation_reason' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            $loan->status = 'cancelled';
            $loan->notes = $request->cancellation_reason;
            $loan->updated_by = Auth::id();
            $loan->save();

            // تحديث حالة الأقساط المتبقية إلى ملغية
            LoanInstallment::where('loan_id', $loan->id)
                ->where('status', 'pending')
                ->update([
                    'status' => 'cancelled',
                    'updated_by' => Auth::id()
                ]);

            DB::commit();

            return redirect()->route('hrm.loans.show', $loan)
                ->with('success', 'تم إلغاء القرض بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إلغاء القرض: ' . $e->getMessage());
        }
    }
}
