<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Employee;
use App\Models\HRM\Leave;
use App\Models\HRM\LeaveType;
use App\Models\HRM\LeaveBalance;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class LeaveController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = Leave::with(['employee', 'leaveType', 'approver'])
            ->where('company_id', $companyId);

        // Apply filters if provided
        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('leave_type_id') && $request->leave_type_id != '') {
            $query->where('leave_type_id', $request->leave_type_id);
        }

        if ($request->has('date_from') && $request->date_from != '') {
            $query->where('start_date', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to != '') {
            $query->where('end_date', '<=', $request->date_to);
        }

        $leaves = $query->orderBy('created_at', 'desc')->paginate(10);

        // Get all employees and leave types for filter dropdowns
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Calculate leave statistics
        $totalLeaves = Leave::where('company_id', $companyId)->count();
        $pendingLeaves = Leave::where('company_id', $companyId)->where('status', 'pending')->count();
        $approvedLeaves = Leave::where('company_id', $companyId)->where('status', 'approved')->count();
        $rejectedLeaves = Leave::where('company_id', $companyId)->where('status', 'rejected')->count();
        $cancelledLeaves = Leave::where('company_id', $companyId)->where('status', 'cancelled')->count();

        // Get leave statistics by type
        $leaveTypeStats = DB::table('leaves')
            ->join('leave_types', 'leaves.leave_type_id', '=', 'leave_types.id')
            ->select('leave_types.name', DB::raw('count(*) as count'))
            ->where('leaves.company_id', $companyId)
            ->groupBy('leave_types.name')
            ->get();

        // Get leave statistics by month
        $leaveMonthStats = DB::table('leaves')
            ->select(DB::raw('MONTHNAME(start_date) as month'), DB::raw('count(*) as count'))
            ->where('leaves.company_id', $companyId)
            ->whereYear('start_date', date('Y'))
            ->groupBy('month')
            ->orderBy(DB::raw('MONTH(start_date)'))
            ->get();

        return view('hrm.leaves.index', compact(
            'leaves',
            'employees',
            'leaveTypes',
            'totalLeaves',
            'pendingLeaves',
            'approvedLeaves',
            'rejectedLeaves',
            'cancelledLeaves',
            'leaveTypeStats',
            'leaveMonthStats'
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.leaves.create', compact('employees', 'leaveTypes'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required|string',
            'attachment' => 'nullable|file|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');

        // Check if employee belongs to the company
        $employee = Employee::findOrFail($request->employee_id);
        if ($employee->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // Check if leave type belongs to the company
        $leaveType = LeaveType::findOrFail($request->leave_type_id);
        if ($leaveType->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // Check for overlapping leave requests
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $totalDays = $endDate->diffInDays($startDate) + 1;

        $overlappingLeaves = Leave::where('employee_id', $request->employee_id)
            ->where('status', '!=', 'rejected')
            ->where('status', '!=', 'cancelled')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate])
                    ->orWhere(function ($q) use ($startDate, $endDate) {
                        $q->where('start_date', '<=', $startDate)
                            ->where('end_date', '>=', $endDate);
                    });
            })
            ->exists();

        if ($overlappingLeaves) {
            return redirect()->back()
                ->with('error', 'يوجد بالفعل طلب إجازة معتمد أو قيد الانتظار لهذه الفترة.')
                ->withInput();
        }

        // Check leave balance if the leave type requires balance check
        if ($leaveType->requires_balance_check) {
            $year = $startDate->year;

            // Get or create leave balance for the employee, leave type and year
            $leaveBalance = LeaveBalance::firstOrCreate(
                [
                    'company_id' => $companyId,
                    'employee_id' => $request->employee_id,
                    'leave_type_id' => $request->leave_type_id,
                    'year' => $year
                ],
                [
                    'total_days' => $leaveType->default_days,
                    'used_days' => 0,
                    'pending_days' => 0,
                    'remaining_days' => $leaveType->default_days,
                    'carried_over_days' => 0,
                    'created_by' => Auth::id()
                ]
            );

            // Check if employee has enough balance
            if ($leaveBalance->remaining_days < $totalDays) {
                return redirect()->back()
                    ->with('error', 'رصيد الإجازة غير كافٍ. الرصيد المتبقي: ' . $leaveBalance->remaining_days . ' يوم.')
                    ->withInput();
            }

            // Update pending days in leave balance
            $leaveBalance->pending_days += $totalDays;
            $leaveBalance->updateRemainingDays();
        }

        // Handle attachment upload
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('leaves/attachments', 'public');
        }

        $leave = new Leave();
        $leave->company_id = $companyId;
        $leave->employee_id = $request->employee_id;
        $leave->leave_type_id = $request->leave_type_id;
        $leave->start_date = $startDate;
        $leave->end_date = $endDate;
        $leave->total_days = $totalDays;
        $leave->reason = $request->reason;
        $leave->attachment = $attachmentPath;
        $leave->status = 'pending';
        $leave->created_by = Auth::id();
        $leave->save();

        return redirect()->route('hrm.leaves.index')
            ->with('success', 'تم تقديم طلب الإجازة بنجاح.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $leave = Leave::with(['employee', 'leaveType', 'approver'])->findOrFail($id);

        // Check if leave belongs to the current company
        $companyId = session('company_id');
        if ($leave->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        return view('hrm.leaves.show', compact('leave'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $companyId = $request->session()->get('company_id');
        $leave = Leave::findOrFail($id);

        // Check if leave belongs to the current company
        if ($leave->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending leaves can be edited
        if ($leave->status != 'pending') {
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'Only pending leave requests can be edited.');
        }

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.leaves.edit', compact('leave', 'employees', 'leaveTypes'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $leave = Leave::findOrFail($id);

        // Check if leave belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($leave->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending leaves can be updated
        if ($leave->status != 'pending') {
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'Only pending leave requests can be updated.');
        }

        $validator = Validator::make($request->all(), [
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required|string',
            'attachment' => 'nullable|file|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Check if leave type belongs to the company
        $leaveType = LeaveType::findOrFail($request->leave_type_id);
        if ($leaveType->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Check for overlapping leave requests (excluding this one)
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $totalDays = $endDate->diffInDays($startDate) + 1;

        $overlappingLeaves = Leave::where('employee_id', $leave->employee_id)
            ->where('id', '!=', $id)
            ->where('status', '!=', 'rejected')
            ->where('status', '!=', 'cancelled')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate])
                    ->orWhere(function ($q) use ($startDate, $endDate) {
                        $q->where('start_date', '<=', $startDate)
                            ->where('end_date', '>=', $endDate);
                    });
            })
            ->exists();

        if ($overlappingLeaves) {
            return redirect()->back()
                ->with('error', 'There is already an approved or pending leave request for this period.')
                ->withInput();
        }

        // Handle attachment upload
        if ($request->hasFile('attachment')) {
            // Delete old attachment if exists
            if ($leave->attachment) {
                Storage::disk('public')->delete($leave->attachment);
            }

            $attachmentPath = $request->file('attachment')->store('leaves/attachments', 'public');
            $leave->attachment = $attachmentPath;
        }

        $leave->leave_type_id = $request->leave_type_id;
        $leave->start_date = $startDate;
        $leave->end_date = $endDate;
        $leave->total_days = $totalDays;
        $leave->reason = $request->reason;
        $leave->updated_by = Auth::id();
        $leave->save();

        return redirect()->route('hrm.leaves.show', $id)
            ->with('success', 'Leave request updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        $leave = Leave::findOrFail($id);

        // Check if leave belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($leave->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending leaves can be deleted
        if ($leave->status != 'pending') {
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'Only pending leave requests can be deleted.');
        }

        // Delete attachment if exists
        if ($leave->attachment) {
            Storage::disk('public')->delete($leave->attachment);
        }

        $leave->delete();

        return redirect()->route('hrm.leaves.index')
            ->with('success', 'Leave request deleted successfully.');
    }

    /**
     * Approve a leave request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve(Request $request, $id)
    {
        $leave = Leave::findOrFail($id);

        // Check if leave belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($leave->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // Only pending leaves can be approved
        if ($leave->status != 'pending') {
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'يمكن الموافقة فقط على طلبات الإجازة المعلقة.');
        }

        DB::beginTransaction();

        try {
            // Update leave balance if needed
            $leaveType = LeaveType::findOrFail($leave->leave_type_id);
            if ($leaveType->requires_balance_check) {
                $year = Carbon::parse($leave->start_date)->year;

                // Get leave balance
                $leaveBalance = LeaveBalance::where('company_id', $companyId)
                    ->where('employee_id', $leave->employee_id)
                    ->where('leave_type_id', $leave->leave_type_id)
                    ->where('year', $year)
                    ->first();

                if ($leaveBalance) {
                    // Move days from pending to used
                    $leaveBalance->pending_days -= $leave->total_days;
                    $leaveBalance->used_days += $leave->total_days;
                    $leaveBalance->updateRemainingDays();
                }
            }

            // Update leave status
            $leave->status = 'approved';
            $leave->approved_by = Auth::id();
            $leave->approved_at = now();
            $leave->save();

            DB::commit();

            return redirect()->route('hrm.leaves.show', $id)
                ->with('success', 'تمت الموافقة على طلب الإجازة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'حدث خطأ أثناء الموافقة على الإجازة: ' . $e->getMessage());
        }
    }

    /**
     * Reject a leave request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reject(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'rejection_reason' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $leave = Leave::findOrFail($id);

        // Check if leave belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($leave->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // Only pending leaves can be rejected
        if ($leave->status != 'pending') {
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'يمكن رفض طلبات الإجازة المعلقة فقط.');
        }

        DB::beginTransaction();

        try {
            // Update leave balance if needed
            $leaveType = LeaveType::findOrFail($leave->leave_type_id);
            if ($leaveType->requires_balance_check) {
                $year = Carbon::parse($leave->start_date)->year;

                // Get leave balance
                $leaveBalance = LeaveBalance::where('company_id', $companyId)
                    ->where('employee_id', $leave->employee_id)
                    ->where('leave_type_id', $leave->leave_type_id)
                    ->where('year', $year)
                    ->first();

                if ($leaveBalance) {
                    // Remove days from pending
                    $leaveBalance->pending_days -= $leave->total_days;
                    $leaveBalance->updateRemainingDays();
                }
            }

            // Update leave status
            $leave->status = 'rejected';
            $leave->rejection_reason = $request->rejection_reason;
            $leave->rejected_by = Auth::id();
            $leave->rejected_at = now();
            $leave->save();

            DB::commit();

            return redirect()->route('hrm.leaves.show', $id)
                ->with('success', 'تم رفض طلب الإجازة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'حدث خطأ أثناء رفض الإجازة: ' . $e->getMessage());
        }
    }

    /**
     * Bulk approve leave requests.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkApprove(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'leave_ids' => 'required|array',
            'leave_ids.*' => 'exists:leaves,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');
        $leaveIds = $request->leave_ids;
        $successCount = 0;
        $errorCount = 0;

        foreach ($leaveIds as $id) {
            $leave = Leave::findOrFail($id);

            // Check if leave belongs to the current company
            if ($leave->company_id != $companyId) {
                $errorCount++;
                continue;
            }

            // Only pending leaves can be approved
            if ($leave->status != 'pending') {
                $errorCount++;
                continue;
            }

            DB::beginTransaction();

            try {
                // Update leave balance if needed
                $leaveType = LeaveType::findOrFail($leave->leave_type_id);
                if ($leaveType->requires_balance_check) {
                    $year = Carbon::parse($leave->start_date)->year;

                    // Get leave balance
                    $leaveBalance = LeaveBalance::where('company_id', $companyId)
                        ->where('employee_id', $leave->employee_id)
                        ->where('leave_type_id', $leave->leave_type_id)
                        ->where('year', $year)
                        ->first();

                    if ($leaveBalance) {
                        // Move days from pending to used
                        $leaveBalance->pending_days -= $leave->total_days;
                        $leaveBalance->used_days += $leave->total_days;
                        $leaveBalance->updateRemainingDays();
                    }
                }

                // Update leave status
                $leave->status = 'approved';
                $leave->approved_by = Auth::id();
                $leave->approved_at = now();
                $leave->save();

                DB::commit();
                $successCount++;
            } catch (\Exception $e) {
                DB::rollBack();
                $errorCount++;
            }
        }

        return redirect()->route('hrm.leaves.index')
            ->with('success', "تمت الموافقة على {$successCount} طلب إجازة بنجاح. فشل {$errorCount} طلب.");
    }

    /**
     * Bulk reject leave requests.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkReject(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'leave_ids' => 'required|array',
            'leave_ids.*' => 'exists:leaves,id',
            'rejection_reason' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');
        $leaveIds = $request->leave_ids;
        $rejectionReason = $request->rejection_reason;
        $successCount = 0;
        $errorCount = 0;

        foreach ($leaveIds as $id) {
            $leave = Leave::findOrFail($id);

            // Check if leave belongs to the current company
            if ($leave->company_id != $companyId) {
                $errorCount++;
                continue;
            }

            // Only pending leaves can be rejected
            if ($leave->status != 'pending') {
                $errorCount++;
                continue;
            }

            DB::beginTransaction();

            try {
                // Update leave balance if needed
                $leaveType = LeaveType::findOrFail($leave->leave_type_id);
                if ($leaveType->requires_balance_check) {
                    $year = Carbon::parse($leave->start_date)->year;

                    // Get leave balance
                    $leaveBalance = LeaveBalance::where('company_id', $companyId)
                        ->where('employee_id', $leave->employee_id)
                        ->where('leave_type_id', $leave->leave_type_id)
                        ->where('year', $year)
                        ->first();

                    if ($leaveBalance) {
                        // Remove days from pending
                        $leaveBalance->pending_days -= $leave->total_days;
                        $leaveBalance->updateRemainingDays();
                    }
                }

                // Update leave status
                $leave->status = 'rejected';
                $leave->rejection_reason = $rejectionReason;
                $leave->rejected_by = Auth::id();
                $leave->rejected_at = now();
                $leave->save();

                DB::commit();
                $successCount++;
            } catch (\Exception $e) {
                DB::rollBack();
                $errorCount++;
            }
        }

        return redirect()->route('hrm.leaves.index')
            ->with('success', "تم رفض {$successCount} طلب إجازة بنجاح. فشل {$errorCount} طلب.");
    }

    /**
     * Display leave calendar.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function calendar(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $month = $request->input('month', date('m'));
        $year = $request->input('year', date('Y'));

        // Get all approved leaves for the selected month and year
        $leaves = Leave::with(['employee', 'leaveType'])
            ->where('company_id', $companyId)
            ->where('status', 'approved')
            ->whereYear('start_date', $year)
            ->whereMonth('start_date', $month)
            ->orWhere(function ($query) use ($year, $month, $companyId) {
                $query->where('company_id', $companyId)
                    ->where('status', 'approved')
                    ->whereYear('end_date', $year)
                    ->whereMonth('end_date', $month);
            })
            ->get();

        // Prepare calendar data
        $calendarData = [];
        $startOfMonth = Carbon::createFromDate($year, $month, 1)->startOfMonth();
        $endOfMonth = Carbon::createFromDate($year, $month, 1)->endOfMonth();

        foreach ($leaves as $leave) {
            $leaveStart = Carbon::parse($leave->start_date);
            $leaveEnd = Carbon::parse($leave->end_date);

            // Adjust dates to be within the current month
            if ($leaveStart->lt($startOfMonth)) {
                $leaveStart = $startOfMonth;
            }

            if ($leaveEnd->gt($endOfMonth)) {
                $leaveEnd = $endOfMonth;
            }

            // Add leave to calendar for each day
            $currentDate = $leaveStart->copy();
            while ($currentDate->lte($leaveEnd)) {
                $dateKey = $currentDate->format('Y-m-d');

                if (!isset($calendarData[$dateKey])) {
                    $calendarData[$dateKey] = [];
                }

                $calendarData[$dateKey][] = [
                    'id' => $leave->id,
                    'employee' => $leave->employee->name,
                    'type' => $leave->leaveType->name,
                    'start_date' => $leave->start_date->format('Y-m-d'),
                    'end_date' => $leave->end_date->format('Y-m-d'),
                    'days' => $leave->total_days,
                ];

                $currentDate->addDay();
            }
        }

        // Get all employees and leave types for filter dropdowns
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.leaves.calendar', compact('calendarData', 'employees', 'leaveTypes', 'month', 'year'));
    }

    /**
     * Cancel a leave request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function cancel(Request $request, $id)
    {
        $leave = Leave::findOrFail($id);

        // Check if leave belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($leave->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // Only pending or approved leaves can be cancelled
        if (!in_array($leave->status, ['pending', 'approved'])) {
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'يمكن إلغاء طلبات الإجازة المعلقة أو المعتمدة فقط.');
        }

        DB::beginTransaction();

        try {
            // Update leave balance if needed
            $leaveType = LeaveType::findOrFail($leave->leave_type_id);
            if ($leaveType->requires_balance_check) {
                $year = Carbon::parse($leave->start_date)->year;

                // Get leave balance
                $leaveBalance = LeaveBalance::where('company_id', $companyId)
                    ->where('employee_id', $leave->employee_id)
                    ->where('leave_type_id', $leave->leave_type_id)
                    ->where('year', $year)
                    ->first();

                if ($leaveBalance) {
                    if ($leave->status == 'pending') {
                        // Remove days from pending
                        $leaveBalance->pending_days -= $leave->total_days;
                    } else if ($leave->status == 'approved') {
                        // Remove days from used
                        $leaveBalance->used_days -= $leave->total_days;
                    }
                    $leaveBalance->updateRemainingDays();
                }
            }

            // Update leave status
            $leave->status = 'cancelled';
            $leave->updated_by = Auth::id();
            $leave->save();

            DB::commit();

            return redirect()->route('hrm.leaves.show', $id)
                ->with('success', 'تم إلغاء طلب الإجازة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('hrm.leaves.show', $id)
                ->with('error', 'حدث خطأ أثناء إلغاء الإجازة: ' . $e->getMessage());
        }
    }

    /**
     * Display leave balance for employees.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function leaveBalance(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $year = $request->year ? $request->year : date('Y');

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveBalances = [];

        foreach ($employees as $employee) {
            $balance = [];

            foreach ($leaveTypes as $leaveType) {
                // Get total allocated days for this leave type
                $allocatedDays = $leaveType->days_per_year;

                // Get total used days for this leave type in the current year
                $usedDays = Leave::where('employee_id', $employee->id)
                    ->where('leave_type_id', $leaveType->id)
                    ->where('status', 'approved')
                    ->whereYear('start_date', $year)
                    ->sum('total_days');

                $balance[$leaveType->id] = [
                    'leave_type' => $leaveType->name,
                    'allocated' => $allocatedDays,
                    'used' => $usedDays,
                    'remaining' => $allocatedDays - $usedDays,
                ];
            }

            $leaveBalances[$employee->id] = $balance;
        }

        return view('hrm.leaves.balance', compact('employees', 'leaveTypes', 'leaveBalances', 'year'));
    }
}
