<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Employee;
use App\Models\HRM\Leave;
use App\Models\HRM\LeaveBalance;
use App\Models\HRM\LeaveType;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class LeaveBalanceController extends Controller
{
    /**
     * عرض قائمة أرصدة الإجازات
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = LeaveBalance::with(['employee', 'leaveType'])
            ->where('company_id', $companyId);

        // تطبيق الفلاتر إذا وجدت
        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->has('leave_type_id') && $request->leave_type_id != '') {
            $query->where('leave_type_id', $request->leave_type_id);
        }

        if ($request->has('year') && $request->year != '') {
            $query->where('year', $request->year);
        } else {
            // افتراضياً، عرض أرصدة السنة الحالية
            $query->where('year', date('Y'));
        }

        $leaveBalances = $query->orderBy('employee_id')->paginate(10);

        // الحصول على قائمة الموظفين وأنواع الإجازات للفلاتر
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // الحصول على قائمة السنوات المتاحة
        $years = LeaveBalance::where('company_id', $companyId)
            ->select('year')
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year')
            ->toArray();

        // إضافة السنة الحالية إذا لم تكن موجودة
        if (!in_array(date('Y'), $years)) {
            array_unshift($years, date('Y'));
        }

        return view('hrm.leaves.balances.index', compact('leaveBalances', 'employees', 'leaveTypes', 'years'));
    }

    /**
     * عرض نموذج إنشاء رصيد إجازة جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // الحصول على قائمة السنوات (السنة الحالية و5 سنوات قادمة)
        $years = [];
        $currentYear = date('Y');
        for ($i = 0; $i <= 5; $i++) {
            $year = $currentYear + $i;
            $years[$year] = $year;
        }

        return view('hrm.leaves.balances.create', compact('employees', 'leaveTypes', 'years'));
    }

    /**
     * تخزين رصيد إجازة جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'leave_type_id' => 'required|exists:leave_types,id',
            'year' => 'required|integer|min:2000|max:2100',
            'total_days' => 'required|numeric|min:0',
            'carried_over_days' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');

        // التحقق من أن الموظف ينتمي للشركة الحالية
        $employee = Employee::findOrFail($request->employee_id);
        if ($employee->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // التحقق من أن نوع الإجازة ينتمي للشركة الحالية
        $leaveType = LeaveType::findOrFail($request->leave_type_id);
        if ($leaveType->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // التحقق من عدم وجود رصيد إجازة للموظف ونوع الإجازة والسنة
        $existingBalance = LeaveBalance::where('employee_id', $request->employee_id)
            ->where('leave_type_id', $request->leave_type_id)
            ->where('year', $request->year)
            ->first();

        if ($existingBalance) {
            return redirect()->back()
                ->with('error', 'يوجد بالفعل رصيد إجازة لهذا الموظف ونوع الإجازة والسنة.')
                ->withInput();
        }

        // حساب الأيام المستخدمة والمعلقة
        $usedDays = $this->calculateUsedDays($request->employee_id, $request->leave_type_id, $request->year);
        $pendingDays = $this->calculatePendingDays($request->employee_id, $request->leave_type_id, $request->year);
        $carriedOverDays = $request->carried_over_days ?? 0;
        $totalDays = $request->total_days;
        $remainingDays = $totalDays + $carriedOverDays - $usedDays - $pendingDays;

        // إنشاء رصيد إجازة جديد
        $leaveBalance = new LeaveBalance();
        $leaveBalance->company_id = $companyId;
        $leaveBalance->employee_id = $request->employee_id;
        $leaveBalance->leave_type_id = $request->leave_type_id;
        $leaveBalance->year = $request->year;
        $leaveBalance->total_days = $totalDays;
        $leaveBalance->used_days = $usedDays;
        $leaveBalance->pending_days = $pendingDays;
        $leaveBalance->remaining_days = $remainingDays;
        $leaveBalance->carried_over_days = $carriedOverDays;
        $leaveBalance->notes = $request->notes;
        $leaveBalance->created_by = Auth::id();
        $leaveBalance->save();

        return redirect()->route('hrm.leave-balances.index')
            ->with('success', 'تم إنشاء رصيد الإجازة بنجاح.');
    }

    /**
     * عرض تفاصيل رصيد إجازة محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $leaveBalance = LeaveBalance::with(['employee', 'leaveType'])->findOrFail($id);

        // التحقق من أن رصيد الإجازة ينتمي للشركة الحالية
        $companyId = session('company_id');
        if ($leaveBalance->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // الحصول على سجل الإجازات للموظف ونوع الإجازة والسنة
        $leaves = Leave::where('employee_id', $leaveBalance->employee_id)
            ->where('leave_type_id', $leaveBalance->leave_type_id)
            ->whereYear('start_date', $leaveBalance->year)
            ->orderBy('start_date', 'desc')
            ->get();

        return view('hrm.leaves.balances.show', compact('leaveBalance', 'leaves'));
    }

    /**
     * عرض نموذج تعديل رصيد إجازة محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $companyId = $request->session()->get('company_id');
        $leaveBalance = LeaveBalance::findOrFail($id);

        // التحقق من أن رصيد الإجازة ينتمي للشركة الحالية
        if ($leaveBalance->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $leaveTypes = LeaveType::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // الحصول على قائمة السنوات (السنة الحالية و5 سنوات قادمة)
        $years = [];
        $currentYear = date('Y');
        for ($i = 0; $i <= 5; $i++) {
            $year = $currentYear + $i;
            $years[$year] = $year;
        }

        return view('hrm.leaves.balances.edit', compact('leaveBalance', 'employees', 'leaveTypes', 'years'));
    }

    /**
     * تحديث رصيد إجازة محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $leaveBalance = LeaveBalance::findOrFail($id);

        // التحقق من أن رصيد الإجازة ينتمي للشركة الحالية
        $companyId = $request->session()->get('company_id');
        if ($leaveBalance->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        $validator = Validator::make($request->all(), [
            'total_days' => 'required|numeric|min:0',
            'carried_over_days' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // تحديث رصيد الإجازة
        $carriedOverDays = $request->carried_over_days ?? 0;
        $totalDays = $request->total_days;
        $remainingDays = $totalDays + $carriedOverDays - $leaveBalance->used_days - $leaveBalance->pending_days;

        $leaveBalance->total_days = $totalDays;
        $leaveBalance->carried_over_days = $carriedOverDays;
        $leaveBalance->remaining_days = $remainingDays;
        $leaveBalance->notes = $request->notes;
        $leaveBalance->updated_by = Auth::id();
        $leaveBalance->save();

        return redirect()->route('hrm.leave-balances.show', $id)
            ->with('success', 'تم تحديث رصيد الإجازة بنجاح.');
    }

    /**
     * حذف رصيد إجازة محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        $leaveBalance = LeaveBalance::findOrFail($id);

        // التحقق من أن رصيد الإجازة ينتمي للشركة الحالية
        $companyId = $request->session()->get('company_id');
        if ($leaveBalance->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // التحقق من عدم وجود إجازات مرتبطة بهذا الرصيد
        $hasLeaves = Leave::where('employee_id', $leaveBalance->employee_id)
            ->where('leave_type_id', $leaveBalance->leave_type_id)
            ->whereYear('start_date', $leaveBalance->year)
            ->exists();

        if ($hasLeaves) {
            return redirect()->route('hrm.leave-balances.index')
                ->with('error', 'لا يمكن حذف رصيد الإجازة لأنه مرتبط بإجازات موجودة.');
        }

        $leaveBalance->delete();

        return redirect()->route('hrm.leave-balances.index')
            ->with('success', 'تم حذف رصيد الإجازة بنجاح.');
    }

    /**
     * تحديث أرصدة الإجازات تلقائياً
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateAllBalances(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $year = $request->input('year', date('Y'));

        // الحصول على جميع أرصدة الإجازات للسنة المحددة
        $leaveBalances = LeaveBalance::where('company_id', $companyId)
            ->where('year', $year)
            ->get();

        $updatedCount = 0;

        foreach ($leaveBalances as $leaveBalance) {
            // حساب الأيام المستخدمة والمعلقة
            $usedDays = $this->calculateUsedDays($leaveBalance->employee_id, $leaveBalance->leave_type_id, $year);
            $pendingDays = $this->calculatePendingDays($leaveBalance->employee_id, $leaveBalance->leave_type_id, $year);
            $remainingDays = $leaveBalance->total_days + $leaveBalance->carried_over_days - $usedDays - $pendingDays;

            // تحديث رصيد الإجازة
            $leaveBalance->used_days = $usedDays;
            $leaveBalance->pending_days = $pendingDays;
            $leaveBalance->remaining_days = $remainingDays;
            $leaveBalance->updated_by = Auth::id();
            $leaveBalance->save();

            $updatedCount++;
        }

        return redirect()->route('hrm.leave-balances.index', ['year' => $year])
            ->with('success', "تم تحديث {$updatedCount} رصيد إجازة بنجاح.");
    }

    /**
     * إنشاء أرصدة إجازات للسنة الجديدة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function createNewYearBalances(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $fromYear = $request->input('from_year', date('Y') - 1);
        $toYear = $request->input('to_year', date('Y'));
        $carryOverPercentage = $request->input('carry_over_percentage', 0);

        // التحقق من صحة المدخلات
        $validator = Validator::make($request->all(), [
            'from_year' => 'required|integer|min:2000|max:2100',
            'to_year' => 'required|integer|min:2000|max:2100',
            'carry_over_percentage' => 'required|integer|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // الحصول على أرصدة الإجازات للسنة السابقة
        $previousBalances = LeaveBalance::where('company_id', $companyId)
            ->where('year', $fromYear)
            ->get();

        $createdCount = 0;
        $updatedCount = 0;

        DB::beginTransaction();

        try {
            foreach ($previousBalances as $previousBalance) {
                // التحقق من وجود رصيد للسنة الجديدة
                $existingBalance = LeaveBalance::where('company_id', $companyId)
                    ->where('employee_id', $previousBalance->employee_id)
                    ->where('leave_type_id', $previousBalance->leave_type_id)
                    ->where('year', $toYear)
                    ->first();

                // حساب الأيام المرحلة
                $remainingDays = $previousBalance->remaining_days;
                $carriedOverDays = ($remainingDays > 0) ? ($remainingDays * $carryOverPercentage / 100) : 0;

                if ($existingBalance) {
                    // تحديث الرصيد الموجود
                    $existingBalance->carried_over_days = $carriedOverDays;
                    $existingBalance->remaining_days = $existingBalance->total_days + $carriedOverDays - $existingBalance->used_days - $existingBalance->pending_days;
                    $existingBalance->updated_by = Auth::id();
                    $existingBalance->save();
                    $updatedCount++;
                } else {
                    // إنشاء رصيد جديد
                    $newBalance = new LeaveBalance();
                    $newBalance->company_id = $companyId;
                    $newBalance->employee_id = $previousBalance->employee_id;
                    $newBalance->leave_type_id = $previousBalance->leave_type_id;
                    $newBalance->year = $toYear;
                    $newBalance->total_days = $previousBalance->total_days; // نفس عدد الأيام كالسنة السابقة
                    $newBalance->used_days = 0;
                    $newBalance->pending_days = 0;
                    $newBalance->remaining_days = $previousBalance->total_days + $carriedOverDays;
                    $newBalance->carried_over_days = $carriedOverDays;
                    $newBalance->notes = "تم إنشاؤه تلقائياً من رصيد سنة {$fromYear}";
                    $newBalance->created_by = Auth::id();
                    $newBalance->save();
                    $createdCount++;
                }
            }

            DB::commit();

            return redirect()->route('hrm.leave-balances.index', ['year' => $toYear])
                ->with('success', "تم إنشاء {$createdCount} رصيد جديد وتحديث {$updatedCount} رصيد موجود للسنة {$toYear}.");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء أرصدة السنة الجديدة: ' . $e->getMessage());
        }
    }

    /**
     * حساب الأيام المستخدمة للموظف ونوع الإجازة والسنة
     *
     * @param  int  $employeeId
     * @param  int  $leaveTypeId
     * @param  int  $year
     * @return float
     */
    private function calculateUsedDays($employeeId, $leaveTypeId, $year)
    {
        return Leave::where('employee_id', $employeeId)
            ->where('leave_type_id', $leaveTypeId)
            ->where('status', 'approved')
            ->whereYear('start_date', $year)
            ->sum('total_days');
    }

    /**
     * حساب الأيام المعلقة للموظف ونوع الإجازة والسنة
     *
     * @param  int  $employeeId
     * @param  int  $leaveTypeId
     * @param  int  $year
     * @return float
     */
    private function calculatePendingDays($employeeId, $leaveTypeId, $year)
    {
        return Leave::where('employee_id', $employeeId)
            ->where('leave_type_id', $leaveTypeId)
            ->where('status', 'pending')
            ->whereYear('start_date', $year)
            ->sum('total_days');
    }
}
