<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Employee;
use App\Models\HRM\EmployeeTask;
use App\Models\HRM\EmployeeTaskAssignee;
use App\Models\HRM\EmployeeTaskAttachment;
use App\Models\HRM\EmployeeTaskComment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class EmployeeTaskController extends Controller
{
    /**
     * عرض قائمة المهام
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $userId = Auth::id();
        $isAdmin = Auth::user()->hasRole('admin');

        // إذا كان المستخدم مدير، يمكنه رؤية جميع المهام
        // وإلا، يرى فقط المهام المسندة إليه أو التي أنشأها
        $query = EmployeeTask::with(['assignees.employee'])
            ->where('company_id', $companyId);

        if (!$isAdmin) {
            $employeeId = Employee::where('user_id', $userId)->value('id');

            if ($employeeId) {
                $query->where(function ($q) use ($userId, $employeeId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId)
                        ->orWhereHas('assignees', function ($q2) use ($employeeId) {
                            $q2->where('employee_id', $employeeId);
                        });
                });
            } else {
                $query->where(function ($q) use ($userId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId);
                });
            }
        }

        // تطبيق الفلاتر إذا وجدت
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority != '') {
            $query->where('priority', $request->priority);
        }

        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->whereHas('assignees', function ($q) use ($request) {
                $q->where('employee_id', $request->employee_id);
            });
        }

        if ($request->has('start_date') && $request->start_date != '') {
            $query->whereDate('start_date', '>=', $request->start_date);
        }

        if ($request->has('due_date') && $request->due_date != '') {
            $query->whereDate('due_date', '<=', $request->due_date);
        }

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $tasks = $query->orderBy('due_date', 'asc')->paginate(10);
        $employees = Employee::where('company_id', $companyId)->get();

        return view('hrm.tasks.index', compact('tasks', 'employees'));
    }

    /**
     * عرض نموذج إنشاء مهمة جديدة
     */
    public function create()
    {
        $companyId = session('company_id');
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.tasks.create', compact('employees'));
    }

    /**
     * حفظ مهمة جديدة
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'status' => 'required|in:pending,in_progress,completed,cancelled,on_hold',
            'start_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:start_date',
            'notes' => 'nullable|string',
            'employees' => 'required|array',
            'employees.*' => 'exists:employees,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = session('company_id');
        $data = $request->all();
        $data['company_id'] = $companyId;
        $data['created_by'] = Auth::id();
        $data['assigned_by'] = Auth::id();

        $task = EmployeeTask::create($data);

        // إضافة الموظفين المكلفين بالمهمة
        foreach ($request->employees as $employeeId) {
            EmployeeTaskAssignee::create([
                'company_id' => $companyId,
                'task_id' => $task->id,
                'employee_id' => $employeeId,
                'status' => 'pending',
                'progress' => 0,
                'created_by' => Auth::id(),
            ]);
        }

        return redirect()->route('hrm.tasks.index')
            ->with('success', 'تم إنشاء المهمة بنجاح');
    }

    /**
     * عرض تفاصيل المهمة
     */
    public function show($id)
    {
        $companyId = session('company_id');
        $task = EmployeeTask::with(['assignees.employee', 'comments.user', 'attachments'])
            ->where('company_id', $companyId)
            ->findOrFail($id);

        return view('hrm.tasks.show', compact('task'));
    }

    /**
     * عرض نموذج تعديل المهمة
     */
    public function edit($id)
    {
        $companyId = session('company_id');
        $task = EmployeeTask::where('company_id', $companyId)->findOrFail($id);

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $assignees = EmployeeTaskAssignee::where('task_id', $id)
            ->pluck('employee_id')
            ->toArray();

        return view('hrm.tasks.edit', compact('task', 'employees', 'assignees'));
    }

    /**
     * تحديث المهمة
     */
    public function update(Request $request, $id)
    {
        $companyId = session('company_id');
        $task = EmployeeTask::where('company_id', $companyId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'status' => 'required|in:pending,in_progress,completed,cancelled,on_hold',
            'start_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:start_date',
            'completed_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'employees' => 'required|array',
            'employees.*' => 'exists:employees,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();
        $data['updated_by'] = Auth::id();

        // إذا تم تغيير الحالة إلى "مكتملة"، قم بتعيين تاريخ الإكمال
        if ($request->status == 'completed' && $task->status != 'completed') {
            $data['completed_date'] = now();
        }

        // إذا تم تغيير الحالة من "مكتملة" إلى حالة أخرى، قم بإزالة تاريخ الإكمال
        if ($request->status != 'completed' && $task->status == 'completed') {
            $data['completed_date'] = null;
        }

        $task->update($data);

        // تحديث الموظفين المكلفين بالمهمة
        // حذف المكلفين الحاليين غير الموجودين في القائمة الجديدة
        EmployeeTaskAssignee::where('task_id', $id)
            ->whereNotIn('employee_id', $request->employees)
            ->delete();

        // إضافة المكلفين الجدد
        $existingAssignees = EmployeeTaskAssignee::where('task_id', $id)
            ->pluck('employee_id')
            ->toArray();

        foreach ($request->employees as $employeeId) {
            if (!in_array($employeeId, $existingAssignees)) {
                EmployeeTaskAssignee::create([
                    'company_id' => $companyId,
                    'task_id' => $id,
                    'employee_id' => $employeeId,
                    'status' => 'pending',
                    'progress' => 0,
                    'created_by' => Auth::id(),
                ]);
            }
        }

        return redirect()->route('hrm.tasks.index')
            ->with('success', 'تم تحديث المهمة بنجاح');
    }

    /**
     * حذف المهمة
     */
    public function destroy($id)
    {
        $companyId = session('company_id');
        $task = EmployeeTask::where('company_id', $companyId)->findOrFail($id);

        // حذف المرفقات أولاً
        $attachments = EmployeeTaskAttachment::where('task_id', $id)->get();
        foreach ($attachments as $attachment) {
            Storage::disk('public')->delete($attachment->file_path);
            $attachment->delete();
        }

        // حذف التعليقات
        EmployeeTaskComment::where('task_id', $id)->delete();

        // حذف المكلفين
        EmployeeTaskAssignee::where('task_id', $id)->delete();

        // ثم حذف المهمة
        $task->delete();

        return redirect()->route('hrm.tasks.index')
            ->with('success', 'تم حذف المهمة بنجاح');
    }

    /**
     * تحديث حالة المهمة للموظف
     */
    public function updateAssigneeStatus(Request $request, $id, $assigneeId)
    {
        $companyId = session('company_id');
        $assignee = EmployeeTaskAssignee::where('company_id', $companyId)
            ->where('task_id', $id)
            ->where('id', $assigneeId)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'status' => 'required|in:pending,in_progress,completed,declined',
            'progress' => 'required|integer|min:0|max:100',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->only(['status', 'progress', 'notes']);
        $data['updated_by'] = Auth::id();

        $assignee->update($data);

        // إذا تم تحديث جميع المكلفين إلى "مكتمل"، قم بتحديث حالة المهمة
        $allCompleted = EmployeeTaskAssignee::where('task_id', $id)
            ->where('status', '!=', 'completed')
            ->count() == 0;

        if ($allCompleted) {
            $task = EmployeeTask::findOrFail($id);
            $task->update([
                'status' => 'completed',
                'completed_date' => now(),
                'updated_by' => Auth::id(),
            ]);
        }

        return redirect()->route('hrm.tasks.show', $id)
            ->with('success', 'تم تحديث حالة المهمة بنجاح');
    }

    /**
     * إضافة تعليق على المهمة
     */
    public function addComment(Request $request, $id)
    {
        $companyId = session('company_id');
        $task = EmployeeTask::where('company_id', $companyId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'comment' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        EmployeeTaskComment::create([
            'company_id' => $companyId,
            'task_id' => $id,
            'user_id' => Auth::id(),
            'comment' => $request->comment,
            'created_by' => Auth::id(),
        ]);

        return redirect()->route('hrm.tasks.show', $id)
            ->with('success', 'تم إضافة التعليق بنجاح');
    }

    /**
     * حذف تعليق
     */
    public function deleteComment($id, $commentId)
    {
        $companyId = session('company_id');
        $comment = EmployeeTaskComment::where('company_id', $companyId)
            ->where('task_id', $id)
            ->where('id', $commentId)
            ->firstOrFail();

        // التحقق من أن المستخدم هو صاحب التعليق أو مدير
        if ($comment->user_id != Auth::id() && !Auth::user()->hasRole('admin')) {
            return redirect()->back()->with('error', 'غير مصرح لك بحذف هذا التعليق');
        }

        $comment->delete();

        return redirect()->route('hrm.tasks.show', $id)
            ->with('success', 'تم حذف التعليق بنجاح');
    }

    /**
     * إضافة مرفق للمهمة
     */
    public function addAttachment(Request $request, $id)
    {
        $companyId = session('company_id');
        $task = EmployeeTask::where('company_id', $companyId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'file' => 'required|file|max:10240', // 10MB max
            'description' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $file = $request->file('file');
        $path = $file->store('tasks/attachments', 'public');

        EmployeeTaskAttachment::create([
            'company_id' => $companyId,
            'task_id' => $id,
            'user_id' => Auth::id(),
            'file_name' => $file->getClientOriginalName(),
            'file_path' => $path,
            'file_type' => $file->getClientMimeType(),
            'file_size' => $file->getSize(),
            'description' => $request->description,
            'created_by' => Auth::id(),
        ]);

        return redirect()->route('hrm.tasks.show', $id)
            ->with('success', 'تم إضافة المرفق بنجاح');
    }

    /**
     * حذف مرفق
     */
    public function deleteAttachment($id, $attachmentId)
    {
        $companyId = session('company_id');
        $attachment = EmployeeTaskAttachment::where('company_id', $companyId)
            ->where('task_id', $id)
            ->where('id', $attachmentId)
            ->firstOrFail();

        // التحقق من أن المستخدم هو من رفع المرفق أو مدير
        if ($attachment->user_id != Auth::id() && !Auth::user()->hasRole('admin')) {
            return redirect()->back()->with('error', 'غير مصرح لك بحذف هذا المرفق');
        }

        // حذف الملف من التخزين
        Storage::disk('public')->delete($attachment->file_path);

        $attachment->delete();

        return redirect()->route('hrm.tasks.show', $id)
            ->with('success', 'تم حذف المرفق بنجاح');
    }

    /**
     * تنزيل مرفق
     */
    public function downloadAttachment($id, $attachmentId)
    {
        $companyId = session('company_id');
        $attachment = EmployeeTaskAttachment::where('company_id', $companyId)
            ->where('task_id', $id)
            ->where('id', $attachmentId)
            ->firstOrFail();

        return Storage::disk('public')->download($attachment->file_path, $attachment->file_name);
    }

    /**
     * عرض المهام المتأخرة
     */
    public function overdueTasks(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $userId = Auth::id();
        $isAdmin = Auth::user()->hasRole('admin');

        // المهام المتأخرة (تاريخ الاستحقاق قبل اليوم والحالة ليست مكتملة أو ملغاة)
        $query = EmployeeTask::with(['assignees.employee'])
            ->where('company_id', $companyId)
            ->whereDate('due_date', '<', now())
            ->whereNotIn('status', ['completed', 'cancelled']);

        if (!$isAdmin) {
            $employeeId = Employee::where('user_id', $userId)->value('id');

            if ($employeeId) {
                $query->where(function ($q) use ($userId, $employeeId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId)
                        ->orWhereHas('assignees', function ($q2) use ($employeeId) {
                            $q2->where('employee_id', $employeeId);
                        });
                });
            } else {
                $query->where(function ($q) use ($userId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId);
                });
            }
        }

        $overdueTasks = $query->orderBy('due_date')->paginate(10);

        return view('hrm.tasks.overdue', compact('overdueTasks'));
    }

    /**
     * عرض المهام القادمة
     */
    public function upcomingTasks(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $userId = Auth::id();
        $isAdmin = Auth::user()->hasRole('admin');

        // المهام القادمة (تاريخ الاستحقاق خلال 7 أيام والحالة ليست مكتملة أو ملغاة)
        $query = EmployeeTask::with(['assignees.employee'])
            ->where('company_id', $companyId)
            ->whereDate('due_date', '>=', now())
            ->whereDate('due_date', '<=', now()->addDays(7))
            ->whereNotIn('status', ['completed', 'cancelled']);

        if (!$isAdmin) {
            $employeeId = Employee::where('user_id', $userId)->value('id');

            if ($employeeId) {
                $query->where(function ($q) use ($userId, $employeeId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId)
                        ->orWhereHas('assignees', function ($q2) use ($employeeId) {
                            $q2->where('employee_id', $employeeId);
                        });
                });
            } else {
                $query->where(function ($q) use ($userId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId);
                });
            }
        }

        $upcomingTasks = $query->orderBy('due_date')->paginate(10);

        return view('hrm.tasks.upcoming', compact('upcomingTasks'));
    }

    /**
     * عرض لوحة معلومات المهام
     */
    public function dashboard(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $userId = Auth::id();
        $isAdmin = Auth::user()->hasRole('admin');

        // إحصائيات المهام
        $query = EmployeeTask::where('company_id', $companyId);

        if (!$isAdmin) {
            $employeeId = Employee::where('user_id', $userId)->value('id');

            if ($employeeId) {
                $query->where(function ($q) use ($userId, $employeeId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId)
                        ->orWhereHas('assignees', function ($q2) use ($employeeId) {
                            $q2->where('employee_id', $employeeId);
                        });
                });
            } else {
                $query->where(function ($q) use ($userId) {
                    $q->where('created_by', $userId)
                        ->orWhere('assigned_by', $userId);
                });
            }
        }

        // إجمالي المهام
        $totalTasks = $query->count();

        // المهام حسب الحالة
        $tasksByStatus = [
            'pending' => (clone $query)->where('status', 'pending')->count(),
            'in_progress' => (clone $query)->where('status', 'in_progress')->count(),
            'completed' => (clone $query)->where('status', 'completed')->count(),
            'cancelled' => (clone $query)->where('status', 'cancelled')->count(),
            'on_hold' => (clone $query)->where('status', 'on_hold')->count(),
        ];

        // المهام حسب الأولوية
        $tasksByPriority = [
            'low' => (clone $query)->where('priority', 'low')->count(),
            'medium' => (clone $query)->where('priority', 'medium')->count(),
            'high' => (clone $query)->where('priority', 'high')->count(),
            'urgent' => (clone $query)->where('priority', 'urgent')->count(),
        ];

        // المهام المتأخرة
        $overdueTasks = (clone $query)
            ->whereDate('due_date', '<', now())
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->count();

        // المهام القادمة خلال الأسبوع
        $upcomingTasks = (clone $query)
            ->whereDate('due_date', '>=', now())
            ->whereDate('due_date', '<=', now()->addDays(7))
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->count();

        // المهام المكتملة هذا الشهر
        $completedThisMonth = (clone $query)
            ->where('status', 'completed')
            ->whereMonth('completed_date', now()->month)
            ->whereYear('completed_date', now()->year)
            ->count();

        // أحدث المهام
        $recentTasks = (clone $query)
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // المهام القادمة
        $nextTasks = (clone $query)
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->whereDate('due_date', '>=', now())
            ->orderBy('due_date', 'asc')
            ->limit(5)
            ->get();

        return view('hrm.tasks.dashboard', compact(
            'totalTasks',
            'tasksByStatus',
            'tasksByPriority',
            'overdueTasks',
            'upcomingTasks',
            'completedThisMonth',
            'recentTasks',
            'nextTasks'
        ));
    }
}
