<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Employee;
use App\Models\HRM\EmployeeEvaluation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class EmployeeEvaluationController extends Controller
{
    /**
     * عرض قائمة تقييمات الموظفين
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = EmployeeEvaluation::with(['employee', 'evaluator'])
            ->where('company_id', $companyId);

        // تطبيق الفلاتر إذا وجدت
        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('period') && $request->period != '') {
            $query->where('evaluation_period', 'like', "%{$request->period}%");
        }

        if ($request->has('date_from') && $request->date_from != '') {
            $query->whereDate('evaluation_date', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to != '') {
            $query->whereDate('evaluation_date', '<=', $request->date_to);
        }

        $evaluations = $query->orderBy('evaluation_date', 'desc')->paginate(10);
        $employees = Employee::where('company_id', $companyId)->get();
        $statuses = ['draft' => 'مسودة', 'completed' => 'مكتمل', 'acknowledged' => 'تم الاطلاع'];

        return view('hrm.evaluations.index', compact('evaluations', 'employees', 'statuses'));
    }

    /**
     * عرض نموذج إنشاء تقييم جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $employees = Employee::where('company_id', $companyId)->get();
        $evaluators = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'manager', 'supervisor']);
        })->get();

        return view('hrm.evaluations.create', compact('employees', 'evaluators'));
    }

    /**
     * تخزين تقييم جديد في قاعدة البيانات
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'evaluator_id' => 'required|exists:users,id',
            'evaluation_date' => 'required|date',
            'evaluation_period' => 'required|string|max:100',
            'performance_score' => 'required|numeric|min:0|max:10',
            'attendance_score' => 'required|numeric|min:0|max:10',
            'quality_score' => 'required|numeric|min:0|max:10',
            'teamwork_score' => 'required|numeric|min:0|max:10',
            'initiative_score' => 'required|numeric|min:0|max:10',
            'communication_score' => 'required|numeric|min:0|max:10',
            'leadership_score' => 'required|numeric|min:0|max:10',
            'technical_score' => 'required|numeric|min:0|max:10',
            'strengths' => 'nullable|array',
            'areas_for_improvement' => 'nullable|array',
            'goals' => 'nullable|array',
            'comments' => 'nullable|string',
            'status' => 'required|in:draft,completed',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');

        // حساب الدرجة الإجمالية
        $totalScore = (
            $request->performance_score +
            $request->attendance_score +
            $request->quality_score +
            $request->teamwork_score +
            $request->initiative_score +
            $request->communication_score +
            $request->leadership_score +
            $request->technical_score
        ) / 8;

        // إنشاء تقييم جديد
        $evaluation = new EmployeeEvaluation();
        $evaluation->company_id = $companyId;
        $evaluation->employee_id = $request->employee_id;
        $evaluation->evaluator_id = $request->evaluator_id;
        $evaluation->evaluation_date = $request->evaluation_date;
        $evaluation->evaluation_period = $request->evaluation_period;
        $evaluation->performance_score = $request->performance_score;
        $evaluation->attendance_score = $request->attendance_score;
        $evaluation->quality_score = $request->quality_score;
        $evaluation->teamwork_score = $request->teamwork_score;
        $evaluation->initiative_score = $request->initiative_score;
        $evaluation->communication_score = $request->communication_score;
        $evaluation->leadership_score = $request->leadership_score;
        $evaluation->technical_score = $request->technical_score;
        $evaluation->total_score = $totalScore;
        $evaluation->strengths = $request->strengths;
        $evaluation->areas_for_improvement = $request->areas_for_improvement;
        $evaluation->goals = $request->goals;
        $evaluation->comments = $request->comments;
        $evaluation->status = $request->status;
        $evaluation->created_by = Auth::id();
        $evaluation->save();

        return redirect()->route('hrm.evaluations.show', $evaluation->id)
            ->with('success', 'تم إنشاء التقييم بنجاح.');
    }

    /**
     * عرض تفاصيل تقييم محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $evaluation = EmployeeEvaluation::with(['employee', 'evaluator'])->findOrFail($id);

        // التحقق من أن التقييم ينتمي للشركة الحالية
        $companyId = session('company_id');
        if ($evaluation->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        return view('hrm.evaluations.show', compact('evaluation'));
    }

    /**
     * عرض نموذج تعديل تقييم محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $companyId = $request->session()->get('company_id');
        $evaluation = EmployeeEvaluation::findOrFail($id);

        // التحقق من أن التقييم ينتمي للشركة الحالية
        if ($evaluation->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // لا يمكن تعديل التقييمات المعتمدة
        if ($evaluation->status == 'acknowledged') {
            return redirect()->route('hrm.evaluations.show', $id)
                ->with('error', 'لا يمكن تعديل التقييمات التي تم الاطلاع عليها.');
        }

        $employees = Employee::where('company_id', $companyId)->get();
        $evaluators = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'manager', 'supervisor']);
        })->get();

        return view('hrm.evaluations.edit', compact('evaluation', 'employees', 'evaluators'));
    }

    /**
     * تحديث تقييم محدد في قاعدة البيانات
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $evaluation = EmployeeEvaluation::findOrFail($id);

        // التحقق من أن التقييم ينتمي للشركة الحالية
        $companyId = $request->session()->get('company_id');
        if ($evaluation->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // لا يمكن تعديل التقييمات المعتمدة
        if ($evaluation->status == 'acknowledged') {
            return redirect()->route('hrm.evaluations.show', $id)
                ->with('error', 'لا يمكن تعديل التقييمات التي تم الاطلاع عليها.');
        }

        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'evaluator_id' => 'required|exists:users,id',
            'evaluation_date' => 'required|date',
            'evaluation_period' => 'required|string|max:100',
            'performance_score' => 'required|numeric|min:0|max:10',
            'attendance_score' => 'required|numeric|min:0|max:10',
            'quality_score' => 'required|numeric|min:0|max:10',
            'teamwork_score' => 'required|numeric|min:0|max:10',
            'initiative_score' => 'required|numeric|min:0|max:10',
            'communication_score' => 'required|numeric|min:0|max:10',
            'leadership_score' => 'required|numeric|min:0|max:10',
            'technical_score' => 'required|numeric|min:0|max:10',
            'strengths' => 'nullable|array',
            'areas_for_improvement' => 'nullable|array',
            'goals' => 'nullable|array',
            'comments' => 'nullable|string',
            'status' => 'required|in:draft,completed',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // حساب الدرجة الإجمالية
        $totalScore = (
            $request->performance_score +
            $request->attendance_score +
            $request->quality_score +
            $request->teamwork_score +
            $request->initiative_score +
            $request->communication_score +
            $request->leadership_score +
            $request->technical_score
        ) / 8;

        // تحديث التقييم
        $evaluation->employee_id = $request->employee_id;
        $evaluation->evaluator_id = $request->evaluator_id;
        $evaluation->evaluation_date = $request->evaluation_date;
        $evaluation->evaluation_period = $request->evaluation_period;
        $evaluation->performance_score = $request->performance_score;
        $evaluation->attendance_score = $request->attendance_score;
        $evaluation->quality_score = $request->quality_score;
        $evaluation->teamwork_score = $request->teamwork_score;
        $evaluation->initiative_score = $request->initiative_score;
        $evaluation->communication_score = $request->communication_score;
        $evaluation->leadership_score = $request->leadership_score;
        $evaluation->technical_score = $request->technical_score;
        $evaluation->total_score = $totalScore;
        $evaluation->strengths = $request->strengths;
        $evaluation->areas_for_improvement = $request->areas_for_improvement;
        $evaluation->goals = $request->goals;
        $evaluation->comments = $request->comments;
        $evaluation->status = $request->status;
        $evaluation->updated_by = Auth::id();
        $evaluation->save();

        return redirect()->route('hrm.evaluations.show', $evaluation->id)
            ->with('success', 'تم تحديث التقييم بنجاح.');
    }

    /**
     * حذف تقييم محدد من قاعدة البيانات
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        $evaluation = EmployeeEvaluation::findOrFail($id);

        // التحقق من أن التقييم ينتمي للشركة الحالية
        $companyId = $request->session()->get('company_id');
        if ($evaluation->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // لا يمكن حذف التقييمات المعتمدة
        if ($evaluation->status == 'acknowledged') {
            return redirect()->route('hrm.evaluations.index')
                ->with('error', 'لا يمكن حذف التقييمات التي تم الاطلاع عليها.');
        }

        $evaluation->delete();

        return redirect()->route('hrm.evaluations.index')
            ->with('success', 'تم حذف التقييم بنجاح.');
    }

    /**
     * إضافة تعليقات الموظف على التقييم
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function addEmployeeComments(Request $request, $id)
    {
        $evaluation = EmployeeEvaluation::findOrFail($id);

        // التحقق من أن التقييم ينتمي للشركة الحالية
        $companyId = $request->session()->get('company_id');
        if ($evaluation->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        $validator = Validator::make($request->all(), [
            'employee_comments' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $evaluation->employee_comments = $request->employee_comments;
        $evaluation->status = 'acknowledged';
        $evaluation->save();

        return redirect()->route('hrm.evaluations.show', $evaluation->id)
            ->with('success', 'تم إضافة تعليقات الموظف بنجاح.');
    }

    /**
     * تصدير تقرير التقييم بصيغة PDF
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function exportPdf($id)
    {
        $evaluation = EmployeeEvaluation::with(['employee', 'evaluator'])->findOrFail($id);

        // التحقق من أن التقييم ينتمي للشركة الحالية
        $companyId = session('company_id');
        if ($evaluation->company_id != $companyId) {
            abort(403, 'غير مصرح بهذا الإجراء.');
        }

        // هنا يمكن إضافة كود لإنشاء ملف PDF
        // يمكن استخدام مكتبات مثل dompdf أو mpdf

        return redirect()->route('hrm.evaluations.show', $evaluation->id)
            ->with('info', 'سيتم تنفيذ تصدير PDF في الإصدار القادم.');
    }

    /**
     * عرض تقرير تقييمات الموظفين
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function report(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $employees = Employee::where('company_id', $companyId)->get();

        $employeeId = $request->input('employee_id');
        $period = $request->input('period');
        $year = $request->input('year');

        $query = EmployeeEvaluation::with(['employee'])
            ->where('company_id', $companyId)
            ->where('status', 'completed');

        if ($employeeId) {
            $query->where('employee_id', $employeeId);
        }

        if ($period) {
            $query->where('evaluation_period', 'like', "%{$period}%");
        }

        if ($year) {
            $query->whereYear('evaluation_date', $year);
        }

        $evaluations = $query->get();

        // تجميع البيانات للتقرير
        $reportData = [];
        foreach ($evaluations as $evaluation) {
            $employeeId = $evaluation->employee_id;

            if (!isset($reportData[$employeeId])) {
                $reportData[$employeeId] = [
                    'employee' => $evaluation->employee,
                    'evaluations_count' => 0,
                    'total_score' => 0,
                    'average_score' => 0,
                    'performance_avg' => 0,
                    'attendance_avg' => 0,
                    'quality_avg' => 0,
                    'teamwork_avg' => 0,
                    'initiative_avg' => 0,
                    'communication_avg' => 0,
                    'leadership_avg' => 0,
                    'technical_avg' => 0,
                    'evaluations' => []
                ];
            }

            $reportData[$employeeId]['evaluations_count']++;
            $reportData[$employeeId]['total_score'] += $evaluation->total_score;
            $reportData[$employeeId]['performance_avg'] += $evaluation->performance_score;
            $reportData[$employeeId]['attendance_avg'] += $evaluation->attendance_score;
            $reportData[$employeeId]['quality_avg'] += $evaluation->quality_score;
            $reportData[$employeeId]['teamwork_avg'] += $evaluation->teamwork_score;
            $reportData[$employeeId]['initiative_avg'] += $evaluation->initiative_score;
            $reportData[$employeeId]['communication_avg'] += $evaluation->communication_score;
            $reportData[$employeeId]['leadership_avg'] += $evaluation->leadership_score;
            $reportData[$employeeId]['technical_avg'] += $evaluation->technical_score;
            $reportData[$employeeId]['evaluations'][] = $evaluation;
        }

        // حساب المتوسطات
        foreach ($reportData as &$data) {
            $count = $data['evaluations_count'];
            if ($count > 0) {
                $data['average_score'] = $data['total_score'] / $count;
                $data['performance_avg'] /= $count;
                $data['attendance_avg'] /= $count;
                $data['quality_avg'] /= $count;
                $data['teamwork_avg'] /= $count;
                $data['initiative_avg'] /= $count;
                $data['communication_avg'] /= $count;
                $data['leadership_avg'] /= $count;
                $data['technical_avg'] /= $count;
            }
        }

        return view('hrm.evaluations.report', compact('reportData', 'employees', 'employeeId', 'period', 'year'));
    }
}
