<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Certificate;
use App\Models\HRM\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CertificateController extends Controller
{
    /**
     * عرض قائمة الشهادات
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = Certificate::with(['employee'])
            ->where('company_id', $companyId);

        // تطبيق الفلاتر إذا وجدت
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->has('certificate_type') && $request->certificate_type != '') {
            $query->where('certificate_type', $request->certificate_type);
        }

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('certificate_number', 'like', "%{$search}%")
                    ->orWhere('issuing_organization', 'like', "%{$search}%");
            });
        }

        $certificates = $query->orderBy('issue_date', 'desc')->paginate(10);
        $employees = Employee::where('company_id', $companyId)->get();

        // استخراج أنواع الشهادات الفريدة
        $certificateTypes = Certificate::where('company_id', $companyId)
            ->distinct()
            ->pluck('certificate_type')
            ->toArray();

        return view('hrm.certificates.index', compact('certificates', 'employees', 'certificateTypes'));
    }

    /**
     * عرض نموذج إنشاء شهادة جديدة
     */
    public function create()
    {
        $companyId = session('company_id');
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.certificates.create', compact('employees'));
    }

    /**
     * حفظ شهادة جديدة
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'title' => 'required|string|max:255',
            'certificate_type' => 'required|string|max:255',
            'issuing_organization' => 'required|string|max:255',
            'issue_date' => 'required|date',
            'expiry_date' => 'nullable|date|after_or_equal:issue_date',
            'certificate_number' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'verification_link' => 'nullable|url|max:255',
            'status' => 'required|in:active,expired,revoked',
            'notes' => 'nullable|string',
            'certificate_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = session('company_id');
        $data = $request->all();
        $data['company_id'] = $companyId;
        $data['created_by'] = Auth::id();

        // تخزين ملف الشهادة إذا تم تحميله
        if ($request->hasFile('certificate_file')) {
            $path = $request->file('certificate_file')->store('certificates', 'public');
            $data['file_path'] = $path;
        }

        Certificate::create($data);

        return redirect()->route('hrm.certificates.index')
            ->with('success', 'تم إنشاء الشهادة بنجاح');
    }

    /**
     * عرض تفاصيل الشهادة
     */
    public function show($id)
    {
        $companyId = session('company_id');
        $certificate = Certificate::with(['employee'])
            ->where('company_id', $companyId)
            ->findOrFail($id);

        return view('hrm.certificates.show', compact('certificate'));
    }

    /**
     * عرض نموذج تعديل الشهادة
     */
    public function edit($id)
    {
        $companyId = session('company_id');
        $certificate = Certificate::where('company_id', $companyId)->findOrFail($id);

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.certificates.edit', compact('certificate', 'employees'));
    }

    /**
     * تحديث الشهادة
     */
    public function update(Request $request, $id)
    {
        $companyId = session('company_id');
        $certificate = Certificate::where('company_id', $companyId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'title' => 'required|string|max:255',
            'certificate_type' => 'required|string|max:255',
            'issuing_organization' => 'required|string|max:255',
            'issue_date' => 'required|date',
            'expiry_date' => 'nullable|date|after_or_equal:issue_date',
            'certificate_number' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'verification_link' => 'nullable|url|max:255',
            'status' => 'required|in:active,expired,revoked',
            'notes' => 'nullable|string',
            'certificate_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();
        $data['updated_by'] = Auth::id();

        // تخزين ملف الشهادة الجديد إذا تم تحميله
        if ($request->hasFile('certificate_file')) {
            // حذف الملف القديم إذا وجد
            if ($certificate->file_path) {
                Storage::disk('public')->delete($certificate->file_path);
            }

            $path = $request->file('certificate_file')->store('certificates', 'public');
            $data['file_path'] = $path;
        }

        $certificate->update($data);

        return redirect()->route('hrm.certificates.index')
            ->with('success', 'تم تحديث الشهادة بنجاح');
    }

    /**
     * حذف الشهادة
     */
    public function destroy($id)
    {
        $companyId = session('company_id');
        $certificate = Certificate::where('company_id', $companyId)->findOrFail($id);

        // حذف ملف الشهادة إذا وجد
        if ($certificate->file_path) {
            Storage::disk('public')->delete($certificate->file_path);
        }

        $certificate->delete();

        return redirect()->route('hrm.certificates.index')
            ->with('success', 'تم حذف الشهادة بنجاح');
    }

    /**
     * تنزيل ملف الشهادة
     */
    public function downloadCertificate($id)
    {
        $companyId = session('company_id');
        $certificate = Certificate::where('company_id', $companyId)->findOrFail($id);

        if (!$certificate->file_path) {
            return redirect()->back()->with('error', 'لا يوجد ملف شهادة مرفق');
        }

        return Storage::disk('public')->download($certificate->file_path, $certificate->title . ' - ' . $certificate->employee->full_name . '.pdf');
    }

    /**
     * عرض الشهادات المنتهية أو قاربت على الانتهاء
     */
    public function expiringSoon(Request $request)
    {
        $companyId = $request->session()->get('company_id');

        // الشهادات التي ستنتهي خلال 30 يوم
        $expiringCertificates = Certificate::with(['employee'])
            ->where('company_id', $companyId)
            ->where('status', 'active')
            ->whereNotNull('expiry_date')
            ->whereDate('expiry_date', '>=', now())
            ->whereDate('expiry_date', '<=', now()->addDays(30))
            ->orderBy('expiry_date')
            ->get();

        // الشهادات المنتهية
        $expiredCertificates = Certificate::with(['employee'])
            ->where('company_id', $companyId)
            ->where(function ($query) {
                $query->where('status', 'active')
                    ->whereDate('expiry_date', '<', now());
            })
            ->orderBy('expiry_date', 'desc')
            ->get();

        return view('hrm.certificates.expiring', compact('expiringCertificates', 'expiredCertificates'));
    }
}
