<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\HRM\Bonus;
use App\Models\HRM\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class BonusController extends Controller
{
    /**
     * عرض قائمة المكافآت
     */
    public function index(Request $request)
    {
        $companyId = session('company_id');

        $bonuses = Bonus::with(['employee', 'approver'])
            ->where('company_id', $companyId)
            ->when($request->employee_id, function ($query, $employeeId) {
                return $query->where('employee_id', $employeeId);
            })
            ->when($request->bonus_type, function ($query, $bonusType) {
                return $query->where('bonus_type', $bonusType);
            })
            ->when($request->status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($request->date_from, function ($query, $dateFrom) {
                return $query->where('date', '>=', $dateFrom);
            })
            ->when($request->date_to, function ($query, $dateTo) {
                return $query->where('date', '<=', $dateTo);
            })
            ->orderBy('date', 'desc')
            ->paginate(15);

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('first_name')
            ->get();

        $bonusTypes = Bonus::BONUS_TYPES;
        $statuses = Bonus::STATUSES;

        return view('hrm.bonuses.index', compact('bonuses', 'employees', 'bonusTypes', 'statuses'));
    }

    /**
     * عرض نموذج إنشاء مكافأة جديدة
     */
    public function create()
    {
        $companyId = session('company_id');

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('first_name')
            ->get();

        $bonusTypes = Bonus::BONUS_TYPES;

        return view('hrm.bonuses.create', compact('employees', 'bonusTypes'));
    }

    /**
     * حفظ مكافأة جديدة
     */
    public function store(Request $request)
    {
        $companyId = session('company_id');

        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'bonus_type' => 'required|in:' . implode(',', array_keys(Bonus::BONUS_TYPES)),
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'description' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $bonus = new Bonus();
            $bonus->company_id = $companyId;
            $bonus->employee_id = $request->employee_id;
            $bonus->bonus_type = $request->bonus_type;
            $bonus->amount = $request->amount;
            $bonus->date = $request->date;
            $bonus->description = $request->description;
            $bonus->status = 'pending';
            $bonus->notes = $request->notes;
            $bonus->created_by = Auth::id();
            $bonus->save();

            DB::commit();

            return redirect()->route('hrm.bonuses.index')
                ->with('success', 'تم إنشاء المكافأة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء المكافأة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * عرض تفاصيل المكافأة
     */
    public function show(Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        $bonus->load(['employee', 'approver', 'creator']);

        return view('hrm.bonuses.show', compact('bonus'));
    }

    /**
     * عرض نموذج تعديل المكافأة
     */
    public function edit(Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        if ($bonus->status != 'pending') {
            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('error', 'لا يمكن تعديل المكافأة بعد الموافقة عليها أو رفضها');
        }

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->orderBy('first_name')
            ->get();

        $bonusTypes = Bonus::BONUS_TYPES;

        return view('hrm.bonuses.edit', compact('bonus', 'employees', 'bonusTypes'));
    }

    /**
     * تحديث المكافأة
     */
    public function update(Request $request, Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        if ($bonus->status != 'pending') {
            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('error', 'لا يمكن تعديل المكافأة بعد الموافقة عليها أو رفضها');
        }

        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'bonus_type' => 'required|in:' . implode(',', array_keys(Bonus::BONUS_TYPES)),
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'description' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $bonus->employee_id = $request->employee_id;
            $bonus->bonus_type = $request->bonus_type;
            $bonus->amount = $request->amount;
            $bonus->date = $request->date;
            $bonus->description = $request->description;
            $bonus->notes = $request->notes;
            $bonus->updated_by = Auth::id();
            $bonus->save();

            DB::commit();

            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('success', 'تم تحديث المكافأة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث المكافأة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * حذف المكافأة
     */
    public function destroy(Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        if ($bonus->status != 'pending') {
            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('error', 'لا يمكن حذف المكافأة بعد الموافقة عليها أو رفضها');
        }

        try {
            DB::beginTransaction();

            $bonus->delete();

            DB::commit();

            return redirect()->route('hrm.bonuses.index')
                ->with('success', 'تم حذف المكافأة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف المكافأة: ' . $e->getMessage());
        }
    }

    /**
     * الموافقة على المكافأة
     */
    public function approve(Request $request, Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        if ($bonus->status != 'pending') {
            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('error', 'تم بالفعل اتخاذ إجراء على هذه المكافأة');
        }

        try {
            DB::beginTransaction();

            $bonus->status = 'approved';
            $bonus->approved_by = Auth::id();
            $bonus->approved_at = now();
            $bonus->save();

            DB::commit();

            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('success', 'تمت الموافقة على المكافأة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء الموافقة على المكافأة: ' . $e->getMessage());
        }
    }

    /**
     * رفض المكافأة
     */
    public function reject(Request $request, Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        if ($bonus->status != 'pending') {
            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('error', 'تم بالفعل اتخاذ إجراء على هذه المكافأة');
        }

        $request->validate([
            'rejection_reason' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            $bonus->status = 'rejected';
            $bonus->notes = $request->rejection_reason;
            $bonus->approved_by = Auth::id();
            $bonus->approved_at = now();
            $bonus->save();

            DB::commit();

            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('success', 'تم رفض المكافأة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء رفض المكافأة: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة المكافأة إلى مدفوعة
     */
    public function markAsPaid(Request $request, Bonus $bonus)
    {
        $companyId = session('company_id');

        if ($bonus->company_id != $companyId) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المكافأة');
        }

        if ($bonus->status != 'approved') {
            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('error', 'يجب الموافقة على المكافأة أولاً قبل تحديدها كمدفوعة');
        }

        try {
            DB::beginTransaction();

            $bonus->status = 'paid';
            $bonus->updated_by = Auth::id();
            $bonus->save();

            DB::commit();

            return redirect()->route('hrm.bonuses.show', $bonus)
                ->with('success', 'تم تحديث حالة المكافأة إلى مدفوعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث حالة المكافأة: ' . $e->getMessage());
        }
    }
}
