<?php

namespace App\Http\Controllers;

use App\Models\FiscalYear;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class FiscalYearController extends Controller
{
    /**
     * عرض قائمة السنوات المالية
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $fiscalYears = FiscalYear::where('company_id', Auth::user()->company_id)
            ->orderBy('start_date', 'desc')
            ->paginate(15);

        return view('accounting.fiscal_years.index', compact('fiscalYears'));
    }

    /**
     * عرض نموذج إنشاء سنة مالية جديدة
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('accounting.fiscal_years.create');
    }

    /**
     * حفظ سنة مالية جديدة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من عدم تداخل السنوات المالية
        $overlapping = FiscalYear::where('company_id', Auth::user()->company_id)
            ->where(function ($query) use ($request) {
                $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                    ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                    ->orWhere(function ($q) use ($request) {
                        $q->where('start_date', '<=', $request->start_date)
                            ->where('end_date', '>=', $request->end_date);
                    });
            })
            ->exists();

        if ($overlapping) {
            return redirect()->back()
                ->with('error', 'هناك تداخل مع سنة مالية أخرى في نفس الفترة')
                ->withInput();
        }

        // تحويل القيم المنطقية
        $data = $request->all();
        $data['is_active'] = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_closed'] = false;
        $data['company_id'] = Auth::user()->company_id;

        // إذا كانت هذه السنة المالية نشطة، نقوم بإلغاء تنشيط السنوات الأخرى
        if ($data['is_active']) {
            FiscalYear::where('company_id', Auth::user()->company_id)
                ->where('is_active', true)
                ->update(['is_active' => false]);
        }

        $fiscalYear = FiscalYear::create($data);

        // إنشاء الفترات المحاسبية
        $periodType = $request->input('period_type', 'monthly');
        $fiscalYear->createAccountingPeriods($periodType);

        return redirect()->route('fiscal-years.index')
            ->with('success', 'تم إنشاء السنة المالية بنجاح');
    }

    /**
     * عرض تفاصيل سنة مالية محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $fiscalYear = FiscalYear::where('company_id', Auth::user()->company_id)
            ->with(['accountingPeriods' => function ($query) {
                $query->orderBy('start_date');
            }])
            ->findOrFail($id);

        return view('accounting.fiscal_years.show', compact('fiscalYear'));
    }

    /**
     * عرض نموذج تعديل سنة مالية
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $fiscalYear = FiscalYear::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من أن السنة المالية غير مغلقة
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.show', $fiscalYear->id)
                ->with('error', 'لا يمكن تعديل السنة المالية لأنها مغلقة');
        }

        return view('accounting.fiscal_years.edit', compact('fiscalYear'));
    }

    /**
     * تحديث سنة مالية محددة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $fiscalYear = FiscalYear::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من أن السنة المالية غير مغلقة
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.show', $fiscalYear->id)
                ->with('error', 'لا يمكن تعديل السنة المالية لأنها مغلقة');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // تحويل القيم المنطقية
        $data = $request->all();
        $data['is_active'] = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);

        // إذا كانت هذه السنة المالية نشطة، نقوم بإلغاء تنشيط السنوات الأخرى
        if ($data['is_active'] && !$fiscalYear->is_active) {
            FiscalYear::where('company_id', Auth::user()->company_id)
                ->where('is_active', true)
                ->update(['is_active' => false]);
        }

        $fiscalYear->update([
            'name' => $data['name'],
            'is_active' => $data['is_active']
        ]);

        return redirect()->route('fiscal-years.index')
            ->with('success', 'تم تحديث السنة المالية بنجاح');
    }

    /**
     * إغلاق سنة مالية محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function close($id)
    {
        $fiscalYear = FiscalYear::where('company_id', Auth::user()->company_id)
            ->with('accountingPeriods')
            ->findOrFail($id);

        // التحقق من أن السنة المالية غير مغلقة بالفعل
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.show', $fiscalYear->id)
                ->with('error', 'السنة المالية مغلقة بالفعل');
        }

        // التحقق من إغلاق جميع الفترات المحاسبية
        $openPeriods = $fiscalYear->accountingPeriods()->where('is_closed', false)->count();
        if ($openPeriods > 0) {
            return redirect()->route('fiscal-years.show', $fiscalYear->id)
                ->with('error', 'لا يمكن إغلاق السنة المالية لوجود فترات محاسبية مفتوحة');
        }

        // إغلاق السنة المالية
        $fiscalYear->is_closed = true;
        $fiscalYear->is_active = false;
        $fiscalYear->save();

        return redirect()->route('fiscal-years.show', $fiscalYear->id)
            ->with('success', 'تم إغلاق السنة المالية بنجاح');
    }

    /**
     * إعادة فتح سنة مالية محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reopen($id)
    {
        $fiscalYear = FiscalYear::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من أن السنة المالية مغلقة
        if (!$fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.show', $fiscalYear->id)
                ->with('error', 'السنة المالية مفتوحة بالفعل');
        }

        // إعادة فتح السنة المالية
        $fiscalYear->is_closed = false;
        $fiscalYear->save();

        // إعادة فتح الفترات المحاسبية
        $fiscalYear->accountingPeriods()->update(['is_closed' => false]);

        return redirect()->route('fiscal-years.show', $fiscalYear->id)
            ->with('success', 'تم إعادة فتح السنة المالية بنجاح');
    }
}
