<?php

namespace App\Http\Controllers;

use App\Services\InvoiceService;
use App\Services\FinancialReportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FinancialReportController extends Controller
{
    protected $invoiceService;
    protected $financialReportService;

    /**
     * Create a new controller instance.
     *
     * @param  InvoiceService  $invoiceService
     * @param  FinancialReportService  $financialReportService
     * @return void
     */
    public function __construct(InvoiceService $invoiceService, FinancialReportService $financialReportService)
    {
        $this->invoiceService = $invoiceService;
        $this->financialReportService = $financialReportService;
        $this->middleware('auth');

        // تعديل الصلاحية للسماح بالوصول للمدير العام والمدير والمحاسب
        $this->middleware(function ($request, $next) {
            if (
                auth()->user()->hasRole('super_admin') ||
                auth()->user()->hasRole('admin') ||
                auth()->user()->hasRole('accountant')
            ) {
                return $next($request);
            }

            if (auth()->user()->hasPermissionTo('financial_reports.view')) {
                return $next($request);
            }

            abort(403, 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        });
    }

    /**
     * عرض صفحة التقارير المالية.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('financial_reports.index');
    }

    /**
     * عرض نموذج تقرير الإيرادات والمصروفات.
     *
     * @return \Illuminate\Http\Response
     */
    public function incomeExpenseForm()
    {
        return view('financial_reports.income_expense_form');
    }

    /**
     * إنشاء تقرير الإيرادات والمصروفات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function incomeExpense(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'group_by' => 'required|in:month,quarter,year',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        $groupBy = $request->input('group_by', 'month');
        $includeDetails = $request->has('include_details');

        $report = $this->financialReportService->generateIncomeExpenseReport(
            $companyId,
            $startDate,
            $endDate,
            $groupBy,
            $includeDetails
        );

        // حفظ التقرير في الجلسة للتصدير لاحقاً
        session(['income_expense_report' => $report]);

        return view('financial_reports.income_expense', compact('report'));
    }

    /**
     * عرض نموذج تقرير الربحية.
     *
     * @return \Illuminate\Http\Response
     */
    public function profitabilityForm()
    {
        return view('financial_reports.profitability_form');
    }

    /**
     * إنشاء تقرير الربحية.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function profitability(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'group_by' => 'required|in:customer,elevator,contract',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        $groupBy = $request->input('group_by', 'customer');
        $includeInactive = $request->has('include_inactive');

        $report = $this->financialReportService->generateProfitabilityReport(
            $companyId,
            $startDate,
            $endDate,
            $groupBy,
            $includeInactive
        );

        // حفظ التقرير في الجلسة للتصدير لاحقاً
        session(['profitability_report' => $report]);

        return view('financial_reports.profitability', compact('report'));
    }

    /**
     * تصدير التقرير المالي.
     *
     * @param  string  $type
     * @return \Illuminate\Http\Response
     */
    public function export($type)
    {
        if ($type === 'income_expense') {
            $report = session('income_expense_report');
            if (!$report) {
                return redirect()->route('financial-reports.income-expense-form')
                    ->with('error', 'لم يتم العثور على تقرير. يرجى إنشاء التقرير أولاً.');
            }

            // استخدام قالب PDF
            return view('financial_reports.pdf.income_expense', compact('report'));

            // يمكن استخدام مكتبة لتحويل HTML إلى PDF
            // $pdf = PDF::loadView('financial_reports.pdf.income_expense', compact('report'));
            // return $pdf->download('income_expense_report_' . date('Ymd') . '.pdf');
        } elseif ($type === 'profitability') {
            $report = session('profitability_report');
            if (!$report) {
                return redirect()->route('financial-reports.profitability-form')
                    ->with('error', 'لم يتم العثور على تقرير. يرجى إنشاء التقرير أولاً.');
            }

            // استخدام قالب PDF
            return view('financial_reports.pdf.profitability', compact('report'));

            // يمكن استخدام مكتبة لتحويل HTML إلى PDF
            // $pdf = PDF::loadView('financial_reports.pdf.profitability', compact('report'));
            // return $pdf->download('profitability_report_' . date('Ymd') . '.pdf');
        } else {
            return redirect()->route('financial-reports.index')
                ->with('error', 'نوع التقرير غير صالح.');
        }
    }

    /**
     * إنشاء محتوى تقرير الإيرادات والمصروفات.
     *
     * @param  array  $report
     * @return string
     */
    protected function generateIncomeExpenseReportContent($report)
    {
        $html = '<!DOCTYPE html>
        <html lang="ar" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <title>تقرير الإيرادات والمصروفات</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                }
                .report-header {
                    text-align: center;
                    margin-bottom: 30px;
                }
                .report-section {
                    margin-bottom: 30px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: right;
                }
                th {
                    background-color: #f2f2f2;
                }
                .total-row {
                    font-weight: bold;
                }
            </style>
        </head>
        <body>
            <div class="report-header">
                <h1>تقرير الإيرادات والمصروفات</h1>
                <p>الفترة: من ' . $report['period']['start_date'] . ' إلى ' . $report['period']['end_date'] . '</p>
            </div>
            
            <div class="report-section">
                <h2>الإيرادات</h2>
                <table>
                    <tr>
                        <th>البند</th>
                        <th>المبلغ</th>
                    </tr>
                    <tr>
                        <td>إجمالي الفواتير</td>
                        <td>' . number_format($report['income']['total_invoiced'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي المدفوعات المحصلة</td>
                        <td>' . number_format($report['income']['total_paid'], 2) . '</td>
                    </tr>
                    <tr class="total-row">
                        <td>إجمالي الإيرادات</td>
                        <td>' . number_format($report['income']['total_paid'], 2) . '</td>
                    </tr>
                </table>
            </div>
            
            <div class="report-section">
                <h2>المصروفات</h2>
                <table>
                    <tr>
                        <th>البند</th>
                        <th>المبلغ</th>
                    </tr>
                    <tr>
                        <td>إجمالي المصروفات</td>
                        <td>' . number_format($report['expenses']['total'], 2) . '</td>
                    </tr>
                    <tr class="total-row">
                        <td>إجمالي المصروفات</td>
                        <td>' . number_format($report['expenses']['total'], 2) . '</td>
                    </tr>
                </table>
            </div>
            
            <div class="report-section">
                <h2>ملخص</h2>
                <table>
                    <tr>
                        <th>البند</th>
                        <th>المبلغ</th>
                    </tr>
                    <tr>
                        <td>إجمالي الإيرادات</td>
                        <td>' . number_format($report['income']['total_paid'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي المصروفات</td>
                        <td>' . number_format($report['expenses']['total'], 2) . '</td>
                    </tr>
                    <tr class="total-row">
                        <td>صافي الدخل</td>
                        <td>' . number_format($report['summary']['net_income'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>نسبة الربح</td>
                        <td>' . number_format($report['summary']['profit_margin'], 2) . '%</td>
                    </tr>
                </table>
            </div>
        </body>
        </html>';

        return $html;
    }

    /**
     * إنشاء محتوى تقرير الربحية.
     *
     * @param  array  $report
     * @return string
     */
    protected function generateProfitabilityReportContent($report)
    {
        $customersHtml = '';
        foreach ($report['customers'] as $customer) {
            $customersHtml .= '<tr>
                <td>' . $customer['customer_name'] . '</td>
                <td>' . $customer['invoices_count'] . '</td>
                <td>' . number_format($customer['total_invoiced'], 2) . '</td>
                <td>' . number_format($customer['total_paid'], 2) . '</td>
                <td>' . number_format($customer['direct_costs'], 2) . '</td>
                <td>' . number_format($customer['profit'], 2) . '</td>
                <td>' . number_format($customer['profit_margin'], 2) . '%</td>
            </tr>';
        }

        $html = '<!DOCTYPE html>
        <html lang="ar" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <title>تقرير الربحية حسب العميل</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                }
                .report-header {
                    text-align: center;
                    margin-bottom: 30px;
                }
                .report-section {
                    margin-bottom: 30px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: right;
                }
                th {
                    background-color: #f2f2f2;
                }
                .total-row {
                    font-weight: bold;
                }
            </style>
        </head>
        <body>
            <div class="report-header">
                <h1>تقرير الربحية حسب العميل</h1>
                <p>الفترة: من ' . $report['period']['start_date'] . ' إلى ' . $report['period']['end_date'] . '</p>
            </div>
            
            <div class="report-section">
                <h2>ملخص</h2>
                <table>
                    <tr>
                        <th>البند</th>
                        <th>القيمة</th>
                    </tr>
                    <tr>
                        <td>عدد العملاء</td>
                        <td>' . $report['summary']['total_customers'] . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي الفواتير</td>
                        <td>' . number_format($report['summary']['total_invoiced'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي المدفوعات</td>
                        <td>' . number_format($report['summary']['total_paid'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي الربح</td>
                        <td>' . number_format($report['summary']['total_profit'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>متوسط نسبة الربح</td>
                        <td>' . number_format($report['summary']['average_profit_margin'], 2) . '%</td>
                    </tr>
                </table>
            </div>
            
            <div class="report-section">
                <h2>الربحية حسب العميل</h2>
                <table>
                    <thead>
                        <tr>
                            <th>العميل</th>
                            <th>عدد الفواتير</th>
                            <th>إجمالي الفواتير</th>
                            <th>إجمالي المدفوعات</th>
                            <th>التكاليف المباشرة</th>
                            <th>الربح</th>
                            <th>نسبة الربح</th>
                        </tr>
                    </thead>
                    <tbody>
                        ' . $customersHtml . '
                    </tbody>
                    <tfoot>
                        <tr class="total-row">
                            <td>الإجمالي</td>
                            <td>' . array_sum(array_column($report['customers'], 'invoices_count')) . '</td>
                            <td>' . number_format($report['summary']['total_invoiced'], 2) . '</td>
                            <td>' . number_format($report['summary']['total_paid'], 2) . '</td>
                            <td>' . number_format(array_sum(array_column($report['customers'], 'direct_costs')), 2) . '</td>
                            <td>' . number_format($report['summary']['total_profit'], 2) . '</td>
                            <td>' . number_format($report['summary']['average_profit_margin'], 2) . '%</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </body>
        </html>';

        return $html;
    }
}
