<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\EmployeeTask;
use App\Models\EmployeeTaskAssignee;
use App\Models\EmployeeTaskComment;
use App\Models\EmployeeTaskAttachment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class EmployeeTaskController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // التحقق من الصلاحيات
        $this->authorize('viewAny', EmployeeTask::class);

        // استعلام قاعدة البيانات
        $query = EmployeeTask::query();

        // تطبيق الفلاتر
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }

        if ($request->filled('employee_id')) {
            $query->whereHas('assignees', function ($q) use ($request) {
                $q->where('employee_id', $request->employee_id);
            });
        }

        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                    ->orWhere('description', 'like', "%{$searchTerm}%");
            });
        }

        // فلتر حسب التاريخ
        if ($request->filled('date_filter')) {
            $today = Carbon::today();

            if ($request->date_filter === 'today') {
                $query->whereDate('due_date', $today);
            } elseif ($request->date_filter === 'tomorrow') {
                $query->whereDate('due_date', $today->copy()->addDay());
            } elseif ($request->date_filter === 'this_week') {
                $query->whereBetween('due_date', [
                    $today->copy()->startOfWeek(),
                    $today->copy()->endOfWeek()
                ]);
            } elseif ($request->date_filter === 'next_week') {
                $query->whereBetween('due_date', [
                    $today->copy()->addWeek()->startOfWeek(),
                    $today->copy()->addWeek()->endOfWeek()
                ]);
            } elseif ($request->date_filter === 'this_month') {
                $query->whereMonth('due_date', $today->month)
                    ->whereYear('due_date', $today->year);
            } elseif ($request->date_filter === 'overdue') {
                $query->where('due_date', '<', $today)
                    ->whereNotIn('status', ['completed', 'cancelled']);
            }
        }

        // ترتيب النتائج
        $query->orderBy('due_date', 'asc');

        // الحصول على النتائج مع التقسيم إلى صفحات
        $tasks = $query->with(['assignees.employee', 'createdBy'])->paginate(15);

        // الحصول على قائمة الموظفين للفلتر
        $employees = Employee::orderBy('first_name')->get();

        // إحصائيات سريعة
        $totalTasks = EmployeeTask::count();
        $completedTasks = EmployeeTask::where('status', 'completed')->count();
        $inProgressTasks = EmployeeTask::where('status', 'in_progress')->count();
        $overdueTasks = EmployeeTask::where('due_date', '<', Carbon::today())
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->count();

        return view('hrm.tasks.index', compact(
            'tasks',
            'employees',
            'totalTasks',
            'completedTasks',
            'inProgressTasks',
            'overdueTasks'
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // التحقق من الصلاحيات
        $this->authorize('create', EmployeeTask::class);

        // الحصول على قائمة الموظفين
        $employees = Employee::orderBy('first_name')->get();

        return view('hrm.tasks.create', compact('employees'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // التحقق من الصلاحيات
        $this->authorize('create', EmployeeTask::class);

        // التحقق من البيانات
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|in:pending,in_progress,completed,cancelled',
            'priority' => 'required|in:low,medium,high,urgent',
            'due_date' => 'required|date',
            'assignees' => 'required|array|min:1',
            'assignees.*' => 'exists:employees,id',
            'attachments.*' => 'nullable|file|max:10240', // 10MB max
        ]);

        // بدء المعاملة
        DB::beginTransaction();

        try {
            // إنشاء المهمة
            $task = new EmployeeTask();
            $task->title = $validated['title'];
            $task->description = $validated['description'] ?? null;
            $task->status = $validated['status'];
            $task->priority = $validated['priority'];
            $task->due_date = $validated['due_date'];
            $task->created_by = Auth::id();
            $task->save();

            // إضافة المكلفين بالمهمة
            foreach ($validated['assignees'] as $employeeId) {
                $assignee = new EmployeeTaskAssignee();
                $assignee->task_id = $task->id;
                $assignee->employee_id = $employeeId;
                $assignee->save();
            }

            // معالجة المرفقات
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('task_attachments', 'public');

                    $attachment = new EmployeeTaskAttachment();
                    $attachment->task_id = $task->id;
                    $attachment->file_path = $path;
                    $attachment->file_name = $file->getClientOriginalName();
                    $attachment->file_type = $file->getClientMimeType();
                    $attachment->file_size = $file->getSize();
                    $attachment->uploaded_by = Auth::id();
                    $attachment->save();
                }
            }

            // إضافة تعليق أولي إذا تم تقديمه
            if ($request->filled('initial_comment')) {
                $comment = new EmployeeTaskComment();
                $comment->task_id = $task->id;
                $comment->comment = $request->initial_comment;
                $comment->user_id = Auth::id();
                $comment->save();
            }

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('hrm.tasks.show', $task->id)
                ->with('success', 'تم إنشاء المهمة بنجاح');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء المهمة: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function show(EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('view', $task);

        // تحميل العلاقات
        $task->load([
            'assignees.employee',
            'comments.user',
            'attachments.uploadedBy',
            'createdBy',
            'updatedBy'
        ]);

        // الحصول على قائمة الموظفين للإضافة
        $employees = Employee::orderBy('first_name')->get();

        // الحصول على المهام المرتبطة (إذا كانت هناك)
        $relatedTasks = EmployeeTask::whereHas('assignees', function ($query) use ($task) {
            $query->whereIn('employee_id', $task->assignees->pluck('employee_id'));
        })
            ->where('id', '!=', $task->id)
            ->limit(5)
            ->get();

        return view('hrm.tasks.show', compact('task', 'employees', 'relatedTasks'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function edit(EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('update', $task);

        // تحميل العلاقات
        $task->load('assignees.employee');

        // الحصول على قائمة الموظفين
        $employees = Employee::orderBy('first_name')->get();

        // الحصول على قائمة المكلفين الحاليين
        $currentAssignees = $task->assignees->pluck('employee_id')->toArray();

        return view('hrm.tasks.edit', compact('task', 'employees', 'currentAssignees'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('update', $task);

        // التحقق من البيانات
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|in:pending,in_progress,completed,cancelled',
            'priority' => 'required|in:low,medium,high,urgent',
            'due_date' => 'required|date',
            'assignees' => 'required|array|min:1',
            'assignees.*' => 'exists:employees,id',
            'attachments.*' => 'nullable|file|max:10240', // 10MB max
        ]);

        // بدء المعاملة
        DB::beginTransaction();

        try {
            // تحديث المهمة
            $task->title = $validated['title'];
            $task->description = $validated['description'] ?? null;
            $task->status = $validated['status'];
            $task->priority = $validated['priority'];
            $task->due_date = $validated['due_date'];
            $task->updated_by = Auth::id();

            // إذا تم تغيير الحالة إلى مكتملة، تحديث تاريخ الإكمال
            if ($task->status === 'completed' && $task->getOriginal('status') !== 'completed') {
                $task->completed_at = Carbon::now();
            } elseif ($task->status !== 'completed') {
                $task->completed_at = null;
            }

            $task->save();

            // تحديث المكلفين بالمهمة
            // حذف المكلفين الحاليين
            EmployeeTaskAssignee::where('task_id', $task->id)->delete();

            // إضافة المكلفين الجدد
            foreach ($validated['assignees'] as $employeeId) {
                $assignee = new EmployeeTaskAssignee();
                $assignee->task_id = $task->id;
                $assignee->employee_id = $employeeId;
                $assignee->save();
            }

            // معالجة المرفقات الجديدة
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('task_attachments', 'public');

                    $attachment = new EmployeeTaskAttachment();
                    $attachment->task_id = $task->id;
                    $attachment->file_path = $path;
                    $attachment->file_name = $file->getClientOriginalName();
                    $attachment->file_type = $file->getClientMimeType();
                    $attachment->file_size = $file->getSize();
                    $attachment->uploaded_by = Auth::id();
                    $attachment->save();
                }
            }

            // إضافة تعليق للتحديث إذا تم تقديمه
            if ($request->filled('update_comment')) {
                $comment = new EmployeeTaskComment();
                $comment->task_id = $task->id;
                $comment->comment = $request->update_comment;
                $comment->user_id = Auth::id();
                $comment->save();
            }

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('hrm.tasks.show', $task->id)
                ->with('success', 'تم تحديث المهمة بنجاح');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث المهمة: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function destroy(EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('delete', $task);

        // بدء المعاملة
        DB::beginTransaction();

        try {
            // حذف المرفقات من التخزين
            foreach ($task->attachments as $attachment) {
                Storage::disk('public')->delete($attachment->file_path);
            }

            // حذف المرفقات من قاعدة البيانات
            EmployeeTaskAttachment::where('task_id', $task->id)->delete();

            // حذف التعليقات
            EmployeeTaskComment::where('task_id', $task->id)->delete();

            // حذف المكلفين
            EmployeeTaskAssignee::where('task_id', $task->id)->delete();

            // حذف المهمة
            $task->delete();

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('hrm.tasks.index')
                ->with('success', 'تم حذف المهمة بنجاح');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف المهمة: ' . $e->getMessage());
        }
    }

    /**
     * إضافة تعليق جديد للمهمة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function addComment(Request $request, EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('update', $task);

        // التحقق من البيانات
        $validated = $request->validate([
            'comment' => 'required|string',
            'attachment' => 'nullable|file|max:10240', // 10MB max
        ]);

        // بدء المعاملة
        DB::beginTransaction();

        try {
            // إضافة التعليق
            $comment = new EmployeeTaskComment();
            $comment->task_id = $task->id;
            $comment->comment = $validated['comment'];
            $comment->user_id = Auth::id();
            $comment->save();

            // معالجة المرفق إذا وجد
            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');
                $path = $file->store('task_attachments', 'public');

                $attachment = new EmployeeTaskAttachment();
                $attachment->task_id = $task->id;
                $attachment->comment_id = $comment->id;
                $attachment->file_path = $path;
                $attachment->file_name = $file->getClientOriginalName();
                $attachment->file_type = $file->getClientMimeType();
                $attachment->file_size = $file->getSize();
                $attachment->uploaded_by = Auth::id();
                $attachment->save();
            }

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('hrm.tasks.show', $task->id)
                ->with('success', 'تم إضافة التعليق بنجاح');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إضافة التعليق: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة المهمة بسرعة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function updateStatus(Request $request, EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('update', $task);

        // التحقق من البيانات
        $validated = $request->validate([
            'status' => 'required|in:pending,in_progress,completed,cancelled',
        ]);

        // تحديث حالة المهمة
        $task->status = $validated['status'];

        // إذا تم تغيير الحالة إلى مكتملة، تحديث تاريخ الإكمال
        if ($task->status === 'completed' && $task->getOriginal('status') !== 'completed') {
            $task->completed_at = Carbon::now();
        } elseif ($task->status !== 'completed') {
            $task->completed_at = null;
        }

        $task->updated_by = Auth::id();
        $task->save();

        // إضافة تعليق تلقائي
        $statusText = [
            'pending' => 'قيد الانتظار',
            'in_progress' => 'قيد التنفيذ',
            'completed' => 'مكتملة',
            'cancelled' => 'ملغية'
        ];

        $comment = new EmployeeTaskComment();
        $comment->task_id = $task->id;
        $comment->comment = 'تم تغيير حالة المهمة إلى: ' . $statusText[$task->status];
        $comment->user_id = Auth::id();
        $comment->is_system = true;
        $comment->save();

        return redirect()->back()->with('success', 'تم تحديث حالة المهمة بنجاح');
    }

    /**
     * تحديث نسبة إنجاز المهمة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\EmployeeTask  $task
     * @return \Illuminate\Http\Response
     */
    public function updateProgress(Request $request, EmployeeTask $task)
    {
        // التحقق من الصلاحيات
        $this->authorize('update', $task);

        // التحقق من البيانات
        $validated = $request->validate([
            'progress' => 'required|integer|min:0|max:100',
        ]);

        // تحديث نسبة الإنجاز
        $oldProgress = $task->progress;
        $task->progress = $validated['progress'];
        $task->updated_by = Auth::id();

        // إذا كانت نسبة الإنجاز 100%، تحديث الحالة إلى مكتملة
        if ($task->progress == 100 && $task->status != 'completed') {
            $task->status = 'completed';
            $task->completed_at = Carbon::now();
        }

        $task->save();

        // إضافة تعليق تلقائي
        $comment = new EmployeeTaskComment();
        $comment->task_id = $task->id;
        $comment->comment = 'تم تحديث نسبة الإنجاز من ' . $oldProgress . '% إلى ' . $task->progress . '%';
        $comment->user_id = Auth::id();
        $comment->is_system = true;
        $comment->save();

        return redirect()->back()->with('success', 'تم تحديث نسبة الإنجاز بنجاح');
    }

    /**
     * حذف مرفق
     *
     * @param  \App\Models\EmployeeTaskAttachment  $attachment
     * @return \Illuminate\Http\Response
     */
    public function deleteAttachment(EmployeeTaskAttachment $attachment)
    {
        // التحقق من الصلاحيات
        $this->authorize('update', $attachment->task);

        // حذف الملف من التخزين
        Storage::disk('public')->delete($attachment->file_path);

        // حذف المرفق من قاعدة البيانات
        $attachment->delete();

        return redirect()->back()->with('success', 'تم حذف المرفق بنجاح');
    }

    /**
     * عرض المهام المتأخرة
     *
     * @return \Illuminate\Http\Response
     */
    public function overdueTasks()
    {
        // التحقق من الصلاحيات
        $this->authorize('viewAny', EmployeeTask::class);

        // الحصول على المهام المتأخرة
        $tasks = EmployeeTask::where('due_date', '<', Carbon::today())
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->with(['assignees.employee', 'createdBy'])
            ->orderBy('due_date', 'asc')
            ->paginate(15);

        return view('hrm.tasks.overdue', compact('tasks'));
    }

    /**
     * عرض المهام المكتملة
     *
     * @return \Illuminate\Http\Response
     */
    public function completedTasks()
    {
        // التحقق من الصلاحيات
        $this->authorize('viewAny', EmployeeTask::class);

        // الحصول على المهام المكتملة
        $tasks = EmployeeTask::where('status', 'completed')
            ->with(['assignees.employee', 'createdBy'])
            ->orderBy('completed_at', 'desc')
            ->paginate(15);

        return view('hrm.tasks.completed', compact('tasks'));
    }

    /**
     * عرض مهام الموظف
     *
     * @param  \App\Models\Employee  $employee
     * @return \Illuminate\Http\Response
     */
    public function employeeTasks(Employee $employee)
    {
        // التحقق من الصلاحيات
        $this->authorize('viewAny', EmployeeTask::class);

        // الحصول على مهام الموظف
        $tasks = EmployeeTask::whereHas('assignees', function ($query) use ($employee) {
            $query->where('employee_id', $employee->id);
        })
            ->with(['assignees.employee', 'createdBy'])
            ->orderBy('due_date', 'asc')
            ->paginate(15);

        return view('hrm.tasks.employee', compact('tasks', 'employee'));
    }

    /**
     * عرض لوحة معلومات المهام
     *
     * @return \Illuminate\Http\Response
     */
    public function dashboard()
    {
        // التحقق من الصلاحيات
        $this->authorize('viewAny', EmployeeTask::class);

        // إحصائيات المهام
        $totalTasks = EmployeeTask::count();
        $completedTasks = EmployeeTask::where('status', 'completed')->count();
        $inProgressTasks = EmployeeTask::where('status', 'in_progress')->count();
        $pendingTasks = EmployeeTask::where('status', 'pending')->count();
        $cancelledTasks = EmployeeTask::where('status', 'cancelled')->count();
        $overdueTasks = EmployeeTask::where('due_date', '<', Carbon::today())
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->count();

        // المهام حسب الأولوية
        $lowPriorityTasks = EmployeeTask::where('priority', 'low')->count();
        $mediumPriorityTasks = EmployeeTask::where('priority', 'medium')->count();
        $highPriorityTasks = EmployeeTask::where('priority', 'high')->count();
        $urgentPriorityTasks = EmployeeTask::where('priority', 'urgent')->count();

        // المهام المتأخرة
        $recentOverdueTasks = EmployeeTask::where('due_date', '<', Carbon::today())
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->with(['assignees.employee', 'createdBy'])
            ->orderBy('due_date', 'asc')
            ->limit(5)
            ->get();

        // المهام القادمة
        $upcomingTasks = EmployeeTask::where('due_date', '>=', Carbon::today())
            ->whereNotIn('status', ['completed', 'cancelled'])
            ->with(['assignees.employee', 'createdBy'])
            ->orderBy('due_date', 'asc')
            ->limit(5)
            ->get();

        // المهام المكتملة مؤخراً
        $recentCompletedTasks = EmployeeTask::where('status', 'completed')
            ->with(['assignees.employee', 'createdBy'])
            ->orderBy('completed_at', 'desc')
            ->limit(5)
            ->get();

        // أكثر الموظفين نشاطاً
        $topEmployees = Employee::withCount(['tasks' => function ($query) {
            $query->where('status', 'completed');
        }])
            ->orderBy('tasks_count', 'desc')
            ->limit(5)
            ->get();

        return view('hrm.tasks.dashboard', compact(
            'totalTasks',
            'completedTasks',
            'inProgressTasks',
            'pendingTasks',
            'cancelledTasks',
            'overdueTasks',
            'lowPriorityTasks',
            'mediumPriorityTasks',
            'highPriorityTasks',
            'urgentPriorityTasks',
            'recentOverdueTasks',
            'upcomingTasks',
            'recentCompletedTasks',
            'topEmployees'
        ));
    }
}
