<?php

namespace App\Http\Controllers;

use App\Models\ElevatorType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ElevatorTypeController extends Controller
{
    /**
     * إنشاء مثيل جديد من التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * عرض قائمة أنواع المصاعد.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorTypes = ElevatorType::where('company_id', $company_id)
            ->orderBy('name')
            ->paginate(10);

        return view('elevator-types.index', compact('elevatorTypes'));
    }

    /**
     * عرض نموذج إنشاء نوع مصعد جديد.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('elevator-types.create');
    }

    /**
     * تخزين نوع مصعد جديد في قاعدة البيانات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();

        $elevatorType = new ElevatorType();
        $elevatorType->company_id = $user->company_id;
        $elevatorType->name = $request->name;
        $elevatorType->code = $request->code;
        $elevatorType->description = $request->description;
        $elevatorType->is_active = $request->has('is_active');
        $elevatorType->save();

        return redirect()->route('elevator-types.index')
            ->with('success', 'تم إضافة نوع المصعد بنجاح');
    }

    /**
     * عرض نوع مصعد محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorType = ElevatorType::where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator-types.show', compact('elevatorType'));
    }

    /**
     * عرض نموذج تعديل نوع مصعد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorType = ElevatorType::where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator-types.edit', compact('elevatorType'));
    }

    /**
     * تحديث نوع مصعد محدد في قاعدة البيانات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorType = ElevatorType::where('company_id', $company_id)
            ->findOrFail($id);

        $elevatorType->name = $request->name;
        $elevatorType->code = $request->code;
        $elevatorType->description = $request->description;
        $elevatorType->is_active = $request->has('is_active');
        $elevatorType->save();

        return redirect()->route('elevator-types.index')
            ->with('success', 'تم تحديث نوع المصعد بنجاح');
    }

    /**
     * حذف نوع مصعد محدد من قاعدة البيانات.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorType = ElevatorType::where('company_id', $company_id)
            ->findOrFail($id);

        // التحقق مما إذا كان هناك مصاعد مرتبطة بهذا النوع
        if ($elevatorType->elevators()->count() > 0) {
            return redirect()->route('elevator-types.index')
                ->with('error', 'لا يمكن حذف نوع المصعد لأنه مرتبط بمصاعد');
        }

        $elevatorType->delete();

        return redirect()->route('elevator-types.index')
            ->with('success', 'تم حذف نوع المصعد بنجاح');
    }
}
