<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\ElevatorInspection;
use App\Models\ElevatorMaintenance;
use App\Models\ElevatorType;
use App\Models\ElevatorModel;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ElevatorReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * عرض صفحة التقارير الرئيسية
     */
    public function index()
    {
        $companyId = Auth::user()->company_id;

        // إحصائيات عامة
        $totalElevators = Elevator::where('company_id', $companyId)->count();
        $totalMaintenances = ElevatorMaintenance::where('company_id', $companyId)->count();
        $totalInspections = ElevatorInspection::where('company_id', $companyId)->count();

        // إحصائيات حسب النوع
        $elevatorsByType = Elevator::where('elevators.company_id', $companyId)
            ->select('type', DB::raw('count(*) as total'))
            ->groupBy('type')
            ->get()
            ->map(function ($item) {
                // تحويل قيم النوع إلى أسماء مقروءة
                $typeNames = [
                    'passenger' => 'مصعد ركاب',
                    'freight' => 'مصعد بضائع',
                    'hospital' => 'مصعد مستشفى',
                    'panoramic' => 'مصعد بانورامي',
                    'residential' => 'مصعد سكني',
                    'commercial' => 'مصعد تجاري',
                    'other' => 'أنواع أخرى'
                ];
                return [
                    'name' => $typeNames[$item->type] ?? $item->type,
                    'total' => $item->total
                ];
            });

        // إحصائيات حسب الموديل
        $elevatorsByModel = Elevator::where('elevators.company_id', $companyId)
            ->select('model', DB::raw('count(*) as total'))
            ->groupBy('model')
            ->get()
            ->map(function ($item) {
                return [
                    'name' => $item->model,
                    'total' => $item->total
                ];
            });

        // إحصائيات الصيانة
        $maintenanceStats = ElevatorMaintenance::where('company_id', $companyId)
            ->select('status', DB::raw('count(*) as total'))
            ->groupBy('status')
            ->get()
            ->pluck('total', 'status')
            ->toArray();

        // إحصائيات الفحص
        $inspectionStats = ElevatorInspection::where('company_id', $companyId)
            ->select('status', DB::raw('count(*) as total'))
            ->groupBy('status')
            ->get()
            ->pluck('total', 'status')
            ->toArray();

        return view('elevator-reports.index', compact(
            'totalElevators',
            'totalMaintenances',
            'totalInspections',
            'elevatorsByType',
            'elevatorsByModel',
            'maintenanceStats',
            'inspectionStats'
        ));
    }

    /**
     * تقرير حالة المصاعد
     */
    public function elevatorStatus()
    {
        $companyId = Auth::user()->company_id;

        $elevators = Elevator::where('elevators.company_id', $companyId)
            ->join('customers', 'elevators.customer_id', '=', 'customers.id')
            ->join('locations', 'elevators.location_id', '=', 'locations.id')
            ->select(
                'elevators.*',
                'customers.name as customer_name',
                'locations.name as building_name',
                'locations.address'
            )
            ->get()
            ->map(function ($elevator) {
                // تحويل قيم النوع إلى أسماء مقروءة
                $typeNames = [
                    'passenger' => 'مصعد ركاب',
                    'freight' => 'مصعد بضائع',
                    'hospital' => 'مصعد مستشفى',
                    'panoramic' => 'مصعد بانورامي',
                    'residential' => 'مصعد سكني',
                    'commercial' => 'مصعد تجاري',
                    'other' => 'أنواع أخرى'
                ];

                $elevator->type_name = $typeNames[$elevator->type] ?? $elevator->type;
                $elevator->model_name = $elevator->model;

                return $elevator;
            });

        return view('elevator-reports.elevator-status', compact('elevators'));
    }

    /**
     * تقرير الصيانة الدورية
     */
    public function maintenanceReport(Request $request)
    {
        $companyId = Auth::user()->company_id;

        $query = ElevatorMaintenance::where('elevator_maintenances.company_id', $companyId)
            ->join('elevators', 'elevator_maintenances.elevator_id', '=', 'elevators.id')
            ->join('customers', 'elevators.customer_id', '=', 'customers.id')
            ->join('locations', 'elevators.location_id', '=', 'locations.id')
            ->select(
                'elevator_maintenances.*',
                'elevators.serial_number',
                'locations.name as building_name',
                'customers.name as customer_name'
            );

        // تصفية حسب الحالة
        if ($request->has('status') && $request->status != '') {
            $query->where('elevator_maintenances.status', $request->status);
        }

        // تصفية حسب التاريخ
        if ($request->has('from_date') && $request->from_date != '') {
            $query->whereDate('elevator_maintenances.maintenance_date', '>=', $request->from_date);
        }

        if ($request->has('to_date') && $request->to_date != '') {
            $query->whereDate('elevator_maintenances.maintenance_date', '<=', $request->to_date);
        }

        $maintenances = $query->orderBy('elevator_maintenances.maintenance_date', 'desc')->get();

        return view('elevator-reports.maintenance-report', compact('maintenances'));
    }

    /**
     * تقرير عمليات الفحص
     */
    public function inspectionReport(Request $request)
    {
        $companyId = Auth::user()->company_id;

        $query = ElevatorInspection::where('elevator_inspections.company_id', $companyId)
            ->join('elevators', 'elevator_inspections.elevator_id', '=', 'elevators.id')
            ->join('customers', 'elevators.customer_id', '=', 'customers.id')
            ->join('locations', 'elevators.location_id', '=', 'locations.id')
            ->select(
                'elevator_inspections.*',
                'elevators.serial_number',
                'locations.name as building_name',
                'customers.name as customer_name'
            );

        // تصفية حسب النوع
        if ($request->has('inspection_type') && $request->inspection_type != '') {
            $query->where('elevator_inspections.inspection_type', $request->inspection_type);
        }

        // تصفية حسب الحالة
        if ($request->has('status') && $request->status != '') {
            $query->where('elevator_inspections.status', $request->status);
        }

        // تصفية حسب التاريخ
        if ($request->has('from_date') && $request->from_date != '') {
            $query->whereDate('elevator_inspections.inspection_date', '>=', $request->from_date);
        }

        if ($request->has('to_date') && $request->to_date != '') {
            $query->whereDate('elevator_inspections.inspection_date', '<=', $request->to_date);
        }

        $inspections = $query->orderBy('elevator_inspections.inspection_date', 'desc')->get();

        return view('elevator-reports.inspection-report', compact('inspections'));
    }

    /**
     * تقرير المصاعد حسب العميل
     */
    public function elevatorsByCustomer()
    {
        $companyId = Auth::user()->company_id;

        $customers = Customer::where('company_id', $companyId)
            ->withCount('elevators')
            ->having('elevators_count', '>', 0)
            ->orderBy('elevators_count', 'desc')
            ->get();

        return view('elevator-reports.elevators-by-customer', compact('customers'));
    }

    /**
     * تقرير المصاعد حسب النوع
     */
    public function elevatorsByType()
    {
        $companyId = Auth::user()->company_id;

        $typeNames = [
            'passenger' => 'مصعد ركاب',
            'freight' => 'مصعد بضائع',
            'hospital' => 'مصعد مستشفى',
            'panoramic' => 'مصعد بانورامي',
            'other' => 'أنواع أخرى'
        ];

        $typeCounts = Elevator::where('company_id', $companyId)
            ->select('type', DB::raw('count(*) as elevators_count'))
            ->groupBy('type')
            ->get();

        $types = collect($typeCounts)->map(function ($item) use ($typeNames) {
            return (object) [
                'id' => $item->type,
                'name' => $typeNames[$item->type] ?? $item->type,
                'description' => 'نوع المصعد: ' . ($typeNames[$item->type] ?? $item->type),
                'elevators_count' => $item->elevators_count
            ];
        });

        return view('elevator-reports.elevators-by-type', compact('types'));
    }

    /**
     * تقرير المصاعد حسب الموديل
     */
    public function elevatorsByModel()
    {
        $companyId = Auth::user()->company_id;

        $modelCounts = Elevator::where('company_id', $companyId)
            ->select('model', 'manufacturer', DB::raw('count(*) as elevators_count'))
            ->groupBy('model', 'manufacturer')
            ->get();

        $models = collect($modelCounts)->map(function ($item) {
            return (object) [
                'id' => $item->model,
                'name' => $item->model,
                'manufacturer' => $item->manufacturer,
                'description' => 'موديل مصعد من إنتاج ' . $item->manufacturer,
                'elevators_count' => $item->elevators_count
            ];
        });

        return view('elevator-reports.elevators-by-model', compact('models'));
    }
}
