<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Elevator;
use App\Models\CashTransaction;
use App\Models\BankTransaction;
use Illuminate\Support\Facades\DB;

class ElevatorPaymentReportController extends Controller
{
    /**
     * عرض تقرير مدفوعات المصاعد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->authorize('view_reports');

        // الحصول على جميع المصاعد
        $elevatorsQuery = Elevator::with('customer');

        // تطبيق الفلاتر إذا وجدت
        if ($request->has('customer_id') && $request->customer_id) {
            $elevatorsQuery->where('customer_id', $request->customer_id);
        }

        if ($request->has('elevator_id') && $request->elevator_id) {
            $elevatorsQuery->where('id', $request->elevator_id);
        }

        $elevators = $elevatorsQuery->get();

        // تجهيز بيانات التقرير
        $reportData = [];
        foreach ($elevators as $elevator) {
            // حساب إجمالي المدفوعات من الخزينة (مباشرة بواسطة elevator_id)
            $cashPaymentsDirect = CashTransaction::where('elevator_id', $elevator->id)
                ->where('type', 'deposit')
                ->sum('amount');

            // حساب إجمالي المدفوعات من الخزينة (بواسطة reference_type و reference_id)
            $cashPaymentsRef = CashTransaction::where('reference_type', 'Elevator')
                ->where('reference_id', $elevator->id)
                ->where('type', 'deposit')
                ->sum('amount');

            // إجمالي مدفوعات الخزينة
            $cashPayments = $cashPaymentsDirect + $cashPaymentsRef;

            // حساب إجمالي المدفوعات من البنوك
            $bankPayments = 0;
            try {
                // نفترض أن reference_number قد يحتوي على معرف المصعد
                $bankPayments = BankTransaction::where(function ($query) use ($elevator) {
                    $query->where('reference_number', 'LIKE', 'ELV-' . $elevator->id . '%')
                        ->orWhere('description', 'LIKE', '%مصعد%' . $elevator->serial_number . '%');
                })
                    ->where(function ($query) {
                        $query->where('transaction_type', 'deposit')
                            ->orWhere('transaction_type', 'credit');
                    })
                    ->sum('amount');
            } catch (\Exception $e) {
                // في حالة وجود خطأ، نتجاهل المدفوعات البنكية
                $bankPayments = 0;
            }

            // إجمالي المدفوعات
            $totalPayments = $cashPayments + $bankPayments;

            // قيمة المصعد
            $elevatorPrice = $elevator->price ?: 0;

            // حساب المبلغ المتبقي
            $remainingAmount = $elevatorPrice - $totalPayments;

            // الحصول على معلومات المشروع المرتبط (إن وجد)
            $projectInfo = DB::table('project_elevators')
                ->join('installation_projects', 'project_elevators.installation_project_id', '=', 'installation_projects.id')
                ->where('project_elevators.elevator_id', $elevator->id)
                ->select('installation_projects.id as project_id', 'installation_projects.title as project_title')
                ->first();

            $reportData[] = [
                'elevator_id' => $elevator->id,
                'serial_number' => $elevator->serial_number,
                'customer_name' => $elevator->customer->name ?? 'غير محدد',
                'elevator_price' => $elevatorPrice,
                'total_payments' => $totalPayments,
                'remaining_amount' => $remainingAmount,
                'project_id' => $projectInfo->project_id ?? null,
                'project_title' => $projectInfo->project_title ?? 'غير مرتبط بمشروع',
            ];
        }

        // الحصول على قائمة العملاء للفلتر
        $customers = \App\Models\Customer::orderBy('name')->get();

        // الحصول على قائمة المصاعد للفلتر
        $elevators = Elevator::orderBy('serial_number')->get();

        return view('reports.elevator_payments', compact('reportData', 'customers', 'elevators'));
    }
}
