<?php

namespace App\Http\Controllers;

use App\Models\ElevatorPart;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ElevatorPartController extends Controller
{
    /**
     * إنشاء مثيل جديد من التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * عرض قائمة قطع غيار المصاعد.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorParts = ElevatorPart::where('company_id', $company_id)
            ->orderBy('name')
            ->paginate(10);

        return view('elevator-parts.index', compact('elevatorParts'));
    }

    /**
     * عرض نموذج إنشاء قطعة غيار جديدة.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('elevator-parts.create');
    }

    /**
     * تخزين قطعة غيار جديدة في قاعدة البيانات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'part_number' => 'nullable|string|max:100',
            'manufacturer' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'specifications' => 'nullable|string',
            'cost_price' => 'nullable|numeric|min:0',
            'selling_price' => 'nullable|numeric|min:0',
            'min_stock_level' => 'nullable|integer|min:0',
            'current_stock' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'unit' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();

        $elevatorPart = new ElevatorPart();
        $elevatorPart->company_id = $user->company_id;
        $elevatorPart->name = $request->name;
        $elevatorPart->part_number = $request->part_number;
        $elevatorPart->manufacturer = $request->manufacturer;
        $elevatorPart->model = $request->model;
        $elevatorPart->description = $request->description;
        $elevatorPart->specifications = $request->specifications;
        $elevatorPart->cost_price = $request->cost_price;
        $elevatorPart->selling_price = $request->selling_price;
        $elevatorPart->min_stock_level = $request->min_stock_level ?? 0;
        $elevatorPart->current_stock = $request->current_stock ?? 0;
        $elevatorPart->location = $request->location;
        $elevatorPart->unit = $request->unit;
        $elevatorPart->is_active = $request->has('is_active');
        $elevatorPart->save();

        return redirect()->route('elevator-parts.index')
            ->with('success', 'تم إضافة قطعة الغيار بنجاح');
    }

    /**
     * عرض قطعة غيار محددة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorPart = ElevatorPart::where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator-parts.show', compact('elevatorPart'));
    }

    /**
     * عرض نموذج تعديل قطعة غيار.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorPart = ElevatorPart::where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator-parts.edit', compact('elevatorPart'));
    }

    /**
     * تحديث قطعة غيار محددة في قاعدة البيانات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'part_number' => 'nullable|string|max:100',
            'manufacturer' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'specifications' => 'nullable|string',
            'cost_price' => 'nullable|numeric|min:0',
            'selling_price' => 'nullable|numeric|min:0',
            'min_stock_level' => 'nullable|integer|min:0',
            'current_stock' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'unit' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorPart = ElevatorPart::where('company_id', $company_id)
            ->findOrFail($id);

        $elevatorPart->name = $request->name;
        $elevatorPart->part_number = $request->part_number;
        $elevatorPart->manufacturer = $request->manufacturer;
        $elevatorPart->model = $request->model;
        $elevatorPart->description = $request->description;
        $elevatorPart->specifications = $request->specifications;
        $elevatorPart->cost_price = $request->cost_price;
        $elevatorPart->selling_price = $request->selling_price;
        $elevatorPart->min_stock_level = $request->min_stock_level ?? 0;
        $elevatorPart->current_stock = $request->current_stock ?? 0;
        $elevatorPart->location = $request->location;
        $elevatorPart->unit = $request->unit;
        $elevatorPart->is_active = $request->has('is_active');
        $elevatorPart->save();

        return redirect()->route('elevator-parts.index')
            ->with('success', 'تم تحديث قطعة الغيار بنجاح');
    }

    /**
     * حذف قطعة غيار محددة من قاعدة البيانات.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorPart = ElevatorPart::where('company_id', $company_id)
            ->findOrFail($id);

        // يمكن إضافة تحقق هنا إذا كانت قطعة الغيار مستخدمة في أي مكان آخر

        $elevatorPart->delete();

        return redirect()->route('elevator-parts.index')
            ->with('success', 'تم حذف قطعة الغيار بنجاح');
    }

    /**
     * عرض قطع الغيار منخفضة المخزون.
     *
     * @return \Illuminate\Http\Response
     */
    public function lowStock()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorParts = ElevatorPart::where('company_id', $company_id)
            ->whereRaw('current_stock <= min_stock_level')
            ->where('is_active', true)
            ->orderBy('current_stock')
            ->paginate(10);

        return view('elevator-parts.low-stock', compact('elevatorParts'));
    }

    /**
     * تعديل مخزون قطعة غيار.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function editStock($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorPart = ElevatorPart::where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator-parts.edit-stock', compact('elevatorPart'));
    }

    /**
     * تحديث مخزون قطعة غيار.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateStock(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'current_stock' => 'required|integer|min:0',
            'min_stock_level' => 'required|integer|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorPart = ElevatorPart::where('company_id', $company_id)
            ->findOrFail($id);

        $elevatorPart->current_stock = $request->current_stock;
        $elevatorPart->min_stock_level = $request->min_stock_level;
        $elevatorPart->save();

        // يمكن إضافة سجل لتغيير المخزون هنا إذا لزم الأمر

        return redirect()->route('elevator-parts.show', $elevatorPart->id)
            ->with('success', 'تم تحديث مخزون قطعة الغيار بنجاح');
    }
}
