<?php

namespace App\Http\Controllers;

use App\Models\ElevatorModel;
use App\Models\ElevatorType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ElevatorModelController extends Controller
{
    /**
     * إنشاء مثيل جديد من التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * عرض قائمة موديلات المصاعد.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorModels = ElevatorModel::with('type')
            ->where('company_id', $company_id)
            ->orderBy('name')
            ->paginate(10);

        return view('elevator-models.index', compact('elevatorModels'));
    }

    /**
     * عرض نموذج إنشاء موديل مصعد جديد.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorTypes = ElevatorType::where('company_id', $company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('elevator-models.create', compact('elevatorTypes'));
    }

    /**
     * تخزين موديل مصعد جديد في قاعدة البيانات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'type_id' => 'nullable|exists:elevator_types,id',
            'code' => 'nullable|string|max:50',
            'manufacturer' => 'nullable|string|max:255',
            'max_weight' => 'nullable|integer|min:1',
            'max_persons' => 'nullable|integer|min:1',
            'speed' => 'nullable|integer|min:1',
            'specifications' => 'nullable|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();

        $elevatorModel = new ElevatorModel();
        $elevatorModel->company_id = $user->company_id;
        $elevatorModel->type_id = $request->type_id;
        $elevatorModel->name = $request->name;
        $elevatorModel->code = $request->code;
        $elevatorModel->manufacturer = $request->manufacturer;
        $elevatorModel->max_weight = $request->max_weight;
        $elevatorModel->max_persons = $request->max_persons;
        $elevatorModel->speed = $request->speed;
        $elevatorModel->specifications = $request->specifications;
        $elevatorModel->description = $request->description;
        $elevatorModel->is_active = $request->has('is_active');
        $elevatorModel->save();

        return redirect()->route('elevator-models.index')
            ->with('success', 'تم إضافة موديل المصعد بنجاح');
    }

    /**
     * عرض موديل مصعد محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorModel = ElevatorModel::with('type')
            ->where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator-models.show', compact('elevatorModel'));
    }

    /**
     * عرض نموذج تعديل موديل مصعد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorModel = ElevatorModel::where('company_id', $company_id)
            ->findOrFail($id);

        $elevatorTypes = ElevatorType::where('company_id', $company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->pluck('name', 'id');

        return view('elevator-models.edit', compact('elevatorModel', 'elevatorTypes'));
    }

    /**
     * تحديث موديل مصعد محدد في قاعدة البيانات.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'type_id' => 'nullable|exists:elevator_types,id',
            'code' => 'nullable|string|max:50',
            'manufacturer' => 'nullable|string|max:255',
            'max_weight' => 'nullable|integer|min:1',
            'max_persons' => 'nullable|integer|min:1',
            'speed' => 'nullable|integer|min:1',
            'specifications' => 'nullable|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorModel = ElevatorModel::where('company_id', $company_id)
            ->findOrFail($id);

        $elevatorModel->type_id = $request->type_id;
        $elevatorModel->name = $request->name;
        $elevatorModel->code = $request->code;
        $elevatorModel->manufacturer = $request->manufacturer;
        $elevatorModel->max_weight = $request->max_weight;
        $elevatorModel->max_persons = $request->max_persons;
        $elevatorModel->speed = $request->speed;
        $elevatorModel->specifications = $request->specifications;
        $elevatorModel->description = $request->description;
        $elevatorModel->is_active = $request->has('is_active');
        $elevatorModel->save();

        return redirect()->route('elevator-models.index')
            ->with('success', 'تم تحديث موديل المصعد بنجاح');
    }

    /**
     * حذف موديل مصعد محدد من قاعدة البيانات.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevatorModel = ElevatorModel::where('company_id', $company_id)
            ->findOrFail($id);

        // التحقق مما إذا كان هناك مصاعد مرتبطة بهذا الموديل
        if ($elevatorModel->elevators()->count() > 0) {
            return redirect()->route('elevator-models.index')
                ->with('error', 'لا يمكن حذف موديل المصعد لأنه مرتبط بمصاعد');
        }

        $elevatorModel->delete();

        return redirect()->route('elevator-models.index')
            ->with('success', 'تم حذف موديل المصعد بنجاح');
    }
}
