<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\ElevatorMaintenance;
use App\Models\ElevatorMaintenanceSchedule;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

/**
 * DESHABILITADO: Este controlador ha sido deshabilitado para evitar la duplicación de programaciones de mantenimiento.
 * Todas las programaciones de mantenimiento deben estar vinculadas a contratos de mantenimiento.
 * Por favor, utilice MaintenanceScheduleController para crear programaciones de mantenimiento.
 */
class ElevatorMaintenanceScheduleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // Redirigir a la página de programaciones de mantenimiento vinculadas a contratos
        return redirect()->route('maintenance-schedules.index')
            ->with('info', 'تم تعطيل جداول الصيانة الدورية. يرجى استخدام جداول الصيانة المرتبطة بالعقود فقط.');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // Redirigir a la página de creación de programaciones de mantenimiento vinculadas a contratos
        return redirect()->route('maintenance-schedules.create')
            ->with('info', 'تم تعطيل جداول الصيانة الدورية. يرجى إنشاء جداول الصيانة المرتبطة بالعقود فقط.');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Redirigir a la página de programaciones de mantenimiento vinculadas a contratos
        return redirect()->route('maintenance-schedules.index')
            ->with('info', 'تم تعطيل جداول الصيانة الدورية. يرجى إنشاء جداول الصيانة المرتبطة بالعقود فقط.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = Auth::user()->company_id;
        $schedule = ElevatorMaintenanceSchedule::where('company_id', $companyId)
            ->where('id', $id)
            ->with(['elevator', 'technician', 'creator', 'maintenanceOperations'])
            ->firstOrFail();

        return view('elevator-maintenance-schedules.show', compact('schedule'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = Auth::user()->company_id;
        $schedule = ElevatorMaintenanceSchedule::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $elevators = Elevator::where('company_id', $companyId)->get();
        $technicians = \App\Models\User::where('company_id', $companyId)
            ->whereHas('roles', function ($query) {
                $query->where('name', 'technician');
            })
            ->get();

        return view('elevator-maintenance-schedules.edit', compact('schedule', 'elevators', 'technicians'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            'maintenance_type' => 'required|string|max:255',
            'description' => 'required|string',
            'frequency' => 'required|in:daily,weekly,monthly,quarterly,semi_annual,annual',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'technician_id' => 'required|exists:users,id',
            'estimated_duration' => 'required|integer|min:1',
            'status' => 'required|in:active,inactive,completed',
            'notes' => 'nullable|string',
        ]);

        $companyId = Auth::user()->company_id;
        $schedule = ElevatorMaintenanceSchedule::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        // Calculate next run date based on frequency and start date
        $startDate = Carbon::parse($request->start_date);
        $nextRunDate = $this->calculateNextRunDate($startDate, $request->frequency, $request->day_of_week, $request->day_of_month, $request->month_of_year);

        $schedule->elevator_id = $request->elevator_id;
        $schedule->maintenance_type = $request->maintenance_type;
        $schedule->description = $request->description;
        $schedule->frequency = $request->frequency;
        $schedule->day_of_week = $request->day_of_week;
        $schedule->day_of_month = $request->day_of_month;
        $schedule->month_of_year = $request->month_of_year;
        $schedule->start_date = $request->start_date;
        $schedule->end_date = $request->end_date;
        $schedule->next_run_date = $nextRunDate;
        $schedule->technician_id = $request->technician_id;
        $schedule->estimated_duration = $request->estimated_duration;
        $schedule->status = $request->status;
        $schedule->notes = $request->notes;
        $schedule->save();

        return redirect()->route('elevator-maintenance-schedules.index')
            ->with('success', 'تم تحديث جدول الصيانة الدورية بنجاح.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = Auth::user()->company_id;
        $schedule = ElevatorMaintenanceSchedule::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        // Check if there are any maintenance operations linked to this schedule
        $hasMaintenanceOperations = ElevatorMaintenance::where('schedule_id', $id)->exists();

        if ($hasMaintenanceOperations) {
            return redirect()->back()->with('error', 'لا يمكن حذف جدول الصيانة لأنه مرتبط بعمليات صيانة.');
        }

        $schedule->delete();

        return redirect()->route('elevator-maintenance-schedules.index')
            ->with('success', 'تم حذف جدول الصيانة الدورية بنجاح.');
    }

    /**
     * Generate maintenance operations for all active schedules.
     *
     * @return \Illuminate\Http\Response
     */
    public function generateMaintenanceOperations()
    {
        $companyId = Auth::user()->company_id;
        $today = Carbon::today();

        // Get all active schedules that are due today
        $schedules = ElevatorMaintenanceSchedule::where('company_id', $companyId)
            ->where('status', 'active')
            ->where('next_run_date', '<=', $today)
            ->whereNull('end_date')
            ->orWhere(function ($query) use ($today) {
                $query->where('end_date', '>=', $today);
            })
            ->get();

        $count = 0;
        foreach ($schedules as $schedule) {
            // Create maintenance operation
            $this->createMaintenanceFromSchedule($schedule);

            // Update last run date and calculate next run date
            $schedule->last_run_date = $today;
            $schedule->next_run_date = $this->calculateNextRunDate($today, $schedule->frequency, $schedule->day_of_week, $schedule->day_of_month, $schedule->month_of_year);
            $schedule->save();

            $count++;
        }

        return redirect()->route('elevator-maintenance-schedules.index')
            ->with('success', "تم إنشاء $count عملية صيانة من الجداول الدورية.");
    }

    /**
     * Calculate the next run date based on frequency and start date.
     *
     * @param  \Carbon\Carbon  $startDate
     * @param  string  $frequency
     * @param  int|null  $dayOfWeek
     * @param  int|null  $dayOfMonth
     * @param  int|null  $monthOfYear
     * @return \Carbon\Carbon
     */
    private function calculateNextRunDate($startDate, $frequency, $dayOfWeek = null, $dayOfMonth = null, $monthOfYear = null)
    {
        $nextRunDate = $startDate->copy();

        switch ($frequency) {
            case 'daily':
                // Next day
                return $nextRunDate->addDay();
            case 'weekly':
                // If day of week is specified, find the next occurrence of that day
                if ($dayOfWeek !== null) {
                    return $nextRunDate->next((int) $dayOfWeek);
                }
                // Otherwise, add 7 days
                return $nextRunDate->addWeek();
            case 'monthly':
                // If day of month is specified, set to that day
                if ($dayOfMonth !== null) {
                    $nextRunDate->day(min((int) $dayOfMonth, $nextRunDate->daysInMonth));
                }
                return $nextRunDate->addMonth();
            case 'quarterly':
                return $nextRunDate->addMonths(3);
            case 'semi_annual':
                return $nextRunDate->addMonths(6);
            case 'annual':
                // If month and day are specified, set to that month and day
                if ($monthOfYear !== null && $dayOfMonth !== null) {
                    $nextRunDate->month((int) $monthOfYear)->day(min((int) $dayOfMonth, $nextRunDate->daysInMonth));
                }
                return $nextRunDate->addYear();
            default:
                return $nextRunDate;
        }
    }

    /**
     * Create a maintenance operation from a schedule.
     *
     * @param  \App\Models\ElevatorMaintenanceSchedule  $schedule
     * @return \App\Models\ElevatorMaintenance
     */
    private function createMaintenanceFromSchedule($schedule)
    {
        $maintenance = new ElevatorMaintenance();
        $maintenance->company_id = $schedule->company_id;
        $maintenance->elevator_id = $schedule->elevator_id;
        $maintenance->schedule_id = $schedule->id;
        $maintenance->maintenance_type = $schedule->maintenance_type;
        $maintenance->description = $schedule->description;
        $maintenance->scheduled_date = Carbon::now()->addDays(1)->startOfDay();
        $maintenance->technician_id = $schedule->technician_id;
        $maintenance->estimated_duration = $schedule->estimated_duration;
        $maintenance->priority = 'medium';
        $maintenance->status = 'scheduled';
        $maintenance->notes = "صيانة دورية مجدولة من جدول الصيانة #{$schedule->id}. " . $schedule->notes;
        $maintenance->created_by = $schedule->created_by;
        $maintenance->save();

        return $maintenance;
    }
}
