<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\ElevatorMaintenance;
use App\Models\ElevatorPart;
use App\Models\MaintenanceAttachment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ElevatorMaintenanceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = Auth::user()->company_id;
        $maintenances = ElevatorMaintenance::where('company_id', $companyId)
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-maintenance.index', compact('maintenances'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $companyId = Auth::user()->company_id;
        $elevators = Elevator::where('company_id', $companyId)
            ->with('location') // تحميل علاقة الموقع (المبنى)
            ->get();
        $technicians = \App\Models\User::where('company_id', $companyId)
            ->whereHas('roles', function ($query) {
                $query->where('slug', 'technician');
            })
            ->get();

        // Si no se encuentran técnicos, intentamos buscar con criterios más amplios
        if ($technicians->isEmpty()) {
            $technicians = \App\Models\User::where('company_id', $companyId)
                ->whereHas('roles', function ($query) {
                    $query->where('name', 'فني')
                        ->orWhere('slug', 'technician')
                        ->orWhere('name', 'like', '%فني%')
                        ->orWhere('name', 'like', '%technician%');
                })
                ->get();
        }

        // Si aún no hay técnicos, mostramos todos los usuarios como último recurso
        if ($technicians->isEmpty()) {
            $technicians = \App\Models\User::where('company_id', $companyId)->get();
        }

        // إضافة قطع الغيار للمصاعد
        $parts = \App\Models\ElevatorPart::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // تحديد نوع الصيانة الافتراضي بناءً على المعلمة المرسلة
        $maintenanceType = $request->query('type');

        // التحقق من صحة نوع الصيانة
        $validTypes = ['corrective', 'emergency', 'preventive', 'inspection', 'upgrade'];
        if (!in_array($maintenanceType, $validTypes)) {
            $maintenanceType = null;
        }

        return view('elevator-maintenance.create', compact('elevators', 'technicians', 'parts', 'maintenanceType'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            'maintenance_type' => 'required|string|in:corrective,emergency',
            'description' => 'required|string',
            'scheduled_date' => 'required|date',
            'priority' => 'required|in:medium,high,urgent',
            'reporter_name' => 'required|string|max:255',
            'reporter_phone' => 'required|string|max:20',
        ]);

        // تعيين الفني تلقائيًا إذا لم يتم تحديده
        if (!$request->has('technician_id') || empty($request->technician_id)) {
            $technician = \App\Models\User::where('company_id', Auth::user()->company_id)
                ->whereHas('roles', function ($query) {
                    $query->where('name', 'technician');
                })
                ->first();

            $technicianId = $technician ? $technician->id : Auth::id();
        } else {
            $technicianId = $request->technician_id;
        }

        // إنشاء بلاغ العطل
        $maintenance = new ElevatorMaintenance();
        $maintenance->company_id = Auth::user()->company_id;
        $maintenance->elevator_id = $request->elevator_id;
        $maintenance->maintenance_type = $request->maintenance_type;
        $maintenance->description = $request->description;
        $maintenance->scheduled_date = $request->scheduled_date;
        $maintenance->technician_id = $technicianId;
        $maintenance->estimated_duration = $request->estimated_duration ?? 1;
        $maintenance->priority = $request->priority;
        $maintenance->status = 'scheduled'; // Changed from 'pending' to 'scheduled' to match the enum values
        $maintenance->created_by = Auth::id();

        // حفظ معلومات المبلغ في حقل الملاحظات
        $maintenance->notes = "المبلغ: " . $request->reporter_name . "\n";
        $maintenance->notes .= "رقم الهاتف: " . $request->reporter_phone . "\n";

        if ($request->has('is_urgent') && $request->is_urgent) {
            $maintenance->notes .= "حالة طارئة: نعم\n";
        }

        $maintenance->save();

        // معالجة المرفقات إذا وجدت
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('maintenance_attachments', 'public');

                // يمكن إضافة كود لحفظ المرفقات في قاعدة البيانات هنا
                // على سبيل المثال:
                // $attachment = new \App\Models\Attachment();
                // $attachment->maintenance_id = $maintenance->id;
                // $attachment->file_path = $path;
                // $attachment->file_name = $file->getClientOriginalName();
                // $attachment->save();
            }
        }

        return redirect()->route('elevator-maintenance.index')
            ->with('success', 'تم تسجيل بلاغ العطل بنجاح.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->with(['elevator.location', 'technician', 'creator', 'parts', 'attachments'])
            ->firstOrFail();

        return view('elevator-maintenance.show', compact('maintenance'));
    }

    /**
     * Show the form for adding parts to a maintenance.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function showAddParts($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $parts = ElevatorPart::where('company_id', $companyId)
            ->where('is_active', true)
            ->where('current_stock', '>', 0)
            ->orderBy('name')
            ->get();

        return view('elevator-maintenance.add-parts', compact('maintenance', 'parts'));
    }

    /**
     * Add parts to a maintenance.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function addParts(Request $request, $id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $parts = $request->input('parts', []);

        if (empty($parts)) {
            return redirect()->back()->with('error', 'يرجى إضافة قطع غيار على الأقل');
        }

        // Sync parts with pivot data
        $syncData = [];
        foreach ($parts as $part) {
            if (!empty($part['part_id'])) {
                $syncData[$part['part_id']] = [
                    'quantity' => $part['quantity'] ?? 1
                ];
            }
        }

        $maintenance->parts()->sync($syncData);

        return redirect()->route('elevator-maintenance.show', $maintenance->id)
            ->with('success', 'تم إضافة قطع الغيار بنجاح');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $elevators = Elevator::where('company_id', $companyId)
            ->with('location') // تحميل علاقة الموقع (المبنى)
            ->get();
        $technicians = \App\Models\User::where('company_id', $companyId)
            ->whereHas('roles', function ($query) {
                $query->where('slug', 'technician');
            })
            ->get();

        // Si no se encuentran técnicos, intentamos buscar con criterios más amplios
        if ($technicians->isEmpty()) {
            $technicians = \App\Models\User::where('company_id', $companyId)
                ->whereHas('roles', function ($query) {
                    $query->where('name', 'فني')
                        ->orWhere('slug', 'technician')
                        ->orWhere('name', 'like', '%فني%')
                        ->orWhere('name', 'like', '%technician%');
                })
                ->get();
        }

        // Si aún no hay técnicos, mostramos todos los usuarios como último recurso
        if ($technicians->isEmpty()) {
            $technicians = \App\Models\User::where('company_id', $companyId)->get();
        }

        // إضافة قطع الغيار للمصاعد
        $parts = \App\Models\ElevatorPart::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('elevator-maintenance.edit', compact('maintenance', 'elevators', 'technicians', 'parts'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            'description' => 'required|string',
            'scheduled_date' => 'required|date',
            'technician_id' => 'required|exists:users,id',
            'estimated_duration' => 'nullable|integer|min:1',
            'priority' => 'required|in:low,medium,high,urgent',
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
        ]);

        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $maintenance->elevator_id = $request->elevator_id;
        $maintenance->maintenance_type = 'corrective'; // Always set to corrective
        $maintenance->description = $request->description;
        $maintenance->scheduled_date = $request->scheduled_date;
        $maintenance->technician_id = $request->technician_id;
        $maintenance->estimated_duration = $request->estimated_duration ?? $maintenance->estimated_duration ?? 1;
        $maintenance->priority = $request->priority;
        $maintenance->status = $request->status;

        if ($request->status == 'completed' && !$maintenance->completion_date) {
            $maintenance->completion_date = now();
            $maintenance->completed_by = Auth::id();
        }

        $maintenance->save();

        return redirect()->route('elevator-maintenance.index')
            ->with('success', 'تم تحديث عملية الصيانة بنجاح.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $maintenance->delete();

        return redirect()->route('elevator-maintenance.index')
            ->with('success', 'تم حذف عملية الصيانة بنجاح.');
    }

    /**
     * Display a listing of pending maintenance operations.
     *
     * @return \Illuminate\Http\Response
     */
    public function pending()
    {
        $companyId = Auth::user()->company_id;
        $maintenances = ElevatorMaintenance::where('company_id', $companyId)
            ->whereIn('status', ['scheduled', 'in_progress'])
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-maintenance.pending', compact('maintenances'));
    }

    /**
     * Display a listing of completed maintenance operations.
     *
     * @return \Illuminate\Http\Response
     */
    public function completed()
    {
        $companyId = Auth::user()->company_id;
        $maintenances = ElevatorMaintenance::where('company_id', $companyId)
            ->where('status', 'completed')
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-maintenance.completed', compact('maintenances'));
    }

    /**
     * Display a listing of scheduled maintenance operations.
     *
     * @return \Illuminate\Http\Response
     */
    public function scheduled()
    {
        $companyId = Auth::user()->company_id;
        $maintenances = ElevatorMaintenance::where('company_id', $companyId)
            ->where('status', 'scheduled')
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-maintenance.scheduled', compact('maintenances'));
    }

    /**
     * Mark a maintenance operation as complete.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsComplete($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $maintenance->status = 'completed';
        $maintenance->completion_date = now();
        $maintenance->completed_by = Auth::id();
        $maintenance->save();

        return redirect()->back()->with('success', 'تم تحديث حالة الصيانة إلى مكتملة بنجاح.');
    }

    /**
     * Mark a maintenance operation as cancelled.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsCancel($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $maintenance->status = 'cancelled';
        $maintenance->save();

        return redirect()->back()->with('success', 'تم إلغاء عملية الصيانة بنجاح.');
    }

    /**
     * Print maintenance details.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function print($id)
    {
        $companyId = Auth::user()->company_id;
        $maintenance = ElevatorMaintenance::where('company_id', $companyId)
            ->where('id', $id)
            ->with(['elevator.location', 'technician', 'creator', 'parts', 'attachments'])
            ->firstOrFail();

        return view('elevator-maintenance.print', compact('maintenance'));
    }

    /**
     * Display a listing of periodic maintenance operations that are not completed.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function periodic(Request $request)
    {
        $companyId = Auth::user()->company_id;

        // تسجيل معلومات عن بداية الاستعلام
        \Log::info('بدء استعلام الصيانات الدورية - عرض جداول الصيانة', [
            'company_id' => $companyId
        ]);

        // بدء الاستعلام الأساسي - استخدام جداول الصيانة بدلاً من سجلات الصيانة
        $query = \App\Models\MaintenanceSchedule::where('company_id', $companyId)
            ->with(['elevator.customer', 'elevator.location', 'maintenance_contract', 'user']);

        // إضافة فلاتر البحث إذا تم تحديدها
        $appliedFilters = ['نوع: جداول الصيانة'];

        // فلتر البحث النصي (اسم العميل / رقم المصعد / السنة)
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->whereHas('elevator', function ($q2) use ($searchTerm) {
                    $q2->where('serial_number', 'like', "%{$searchTerm}%")
                        ->orWhereHas('customer', function ($q3) use ($searchTerm) {
                            $q3->where('name', 'like', "%{$searchTerm}%");
                        });
                })
                    ->orWhere('scheduled_date', 'like', "%{$searchTerm}%");
            });
            $appliedFilters[] = "بحث: {$searchTerm}";
        }

        // فلتر المصعد
        if ($request->has('elevator_id') && !empty($request->elevator_id)) {
            $query->where('elevator_id', $request->elevator_id);
            $elevator = \App\Models\Elevator::find($request->elevator_id);
            if ($elevator) {
                $appliedFilters[] = "المصعد: {$elevator->serial_number}";
            }
        }

        // فلتر التاريخ (من)
        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->where('scheduled_date', '>=', $request->start_date);
            $appliedFilters[] = "من تاريخ: {$request->start_date}";
        }

        // فلتر التاريخ (إلى)
        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->where('scheduled_date', '<=', $request->end_date);
            $appliedFilters[] = "إلى تاريخ: {$request->end_date}";
        }

        // فلتر الحالة (عرض المكتملة والملغاة)
        if (!$request->has('show_completed') || !$request->show_completed) {
            $query->where('status', 'scheduled');
            $appliedFilters[] = "الحالة: مجدولة فقط";
        } else {
            $appliedFilters[] = "الحالة: جميع الحالات (بما في ذلك المكتملة والملغاة)";
        }

        // الحصول على إجمالي سجلات جداول الصيانة بعد تطبيق الفلاتر
        $totalAllRecords = $query->count();
        \Log::info('إجمالي سجلات جداول الصيانة بعد تطبيق الفلاتر', ['count' => $totalAllRecords]);

        // إضافة معلومات تشخيصية عن عدد السجلات حسب الحالة
        $statusCounts = \App\Models\MaintenanceSchedule::where('company_id', $companyId)
            ->selectRaw('status, count(*) as count')
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();
        \Log::info('توزيع سجلات جداول الصيانة حسب الحالة', $statusCounts);

        // التحقق من وجود جداول صيانة في قاعدة البيانات
        $allMaintenanceSchedules = \App\Models\MaintenanceSchedule::where('company_id', $companyId)
            ->get();
        $maintenanceStatuses = $allMaintenanceSchedules->pluck('status')->unique()->toArray();

        \Log::info('حالات جداول الصيانة الموجودة', ['statuses' => $maintenanceStatuses]);
        \Log::info('إجمالي عدد سجلات جداول الصيانة في النظام', ['count' => count($allMaintenanceSchedules)]);

        // معلومات إضافية عن أنواع الصيانة (للمقارنة)
        $allMaintenanceTypes = ElevatorMaintenance::where('company_id', $companyId)
            ->selectRaw('maintenance_type, count(*) as count')
            ->groupBy('maintenance_type')
            ->pluck('count', 'maintenance_type')
            ->toArray();
        \Log::info('توزيع جميع سجلات الصيانة حسب النوع', $allMaintenanceTypes);

        // تعيين متغير $typeCounts ليكون نفس $allMaintenanceTypes
        $typeCounts = $allMaintenanceTypes;

        // استخراج أنواع الصيانة من المفاتيح
        $maintenanceTypes = array_keys($allMaintenanceTypes);

        // تعريف متغير $allMaintenances ليكون نفس $allMaintenanceSchedules
        $allMaintenances = $allMaintenanceSchedules;

        // تعريف متغير $totalAllRecords ليكون عدد السجلات
        $totalAllRecords = count($allMaintenanceSchedules);

        // تنفيذ الاستعلام وترتيب النتائج تصاعديًا حسب تاريخ الجدولة
        $maintenances = $query->orderBy('scheduled_date', 'asc')->paginate(10);

        // تسجيل عدد النتائج النهائية
        \Log::info('نتائج الاستعلام النهائية', [
            'count' => $maintenances->count(),
            'total' => $maintenances->total(),
            'current_page' => $maintenances->currentPage(),
            'last_page' => $maintenances->lastPage()
        ]);

        // إضافة معلومات تشخيصية عن الاستعلام SQL
        $sql = $query->toSql();
        $bindings = $query->getBindings();

        // تحويل الباراميترات إلى نص
        $bindingsText = implode(', ', array_map(function ($binding) {
            return is_string($binding) ? "'$binding'" : $binding;
        }, $bindings));

        // إنشاء رسالة تشخيصية
        $diagnosticMessage = "<div class='alert alert-info'>";
        $diagnosticMessage .= "<strong>عرض جداول الصيانة</strong><br>";
        $diagnosticMessage .= "يتم الآن عرض جداول الصيانة من جدول maintenance_schedules.<br>";
        $diagnosticMessage .= "عدد جداول الصيانة: {$totalAllRecords}<br>";

        if ($totalAllRecords == 0) {
            $diagnosticMessage .= "<strong class='text-danger'>لا توجد أي جداول صيانة مسجلة في النظام حالياً.</strong>";
        }

        $diagnosticMessage .= "</div>";

        // إضافة معلومات حول الفلاتر المطبقة
        $diagnosticMessage .= "<br><strong>الفلاتر المطبقة:</strong> " . implode(" | ", $appliedFilters);

        // إضافة معلومات عن عدد السجلات حسب الحالة
        if (!empty($statusCounts)) {
            $diagnosticMessage .= "<br><strong>توزيع السجلات حسب الحالة:</strong> ";
            $statusLabels = [
                'scheduled' => 'مجدولة',
                'completed' => 'مكتملة',
                'missed' => 'فائتة',
                'cancelled' => 'ملغاة'
            ];

            $statusInfo = [];
            foreach ($statusCounts as $status => $count) {
                $label = $statusLabels[$status] ?? $status;
                $statusInfo[] = "{$label}: {$count}";
            }

            $diagnosticMessage .= implode(" | ", $statusInfo);
        }

        // إضافة معلومات عن عدد السجلات حسب نوع الصيانة
        if (!empty($typeCounts)) {
            $diagnosticMessage .= "<br><strong>توزيع سجلات الصيانة حسب النوع:</strong> ";
            $typeLabels = [
                'preventive' => 'دورية',
                'corrective' => 'تصحيحية',
                'inspection' => 'فحص',
                'installation' => 'تركيب',
                'emergency' => 'طارئة'
            ];

            $typeInfo = [];
            foreach ($typeCounts as $type => $count) {
                $label = $typeLabels[$type] ?? $type;
                $typeInfo[] = "{$label}: {$count}";
            }

            $diagnosticMessage .= implode(" | ", $typeInfo);
        }

        // إضافة معلومات تشخيصية للمطورين (تظهر فقط للمشرفين)
        if (Auth::user()->hasRole('admin')) {
            $diagnosticMessage .= "<br><div class='alert alert-secondary'>";
            $diagnosticMessage .= "<strong>معلومات تشخيصية للمطورين:</strong><br>";
            $diagnosticMessage .= "استعلام SQL: " . $sql . "<br>";
            $diagnosticMessage .= "المعاملات: " . $bindingsText . "<br>";
            $diagnosticMessage .= "أنواع الصيانة الموجودة: " . implode(", ", $maintenanceTypes) . "<br>";
            $diagnosticMessage .= "حالات جداول الصيانة الموجودة: " . implode(", ", $maintenanceStatuses) . "<br>";
            $diagnosticMessage .= "إجمالي عدد السجلات: " . count($allMaintenances);
            $diagnosticMessage .= "</div>";
        }

        // إضافة نصيحة واضحة للمستخدم
        $diagnosticMessage .= "<div class='alert alert-success'>";
        $diagnosticMessage .= "<h4><i class='fa fa-check-circle'></i> صفحة جداول الصيانة الدورية</h4>";
        $diagnosticMessage .= "<p>يتم الآن عرض <strong>جداول الصيانة</strong> من جدول maintenance_schedules.</p>";

        if ($totalAllRecords == 0) {
            $diagnosticMessage .= "<div class='alert alert-warning'>";
            $diagnosticMessage .= "<p><i class='fa fa-exclamation-triangle'></i> <strong>لا توجد أي جداول صيانة مسجلة في النظام حاليًا.</strong></p>";
            $diagnosticMessage .= "<p>يمكنك إضافة جدول صيانة جديد من <a href='" . route('maintenance-schedules.create') . "'>صفحة إنشاء جداول الصيانة</a>.</p>";
            $diagnosticMessage .= "</div>";
        }

        $diagnosticMessage .= "</div>";

        return view('elevator-maintenance.periodic', compact(
            'maintenances',
            'diagnosticMessage',
            'statusCounts',
            'typeCounts',
            'maintenanceTypes',
            'maintenanceStatuses'
        ));
    }
}
