<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\ElevatorInventoryRequest;
use App\Models\ElevatorInventoryRequestItem;
use App\Models\InventoryItem;
use App\Models\InventoryTransaction;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ElevatorInventoryRequestController extends Controller
{
    /**
     * عرض قائمة طلبات المخزون للمصاعد
     */
    public function index()
    {
        // عرض طلبات المخزون الخاصة بالشركة الحالية فقط
        $requests = ElevatorInventoryRequest::with(['requester', 'approver'])
            ->where('company_id', Auth::user()->company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('elevator_inventory.index', compact('requests'));
    }

    /**
     * عرض نموذج إنشاء طلب مخزون جديد للمصاعد
     */
    public function create()
    {
        // الحصول على الشركة الحالية للمستخدم
        $companies = Company::where('id', Auth::user()->company_id)->get();

        // الحصول على المستودعات التابعة للشركة الحالية
        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)->get();

        // الحصول على عناصر المخزون للشركة الحالية
        $inventoryItems = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        return view('elevator_inventory.create', compact('companies', 'warehouses', 'inventoryItems'));
    }

    /**
     * تخزين طلب مخزون جديد للمصاعد
     */
    public function store(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'company_id' => 'required|exists:companies,id',
            'required_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.warehouse_id' => 'required|exists:warehouses,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.priority' => 'required|in:low,medium,high,urgent',
            'items.*.notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // بدء معاملة قاعدة البيانات
        DB::beginTransaction();

        try {
            // إنشاء رقم طلب فريد
            $requestNumber = 'EIR-' . date('Ymd') . '-' . rand(1000, 9999);

            // إنشاء طلب المخزون الجديد
            $elevatorInventoryRequest = ElevatorInventoryRequest::create([
                'company_id' => $request->company_id,
                'request_number' => $requestNumber,
                'status' => 'pending',
                'requested_by' => Auth::id(),
                'requested_date' => now(),
                'required_date' => $request->required_date,
                'notes' => $request->notes,
            ]);

            // إضافة عناصر الطلب
            foreach ($request->items as $item) {
                ElevatorInventoryRequestItem::create([
                    'elevator_inventory_request_id' => $elevatorInventoryRequest->id,
                    'inventory_item_id' => $item['inventory_item_id'],
                    'warehouse_id' => $item['warehouse_id'],
                    'quantity' => $item['quantity'],
                    'fulfilled_quantity' => 0,
                    'status' => 'pending',
                    'priority' => $item['priority'],
                    'notes' => $item['notes'] ?? null,
                ]);
            }

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('elevator-inventory.index')
                ->with('success', 'تم إنشاء طلب المخزون للمصاعد بنجاح.');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء الطلب: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل طلب مخزون محدد
     */
    public function show(ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        $elevatorInventoryRequest->load(['requester', 'approver', 'items.item', 'items.warehouse']);

        return view('elevator_inventory.show', compact('elevatorInventoryRequest'));
    }

    /**
     * عرض نموذج تعديل طلب مخزون
     */
    public function edit(ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        // التحقق من أن الطلب لم يتم اعتماده بعد
        if (!in_array($elevatorInventoryRequest->status, ['draft', 'pending', 'rejected'])) {
            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('error', 'لا يمكن تعديل الطلب بعد اعتماده أو تنفيذه');
        }

        $companies = Company::where('id', Auth::user()->company_id)->get();
        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)->get();
        $inventoryItems = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        $elevatorInventoryRequest->load(['items.item', 'items.warehouse']);

        return view('elevator_inventory.edit', compact('elevatorInventoryRequest', 'companies', 'warehouses', 'inventoryItems'));
    }

    /**
     * تحديث طلب مخزون محدد
     */
    public function update(Request $request, ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        // التحقق من أن الطلب لم يتم اعتماده بعد
        if (!in_array($elevatorInventoryRequest->status, ['draft', 'pending', 'rejected'])) {
            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('error', 'لا يمكن تعديل الطلب بعد اعتماده أو تنفيذه');
        }

        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'required_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:elevator_inventory_request_items,id',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.warehouse_id' => 'required|exists:warehouses,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.priority' => 'required|in:low,medium,high,urgent',
            'items.*.notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // بدء معاملة قاعدة البيانات
        DB::beginTransaction();

        try {
            // تحديث بيانات الطلب
            $elevatorInventoryRequest->update([
                'required_date' => $request->required_date,
                'notes' => $request->notes,
                'status' => 'pending', // إعادة تعيين الحالة إلى معلق بعد التعديل
            ]);

            // الحصول على معرفات العناصر الحالية
            $currentItemIds = $elevatorInventoryRequest->items->pluck('id')->toArray();
            $updatedItemIds = [];

            // تحديث أو إنشاء عناصر الطلب
            foreach ($request->items as $item) {
                if (isset($item['id']) && in_array($item['id'], $currentItemIds)) {
                    // تحديث العنصر الموجود
                    $requestItem = ElevatorInventoryRequestItem::find($item['id']);
                    $requestItem->update([
                        'inventory_item_id' => $item['inventory_item_id'],
                        'warehouse_id' => $item['warehouse_id'],
                        'quantity' => $item['quantity'],
                        'status' => 'pending', // إعادة تعيين الحالة إلى معلق بعد التعديل
                        'priority' => $item['priority'],
                        'notes' => $item['notes'] ?? null,
                    ]);
                    $updatedItemIds[] = $item['id'];
                } else {
                    // إنشاء عنصر جديد
                    $requestItem = ElevatorInventoryRequestItem::create([
                        'elevator_inventory_request_id' => $elevatorInventoryRequest->id,
                        'inventory_item_id' => $item['inventory_item_id'],
                        'warehouse_id' => $item['warehouse_id'],
                        'quantity' => $item['quantity'],
                        'fulfilled_quantity' => 0,
                        'status' => 'pending',
                        'priority' => $item['priority'],
                        'notes' => $item['notes'] ?? null,
                    ]);
                    $updatedItemIds[] = $requestItem->id;
                }
            }

            // حذف العناصر التي لم يتم تحديثها
            $itemsToDelete = array_diff($currentItemIds, $updatedItemIds);
            if (!empty($itemsToDelete)) {
                ElevatorInventoryRequestItem::whereIn('id', $itemsToDelete)->delete();
            }

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('success', 'تم تحديث طلب المخزون للمصاعد بنجاح.');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث الطلب: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف طلب مخزون محدد
     */
    public function destroy(ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        // التحقق من أن الطلب لم يتم تنفيذه بعد
        if (in_array($elevatorInventoryRequest->status, ['fulfilled', 'partially_fulfilled'])) {
            return redirect()->route('elevator-inventory.index')
                ->with('error', 'لا يمكن حذف الطلب بعد تنفيذه');
        }

        // حذف الطلب وعناصره
        $elevatorInventoryRequest->delete();

        return redirect()->route('elevator-inventory.index')
            ->with('success', 'تم حذف طلب المخزون للمصاعد بنجاح.');
    }

    /**
     * اعتماد طلب مخزون محدد
     */
    public function approve(ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        // التحقق من أن الطلب في حالة معلق
        if ($elevatorInventoryRequest->status !== 'pending') {
            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('error', 'لا يمكن اعتماد الطلب في حالته الحالية');
        }

        // تحديث حالة الطلب
        $elevatorInventoryRequest->update([
            'status' => 'approved',
            'approved_by' => Auth::id(),
            'approval_date' => now(),
        ]);

        // تحديث حالة عناصر الطلب
        foreach ($elevatorInventoryRequest->items as $item) {
            $item->update(['status' => 'approved']);
        }

        return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
            ->with('success', 'تم اعتماد طلب المخزون للمصاعد بنجاح.');
    }

    /**
     * رفض طلب مخزون محدد
     */
    public function reject(Request $request, ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        // التحقق من أن الطلب في حالة معلق
        if ($elevatorInventoryRequest->status !== 'pending') {
            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('error', 'لا يمكن رفض الطلب في حالته الحالية');
        }

        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'rejection_reason' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // تحديث حالة الطلب
        $elevatorInventoryRequest->update([
            'status' => 'rejected',
            'rejection_reason' => $request->rejection_reason,
        ]);

        // تحديث حالة عناصر الطلب
        foreach ($elevatorInventoryRequest->items as $item) {
            $item->update(['status' => 'rejected']);
        }

        return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
            ->with('success', 'تم رفض طلب المخزون للمصاعد بنجاح.');
    }

    /**
     * تنفيذ طلب مخزون محدد
     */
    public function fulfill(Request $request, ElevatorInventoryRequest $elevatorInventoryRequest)
    {
        // التحقق من أن الطلب ينتمي للشركة الحالية
        if ($elevatorInventoryRequest->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح بالوصول إلى هذا الطلب');
        }

        // التحقق من أن الطلب في حالة معتمد
        if ($elevatorInventoryRequest->status !== 'approved') {
            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('error', 'لا يمكن تنفيذ الطلب في حالته الحالية');
        }

        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'items' => 'required|array',
            'items.*.id' => 'required|exists:elevator_inventory_request_items,id',
            'items.*.fulfilled_quantity' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // بدء معاملة قاعدة البيانات
        DB::beginTransaction();

        try {
            $allFulfilled = true;
            $anyFulfilled = false;

            // تحديث كميات التنفيذ وإنشاء معاملات المخزون
            foreach ($request->items as $itemData) {
                $item = ElevatorInventoryRequestItem::find($itemData['id']);

                // التحقق من أن العنصر ينتمي للطلب
                if ($item->elevator_inventory_request_id != $elevatorInventoryRequest->id) {
                    throw new \Exception('بيانات غير صالحة');
                }

                $fulfilledQuantity = (float) $itemData['fulfilled_quantity'];

                // تحديث الكمية المنفذة
                $item->fulfilled_quantity = $fulfilledQuantity;

                // تحديد حالة العنصر
                if ($fulfilledQuantity <= 0) {
                    $item->status = 'rejected';
                    $allFulfilled = false;
                } elseif ($fulfilledQuantity < $item->quantity) {
                    $item->status = 'partially_fulfilled';
                    $allFulfilled = false;
                    $anyFulfilled = true;
                } else {
                    $item->status = 'fulfilled';
                    $anyFulfilled = true;
                }

                $item->save();

                // إنشاء معاملة مخزون إذا تم تنفيذ كمية
                if ($fulfilledQuantity > 0) {
                    InventoryTransaction::create([
                        'company_id' => $elevatorInventoryRequest->company_id,
                        'transaction_type' => 'issue',
                        'item_id' => $item->inventory_item_id,
                        'warehouse_id' => $item->warehouse_id,
                        'quantity' => $fulfilledQuantity,
                        'reference_number' => $elevatorInventoryRequest->request_number,
                        'reference_type' => 'ElevatorInventoryRequest',
                        'reference_id' => $elevatorInventoryRequest->id,
                        'created_by' => Auth::id(),
                        'notes' => 'صرف بضاعة للمصاعد - ' . $elevatorInventoryRequest->request_number,
                    ]);
                }
            }

            // تحديث حالة الطلب
            if ($allFulfilled) {
                $status = 'fulfilled';
            } elseif ($anyFulfilled) {
                $status = 'partially_fulfilled';
            } else {
                $status = 'rejected';
            }

            $elevatorInventoryRequest->update([
                'status' => $status,
                'fulfillment_date' => now(),
            ]);

            // تأكيد المعاملة
            DB::commit();

            return redirect()->route('elevator-inventory.show', $elevatorInventoryRequest)
                ->with('success', 'تم تنفيذ طلب المخزون للمصاعد بنجاح.');
        } catch (\Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تنفيذ الطلب: ' . $e->getMessage())->withInput();
        }
    }
}
