<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\ElevatorInspection;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ElevatorInspectionController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = Auth::user()->company_id;
        $inspections = ElevatorInspection::where('company_id', $companyId)
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-inspections.index', compact('inspections'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companyId = Auth::user()->company_id;
        $elevators = Elevator::where('company_id', $companyId)->get();
        $technicians = User::where('company_id', $companyId)
            ->whereHas('roles', function ($query) {
                $query->where('name', 'technician');
            })
            ->get();

        return view('elevator-inspections.create', compact('elevators', 'technicians'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            'technician_id' => 'required|exists:users,id',
            'inspection_date' => 'required|date',
            'inspection_type' => 'required|in:routine,safety,certification,complaint,other',
            'notes' => 'nullable|string',
        ]);

        $companyId = Auth::user()->company_id;

        $inspection = new ElevatorInspection();
        $inspection->company_id = $companyId;
        $inspection->elevator_id = $request->elevator_id;
        $inspection->technician_id = $request->technician_id;
        $inspection->inspection_date = $request->inspection_date;
        $inspection->inspection_type = $request->inspection_type;
        $inspection->notes = $request->notes;
        $inspection->status = 'scheduled';
        $inspection->save();

        return redirect()->route('elevator-inspections.index')
            ->with('success', 'تم إضافة عملية الفحص بنجاح.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = Auth::user()->company_id;
        $inspection = ElevatorInspection::where('company_id', $companyId)
            ->where('id', $id)
            ->with(['elevator', 'technician'])
            ->firstOrFail();

        return view('elevator-inspections.show', compact('inspection'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = Auth::user()->company_id;
        $inspection = ElevatorInspection::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $elevators = Elevator::where('company_id', $companyId)->get();
        $technicians = User::where('company_id', $companyId)
            ->whereHas('roles', function ($query) {
                $query->where('name', 'technician');
            })
            ->get();

        return view('elevator-inspections.edit', compact('inspection', 'elevators', 'technicians'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            'technician_id' => 'required|exists:users,id',
            'inspection_date' => 'required|date',
            'inspection_type' => 'required|in:routine,safety,certification,complaint,other',
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
            'notes' => 'nullable|string',
            'findings' => 'nullable|string',
            'recommendations' => 'nullable|string',
            'passed' => 'nullable|boolean',
            'certificate_number' => 'nullable|string',
            'certificate_expiry_date' => 'nullable|date',
        ]);

        $companyId = Auth::user()->company_id;
        $inspection = ElevatorInspection::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $inspection->elevator_id = $request->elevator_id;
        $inspection->technician_id = $request->technician_id;
        $inspection->inspection_date = $request->inspection_date;
        $inspection->inspection_type = $request->inspection_type;
        $inspection->status = $request->status;
        $inspection->notes = $request->notes;
        $inspection->findings = $request->findings;
        $inspection->recommendations = $request->recommendations;
        $inspection->passed = $request->passed;
        $inspection->certificate_number = $request->certificate_number;
        $inspection->certificate_expiry_date = $request->certificate_expiry_date;
        $inspection->save();

        return redirect()->route('elevator-inspections.index')
            ->with('success', 'تم تحديث عملية الفحص بنجاح.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = Auth::user()->company_id;
        $inspection = ElevatorInspection::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $inspection->delete();

        return redirect()->route('elevator-inspections.index')
            ->with('success', 'تم حذف عملية الفحص بنجاح.');
    }

    /**
     * Display a listing of pending inspections.
     *
     * @return \Illuminate\Http\Response
     */
    public function pending()
    {
        $companyId = Auth::user()->company_id;
        $inspections = ElevatorInspection::where('company_id', $companyId)
            ->whereIn('status', ['scheduled', 'in_progress'])
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-inspections.pending', compact('inspections'));
    }

    /**
     * Display a listing of completed inspections.
     *
     * @return \Illuminate\Http\Response
     */
    public function completed()
    {
        $companyId = Auth::user()->company_id;
        $inspections = ElevatorInspection::where('company_id', $companyId)
            ->where('status', 'completed')
            ->with(['elevator', 'technician'])
            ->latest()
            ->paginate(10);

        return view('elevator-inspections.completed', compact('inspections'));
    }

    /**
     * Mark an inspection as complete.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsComplete(Request $request, $id)
    {
        $request->validate([
            'findings' => 'required|string',
            'recommendations' => 'required|string',
            'passed' => 'required|boolean',
        ]);

        $companyId = Auth::user()->company_id;
        $inspection = ElevatorInspection::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $inspection->status = 'completed';
        $inspection->findings = $request->findings;
        $inspection->recommendations = $request->recommendations;
        $inspection->passed = $request->passed;
        $inspection->certificate_number = $request->certificate_number;
        $inspection->certificate_expiry_date = $request->certificate_expiry_date;
        $inspection->save();

        return redirect()->route('elevator-inspections.index')
            ->with('success', 'تم إكمال عملية الفحص بنجاح.');
    }

    /**
     * Mark an inspection as cancelled.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsCancel(Request $request, $id)
    {
        $request->validate([
            'notes' => 'required|string',
        ]);

        $companyId = Auth::user()->company_id;
        $inspection = ElevatorInspection::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $inspection->status = 'cancelled';
        $inspection->notes = $request->notes;
        $inspection->save();

        return redirect()->route('elevator-inspections.index')
            ->with('success', 'تم إلغاء عملية الفحص بنجاح.');
    }
}
