<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\ElevatorDelivery;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ElevatorDeliveryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $deliveries = ElevatorDelivery::with(['elevator', 'elevator.customer'])
            ->where('company_id', $company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('elevator_deliveries.index', compact('deliveries'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $elevators = Elevator::where('company_id', $company_id)
            ->orderBy('serial_number')
            ->get();

        return view('elevator_deliveries.create', compact('elevators'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'elevator_id' => 'required|exists:elevators,id',
            'delivery_date' => 'required|date',
            'warranty_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'free_maintenance_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'control_warranty_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'machine_warranty_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        // Crear el registro de entrega
        $delivery = new ElevatorDelivery();
        $delivery->company_id = $company_id;
        $delivery->elevator_id = $request->elevator_id;
        $delivery->user_id = $user->id;
        $delivery->delivery_date = $request->delivery_date;
        $delivery->warranty_end_date = $request->warranty_end_date;
        $delivery->free_maintenance_end_date = $request->free_maintenance_end_date;
        $delivery->control_warranty_end_date = $request->control_warranty_end_date;
        $delivery->machine_warranty_end_date = $request->machine_warranty_end_date;
        $delivery->notes = $request->notes;
        $delivery->save();

        // Actualizar el elevador con las fechas de la entrega
        $elevator = Elevator::findOrFail($request->elevator_id);
        $elevator->delivery_date = $request->delivery_date;
        $elevator->warranty_end_date = $request->warranty_end_date;
        $elevator->free_maintenance_end_date = $request->free_maintenance_end_date;
        $elevator->control_warranty_end_date = $request->control_warranty_end_date;
        $elevator->machine_warranty_end_date = $request->machine_warranty_end_date;
        $elevator->save();

        return redirect()->route('elevator-deliveries.index')
            ->with('success', 'تم تسجيل محضر تسليم المصعد بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $delivery = ElevatorDelivery::with(['elevator', 'elevator.customer'])
            ->where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator_deliveries.show', compact('delivery'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $delivery = ElevatorDelivery::where('company_id', $company_id)->findOrFail($id);
        $elevators = Elevator::where('company_id', $company_id)
            ->orderBy('serial_number')
            ->get();

        return view('elevator_deliveries.edit', compact('delivery', 'elevators'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'elevator_id' => 'required|exists:elevators,id',
            'delivery_date' => 'required|date',
            'warranty_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'free_maintenance_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'control_warranty_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'machine_warranty_end_date' => 'nullable|date|after_or_equal:delivery_date',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        // Actualizar el registro de entrega
        $delivery = ElevatorDelivery::where('company_id', $company_id)->findOrFail($id);
        $delivery->elevator_id = $request->elevator_id;
        $delivery->delivery_date = $request->delivery_date;
        $delivery->warranty_end_date = $request->warranty_end_date;
        $delivery->free_maintenance_end_date = $request->free_maintenance_end_date;
        $delivery->control_warranty_end_date = $request->control_warranty_end_date;
        $delivery->machine_warranty_end_date = $request->machine_warranty_end_date;
        $delivery->notes = $request->notes;
        $delivery->save();

        // Actualizar el elevador con las fechas de la entrega
        $elevator = Elevator::findOrFail($request->elevator_id);
        $elevator->delivery_date = $request->delivery_date;
        $elevator->warranty_end_date = $request->warranty_end_date;
        $elevator->free_maintenance_end_date = $request->free_maintenance_end_date;
        $elevator->control_warranty_end_date = $request->control_warranty_end_date;
        $elevator->machine_warranty_end_date = $request->machine_warranty_end_date;
        $elevator->save();

        return redirect()->route('elevator-deliveries.index')
            ->with('success', 'تم تحديث محضر تسليم المصعد بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $delivery = ElevatorDelivery::where('company_id', $company_id)->findOrFail($id);
        $delivery->delete();

        return redirect()->route('elevator-deliveries.index')
            ->with('success', 'تم حذف محضر تسليم المصعد بنجاح');
    }

    /**
     * Print the delivery report.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function print($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $delivery = ElevatorDelivery::with(['elevator', 'elevator.customer', 'company'])
            ->where('company_id', $company_id)
            ->findOrFail($id);

        return view('elevator_deliveries.print', compact('delivery'));
    }

    /**
     * Get elevator details via AJAX.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getElevatorDetails(Request $request)
    {
        $elevator_id = $request->input('elevator_id');
        $elevator = Elevator::with('customer')->findOrFail($elevator_id);

        return response()->json([
            'customer_name' => $elevator->customer->name,
            'model' => $elevator->model
        ]);
    }
}
