<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Company;
use App\Models\Branch;
use App\Models\Customer;
use App\Models\Elevator;
use App\Models\MaintenanceContract;
use App\Models\MaintenanceSchedule;
use App\Models\Invoice;
use App\Models\Notification;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        // If user doesn't have a company yet, show first-time setup page
        if (!$company) {
            return view('companies.first-time-setup');
        }

        // Get today and this week dates
        $today = Carbon::today();
        $tomorrow = Carbon::today()->addDay();
        $urgentDate = Carbon::today()->addDays(2);
        $weekEnd = Carbon::today()->addDays(7);

        // Get counts for dashboard
        $data = [
            // Basic counts
            'branches_count' => Branch::where('company_id', $company->id)->count(),
            'customers_count' => Customer::where('company_id', $company->id)->count(),
            'elevators_count' => Elevator::where('company_id', $company->id)->count(),
            'contracts_count' => MaintenanceContract::where('company_id', $company->id)->count(),
            'active_contracts_count' => MaintenanceContract::where('company_id', $company->id)
                ->where('status', 'active')
                ->count(),
            'invoices_count' => Invoice::where('company_id', $company->id)->count(),
            'unpaid_invoices_count' => Invoice::where('company_id', $company->id)
                ->whereIn('status', ['draft', 'sent', 'partially_paid', 'overdue'])
                ->count(),
            'total_invoices_amount' => Invoice::where('company_id', $company->id)->sum('total_amount'),
            'total_due_amount' => Invoice::where('company_id', $company->id)->sum('due_amount'),

            // Maintenance counts
            'maintenance_due_today' => MaintenanceSchedule::where('company_id', $company->id)
                ->where('status', 'scheduled')
                ->whereDate('scheduled_date', $today)
                ->count(),
            'maintenance_urgent' => MaintenanceSchedule::where('company_id', $company->id)
                ->where('status', 'scheduled')
                ->whereBetween('scheduled_date', [$tomorrow, $urgentDate])
                ->count(),
            'maintenance_due_this_week' => MaintenanceSchedule::where('company_id', $company->id)
                ->where('status', 'scheduled')
                ->whereBetween('scheduled_date', [$tomorrow, $weekEnd])
                ->count(),
            'maintenance_overdue' => MaintenanceSchedule::where('company_id', $company->id)
                ->where('status', 'scheduled')
                ->where('scheduled_date', '<', $today)
                ->count(),
            'maintenance_critical_overdue' => MaintenanceSchedule::where('company_id', $company->id)
                ->where('status', 'scheduled')
                ->where('scheduled_date', '<', $today->copy()->subDays(7))
                ->count(),
            'contracts_expiring_soon' => MaintenanceContract::where('company_id', $company->id)
                ->where('status', 'active')
                ->whereBetween('end_date', [$today, $today->copy()->addDays(30)])
                ->count(),

            // Recent notifications
            'recent_notifications' => Notification::where('company_id', $company->id)
                ->where(function ($query) use ($user) {
                    $query->where('user_id', $user->id)
                        ->orWhereNull('user_id');
                })
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get(),

            // Urgent notifications
            'urgent_notifications' => Notification::where('company_id', $company->id)
                ->where(function ($query) use ($user) {
                    $query->where('user_id', $user->id)
                        ->orWhereNull('user_id');
                })
                ->whereIn('priority', ['high', 'critical'])
                ->whereIn('type', [
                    'maintenance_urgent',
                    'maintenance_today',
                    'maintenance_overdue',
                    'maintenance_critical_overdue'
                ])
                ->whereNull('read_at')
                ->orderBy('priority', 'desc')
                ->orderBy('created_at', 'desc')
                ->limit(3)
                ->get(),
        ];

        return view('dashboard', compact('data', 'company'));
    }
}
