<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class CustomerSurveyController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $surveys = \App\Models\CustomerSurvey::where('company_id', auth()->user()->company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('customer_surveys.index', compact('surveys'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('customer_surveys.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'required|in:draft,active,completed',
            'notes' => 'nullable|string',
            'questions' => 'required|array|min:1',
        ]);

        // تحويل الأسئلة إلى JSON
        $questions = json_encode($request->questions);

        $survey = new \App\Models\CustomerSurvey();
        $survey->company_id = auth()->user()->company_id;
        $survey->user_id = auth()->id();
        $survey->title = $validated['title'];
        $survey->description = $validated['description'];
        $survey->start_date = $validated['start_date'];
        $survey->end_date = $validated['end_date'];
        $survey->status = $validated['status'];
        $survey->questions = $questions;
        $survey->total_responses = 0;
        $survey->average_rating = 0;
        $survey->notes = $validated['notes'];
        $survey->save();

        return redirect()->route('customer-surveys.index')
            ->with('success', 'تم إنشاء الاستبيان بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $survey = \App\Models\CustomerSurvey::with(['survey_responses.customer'])
            ->findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($survey->company_id != auth()->user()->company_id) {
            abort(403, 'غير مصرح لك بعرض هذا الاستبيان');
        }

        return view('customer_surveys.show', compact('survey'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $survey = \App\Models\CustomerSurvey::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($survey->company_id != auth()->user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذا الاستبيان');
        }

        return view('customer_surveys.edit', compact('survey'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $survey = \App\Models\CustomerSurvey::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($survey->company_id != auth()->user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذا الاستبيان');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'required|in:draft,active,completed',
            'notes' => 'nullable|string',
            'questions' => 'required|array|min:1',
        ]);

        // تحويل الأسئلة إلى JSON
        $questions = json_encode($request->questions);

        $survey->title = $validated['title'];
        $survey->description = $validated['description'];
        $survey->start_date = $validated['start_date'];
        $survey->end_date = $validated['end_date'];
        $survey->status = $validated['status'];
        $survey->questions = $questions;
        $survey->notes = $validated['notes'];
        $survey->save();

        return redirect()->route('customer-surveys.index')
            ->with('success', 'تم تحديث الاستبيان بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $survey = \App\Models\CustomerSurvey::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($survey->company_id != auth()->user()->company_id) {
            abort(403, 'غير مصرح لك بحذف هذا الاستبيان');
        }

        // التحقق من عدم وجود ردود على الاستبيان
        if ($survey->survey_responses()->count() > 0) {
            return redirect()->route('customer-surveys.index')
                ->with('error', 'لا يمكن حذف الاستبيان لأنه يحتوي على ردود');
        }

        $survey->delete();

        return redirect()->route('customer-surveys.index')
            ->with('success', 'تم حذف الاستبيان بنجاح');
    }
}
