<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CustomerStatementController extends Controller
{
    protected $invoiceService;

    /**
     * Create a new controller instance.
     *
     * @param  InvoiceService  $invoiceService
     * @return void
     */
    public function __construct(InvoiceService $invoiceService)
    {
        $this->invoiceService = $invoiceService;
        $this->middleware('auth');

        // تعديل الصلاحية للسماح بالوصول للمدير العام والمدير والمحاسب ومندوب المبيعات
        $this->middleware(function ($request, $next) {
            if (
                auth()->user()->hasRole('super_admin') ||
                auth()->user()->hasRole('admin') ||
                auth()->user()->hasRole('accountant') ||
                auth()->user()->hasRole('sales')
            ) {
                return $next($request);
            }

            if (auth()->user()->hasPermissionTo('customer_statements.view')) {
                return $next($request);
            }

            abort(403, 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        });
    }

    /**
     * عرض صفحة كشوف حساب العملاء.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = auth()->user()->company_id;

        // استرجاع العملاء مع حساب عدد الفواتير والمبالغ
        $customers = Customer::where('company_id', $companyId)
            ->withCount('invoices')
            ->withSum('invoices', 'total_amount')
            ->withSum('invoices', 'paid_amount')
            ->withSum('invoices', 'due_amount')
            ->orderBy('name')
            ->get();

        // حساب المبالغ لكل عميل إذا كانت القيم صفرية
        foreach ($customers as $customer) {
            if ($customer->invoices_sum_total_amount == 0 || $customer->invoices_sum_paid_amount == 0) {
                // استرجاع الفواتير يدوياً إذا كانت القيم صفرية
                $invoices = $customer->invoices;
                $customer->total_amount = $invoices->sum('total_amount');
                $customer->paid_amount = $invoices->sum('paid_amount');
                $customer->due_amount = $invoices->sum('due_amount');
            } else {
                // استخدام القيم المحسوبة مسبقاً
                $customer->total_amount = $customer->invoices_sum_total_amount;
                $customer->paid_amount = $customer->invoices_sum_paid_amount;
                $customer->due_amount = $customer->invoices_sum_due_amount;
            }
        }

        return view('customer_statements.index', compact('customers'));
    }

    /**
     * عرض نموذج إنشاء كشف حساب.
     *
     * @return \Illuminate\Http\Response
     */
    public function generateForm()
    {
        $companyId = auth()->user()->company_id;
        $customers = Customer::where('company_id', $companyId)
            ->orderBy('name')
            ->get();

        return view('customer_statements.generate_form', compact('customers'));
    }

    /**
     * إنشاء كشف حساب للعميل.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $customerId = $request->input('customer_id');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $statement = $this->invoiceService->generateCustomerStatement(
            $customerId,
            $startDate,
            $endDate
        );

        // حفظ كشف الحساب في الجلسة للتصدير لاحقاً
        session(['customer_statement' => $statement]);

        return view('customer_statements.show', compact('statement'));
    }

    /**
     * تصدير كشف حساب العميل.
     *
     * @param  int  $customerId
     * @return \Illuminate\Http\Response
     */
    public function export($customerId)
    {
        $statement = session('customer_statement');
        if (!$statement || $statement['customer']['id'] != $customerId) {
            return redirect()->route('customer-statements.generate-form')
                ->with('error', 'لم يتم العثور على كشف حساب. يرجى إنشاء كشف الحساب أولاً.');
        }

        $content = $this->generateStatementContent($statement);
        $filename = 'customer_statement_' . $customerId . '_' . date('Ymd') . '.pdf';

        // في هذه المرحلة سنعيد المحتوى كنص HTML
        return response($content)
            ->header('Content-Type', 'text/html');

        // يمكن استخدام مكتبة لتحويل HTML إلى PDF
        // return response($pdf)
        //     ->header('Content-Type', 'application/pdf')
        //     ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    /**
     * إنشاء محتوى كشف حساب العميل.
     *
     * @param  array  $statement
     * @return string
     */
    protected function generateStatementContent($statement)
    {
        $transactionsHtml = '';
        foreach ($statement['transactions'] as $transaction) {
            $transactionsHtml .= '<tr>
                <td>' . $transaction['date'] . '</td>
                <td>' . $transaction['description'] . '</td>
                <td>' . $transaction['reference'] . '</td>
                <td>' . number_format($transaction['debit'], 2) . '</td>
                <td>' . number_format($transaction['credit'], 2) . '</td>
                <td>' . number_format($transaction['balance'], 2) . '</td>
            </tr>';
        }

        $html = '<!DOCTYPE html>
        <html lang="ar" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <title>كشف حساب العميل</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                }
                .statement-header {
                    text-align: center;
                    margin-bottom: 30px;
                }
                .customer-info {
                    margin-bottom: 20px;
                }
                .statement-summary {
                    margin-bottom: 30px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: right;
                }
                th {
                    background-color: #f2f2f2;
                }
                .total-row {
                    font-weight: bold;
                }
            </style>
        </head>
        <body>
            <div class="statement-header">
                <h1>كشف حساب العميل</h1>
                <p>الفترة: من ' . $statement['period']['start_date'] . ' إلى ' . $statement['period']['end_date'] . '</p>
            </div>
            
            <div class="customer-info">
                <h2>معلومات العميل</h2>
                <p><strong>اسم العميل:</strong> ' . $statement['customer']['name'] . '</p>
                <p><strong>العنوان:</strong> ' . $statement['customer']['address'] . '</p>
                <p><strong>رقم الهاتف:</strong> ' . $statement['customer']['phone'] . '</p>
                <p><strong>البريد الإلكتروني:</strong> ' . $statement['customer']['email'] . '</p>
            </div>
            
            <div class="statement-summary">
                <h2>ملخص الحساب</h2>
                <table>
                    <tr>
                        <th>البند</th>
                        <th>المبلغ</th>
                    </tr>
                    <tr>
                        <td>الرصيد الافتتاحي</td>
                        <td>' . number_format($statement['opening_balance'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي الفواتير</td>
                        <td>' . number_format($statement['summary']['total_invoiced'], 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي المدفوعات</td>
                        <td>' . number_format($statement['summary']['total_paid'], 2) . '</td>
                    </tr>
                    <tr class="total-row">
                        <td>الرصيد الختامي</td>
                        <td>' . number_format($statement['closing_balance'], 2) . '</td>
                    </tr>
                </table>
            </div>
            
            <div class="transactions">
                <h2>المعاملات</h2>
                <table>
                    <thead>
                        <tr>
                            <th>التاريخ</th>
                            <th>الوصف</th>
                            <th>المرجع</th>
                            <th>مدين</th>
                            <th>دائن</th>
                            <th>الرصيد</th>
                        </tr>
                    </thead>
                    <tbody>
                        ' . $transactionsHtml . '
                    </tbody>
                </table>
            </div>
        </body>
        </html>';

        return $html;
    }
}
