<?php

namespace App\Http\Controllers;

use App\Models\CustomerReward;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CustomerRewardController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $status = $request->input('status');

        $rewards = CustomerReward::where('company_id', $company_id);

        if ($status) {
            $rewards->where('status', $status);
        }

        $rewards = $rewards->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('customer_rewards.index', compact('rewards'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('customer_rewards.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:discount_amount,discount_percentage,free_service,gift',
            'points_required' => 'required|integer|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'discount_percentage' => 'nullable|integer|min:0|max:100',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'required|in:active,inactive',
            'usage_limit' => 'nullable|integer|min:0',
            'terms_conditions' => 'nullable|string',
        ]);

        $company_id = Auth::user()->company_id;

        $reward = new CustomerReward($validated);
        $reward->company_id = $company_id;
        $reward->usage_count = 0;
        $reward->save();

        return redirect()->route('customer-rewards.index')
            ->with('success', 'تم إنشاء المكافأة بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $reward = CustomerReward::with(['redemptions.customer'])
            ->findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض هذه المكافأة');
        }

        return view('customer_rewards.show', compact('reward'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $reward = CustomerReward::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذه المكافأة');
        }

        return view('customer_rewards.edit', compact('reward'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $reward = CustomerReward::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذه المكافأة');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:discount_amount,discount_percentage,free_service,gift',
            'points_required' => 'required|integer|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'discount_percentage' => 'nullable|integer|min:0|max:100',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'required|in:active,inactive',
            'usage_limit' => 'nullable|integer|min:0',
            'terms_conditions' => 'nullable|string',
        ]);

        $reward->update($validated);

        return redirect()->route('customer-rewards.index')
            ->with('success', 'تم تحديث المكافأة بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $reward = CustomerReward::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بحذف هذه المكافأة');
        }

        // التحقق من عدم وجود استخدامات للمكافأة
        if ($reward->redemptions()->count() > 0) {
            return redirect()->route('customer-rewards.index')
                ->with('error', 'لا يمكن حذف المكافأة لأنها مستخدمة بالفعل');
        }

        $reward->delete();

        return redirect()->route('customer-rewards.index')
            ->with('success', 'تم حذف المكافأة بنجاح');
    }
}
