<?php

namespace App\Http\Controllers;

use App\Models\CustomerInteraction;
use App\Models\Customer;
use App\Models\Company;
use App\Models\InteractionFollowUp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CustomerInteractionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $customer_id = $request->input('customer_id');

        $interactions = CustomerInteraction::with(['customer', 'user'])
            ->where('company_id', $company_id);

        if ($customer_id) {
            $interactions->where('customer_id', $customer_id);
        }

        $interactions = $interactions
            ->orderByRaw('CASE WHEN follow_up_date IS NULL THEN 1 ELSE 0 END')
            ->orderBy('follow_up_date', 'asc')
            ->paginate(15);

        return view('customer_interactions.index', compact('interactions'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $customer_id = $request->input('customer_id');

        $customers = Customer::where('company_id', $company_id)->get();
        $customer = $customer_id ? Customer::findOrFail($customer_id) : null;

        return view('customer_interactions.create', compact('customers', 'customer', 'customer_id'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'type' => 'required|in:call,meeting,email,note,task,other',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'interaction_date' => 'required|date',
            'follow_up_date' => 'nullable|date',
            'status' => 'required|in:pending,completed,cancelled',
            'outcome' => 'required|in:positive,negative,neutral,pending',
            'notes' => 'nullable|string',
        ]);

        $company_id = $request->input('company_id', Auth::user()->company_id);

        $interaction = new CustomerInteraction($validated);
        $interaction->company_id = $company_id;
        $interaction->user_id = Auth::id();
        $interaction->save();

        return redirect()->route('customer-interactions.index')
            ->with('success', 'تم إنشاء التفاعل بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $interaction = CustomerInteraction::with(['customer', 'user', 'followUps.user'])
            ->findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($interaction->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض هذا التفاعل');
        }

        return view('customer_interactions.show', compact('interaction'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $interaction = CustomerInteraction::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($interaction->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذا التفاعل');
        }

        $company_id = Auth::user()->company_id;
        $customers = Customer::where('company_id', $company_id)->get();

        return view('customer_interactions.edit', compact('interaction', 'customers'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $interaction = CustomerInteraction::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($interaction->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذا التفاعل');
        }

        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'type' => 'required|in:call,meeting,email,note,task,other',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'interaction_date' => 'required|date',
            'follow_up_date' => 'nullable|date',
            'status' => 'required|in:pending,completed,cancelled',
            'outcome' => 'required|in:positive,negative,neutral,pending',
            'notes' => 'nullable|string',
        ]);

        $interaction->update($validated);

        return redirect()->route('customer-interactions.index')
            ->with('success', 'تم تحديث التفاعل بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $interaction = CustomerInteraction::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($interaction->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بحذف هذا التفاعل');
        }

        $interaction->delete();

        return redirect()->route('customer-interactions.index')
            ->with('success', 'تم حذف التفاعل بنجاح');
    }

    /**
     * عرض التفاعلات المرتبطة بعميل معين
     *
     * @param  int  $customerId
     * @return \Illuminate\Http\Response
     */
    public function customerInteractions($customerId)
    {
        $customer = Customer::findOrFail($customerId);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($customer->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض تفاعلات هذا العميل');
        }

        $interactions = CustomerInteraction::with(['user'])
            ->where('customer_id', $customerId)
            ->orderByRaw('CASE WHEN follow_up_date IS NULL THEN 1 ELSE 0 END')
            ->orderBy('follow_up_date', 'asc')
            ->paginate(15);

        return view('customer_interactions.customer_interactions', compact('interactions', 'customer'));
    }

    /**
     * عرض صفحة إضافة متابعة جديدة للتفاعل
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function addFollowUp($id)
    {
        $interaction = CustomerInteraction::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($interaction->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بإضافة متابعة لهذا التفاعل');
        }

        return view('customer_interactions.add_follow_up', compact('interaction'));
    }

    /**
     * حفظ متابعة جديدة للتفاعل
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function storeFollowUp(Request $request, $id)
    {
        $interaction = CustomerInteraction::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($interaction->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بإضافة متابعة لهذا التفاعل');
        }

        $validated = $request->validate([
            'follow_up_date' => 'required|date',
            'notes' => 'required|string',
            'outcome' => 'required|in:positive,negative,neutral,pending',
            'next_follow_up_date' => 'nullable|date',
        ]);

        // إنشاء سجل متابعة جديد
        $followUp = new InteractionFollowUp($validated);
        $followUp->interaction_id = $id;
        $followUp->user_id = Auth::id();
        $followUp->save();

        // تحديث تاريخ المتابعة التالية في التفاعل الأصلي
        if ($request->has('next_follow_up_date') && $request->next_follow_up_date) {
            $interaction->follow_up_date = $request->next_follow_up_date;
        }

        // تحديث حالة التفاعل إذا لزم الأمر
        if ($request->has('update_interaction_status') && $request->update_interaction_status) {
            $interaction->status = $request->interaction_status;
        }

        // تحديث نتيجة التفاعل إذا لزم الأمر
        if ($request->has('update_interaction_outcome') && $request->update_interaction_outcome) {
            $interaction->outcome = $request->outcome;
        }

        $interaction->save();

        return redirect()->route('customer-interactions.show', $id)
            ->with('success', 'تم إضافة متابعة جديدة بنجاح');
    }
}
