<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\Branch;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CustomersImport;
use App\Exports\CustomersExport;
use App\Services\CustomerAccountService;
use App\Services\CustomerNotificationService;
use Illuminate\Support\Facades\Log;

class CustomerController extends Controller
{
    /**
     * خدمة حسابات العملاء
     *
     * @var CustomerAccountService
     */
    protected $customerAccountService;

    /**
     * خدمة إشعارات العملاء
     *
     * @var CustomerNotificationService
     */
    protected $customerNotificationService;

    /**
     * Create a new controller instance.
     *
     * @param CustomerAccountService $customerAccountService
     * @param CustomerNotificationService $customerNotificationService
     * @return void
     */
    public function __construct(CustomerAccountService $customerAccountService, CustomerNotificationService $customerNotificationService)
    {
        $this->middleware('auth');
        $this->customerAccountService = $customerAccountService;
        $this->customerNotificationService = $customerNotificationService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        // تطبيق عزل الشركات - المستخدمون الذين لديهم صلاحية مدير نظام يرون كل شيء
        if ($user->isSystemAdmin()) {
            // مدراء النظام يمكنهم رؤية جميع العملاء
            $query = Customer::query();
            $company = null;
        } else {
            // جميع المستخدمين المرتبطين بشركة يرون فقط عملاء شركتهم
            $company = $user->company;

            if (!$company) {
                return redirect()->route('companies.index')
                    ->with('error', 'You need to be associated with a company first.');
            }

            $query = Customer::where('company_id', $user->company_id);
        }

        // Filter by branch if specified
        if ($request->has('branch_id') && $request->branch_id) {
            $query->where('branch_id', $request->branch_id);
            $branch = Branch::findOrFail($request->branch_id);
        }

        // Filter by user if specified
        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
            $assignedUser = User::findOrFail($request->user_id);
        }

        // Filter by type if specified
        if ($request->has('type') && $request->type) {
            $query->where('type', $request->type);
        }

        // Filter by status if specified
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        // Search by name, email, or phone
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $customers = $query->orderBy('name')->paginate(10);

        // Get all branches for filter dropdown
        if ($user->isSystemAdmin()) {
            // مدراء النظام يمكنهم رؤية جميع الفروع
            $branches = Branch::where('is_active', true)
                ->orderBy('name')
                ->get();

            // مدراء النظام يمكنهم رؤية جميع المستخدمين
            $users = User::where('is_active', true)
                ->orderBy('name')
                ->get();

            // إحصائيات العملاء لمدراء النظام (جميع العملاء)
            $customerStats = [
                'total' => Customer::count(),
                'active' => Customer::where('status', 'active')->count(),
                'inactive' => Customer::where('status', 'inactive')->count(),
                'lead' => Customer::where('status', 'lead')->count(),
                'prospect' => Customer::where('status', 'prospect')->count(),
            ];
        } else {
            // المستخدمون العاديون يرون فقط فروع شركتهم
            $branches = Branch::where('company_id', $user->company_id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();

            // المستخدمون العاديون يرون فقط مستخدمي شركتهم
            $users = User::where('company_id', $user->company_id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();

            // إحصائيات العملاء للمستخدمين العاديين (عملاء شركتهم فقط)
            $customerStats = [
                'total' => Customer::where('company_id', $user->company_id)->count(),
                'active' => Customer::where('company_id', $user->company_id)->where('status', 'active')->count(),
                'inactive' => Customer::where('company_id', $user->company_id)->where('status', 'inactive')->count(),
                'lead' => Customer::where('company_id', $user->company_id)->where('status', 'lead')->count(),
                'prospect' => Customer::where('company_id', $user->company_id)->where('status', 'prospect')->count(),
            ];
        }

        // Customer types and statuses for filter dropdowns
        $types = ['individual', 'company', 'government'];
        $statuses = ['active', 'inactive', 'lead', 'prospect'];

        return view('customers.index', compact('customers', 'company', 'branches', 'users', 'types', 'statuses', 'customerStats'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company = $user->company;

        if (!$company && !$user->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'You need to be associated with a company first.');
        }

        // Check if user has permission to create customers
        if (!$user->hasPermissionTo('create_customer') && !$user->hasPermissionTo('manage_customers') && !$user->isSystemAdmin()) {
            return redirect()->route('customers.index')
                ->with('error', 'You do not have permission to create customers.');
        }

        // Get all branches for dropdown
        if ($company) {
            $branches = Branch::where('company_id', $company->id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();

            // Get all users for dropdown
            $users = User::where('company_id', $company->id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        } else {
            // للمدراء النظام غير المرتبطين بشركة
            $branches = Branch::where('is_active', true)
                ->orderBy('name')
                ->get();

            $users = User::where('is_active', true)
                ->orderBy('name')
                ->get();
        }

        // Customer types and statuses for dropdowns
        $types = ['individual', 'company', 'government'];
        $statuses = ['active', 'inactive', 'lead', 'prospect'];

        return view('customers.create', compact('company', 'branches', 'users', 'types', 'statuses'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        if (!$company && !$user->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'You need to be associated with a company first.');
        }

        // Check if user has permission to create customers
        if (!$user->hasPermissionTo('create_customer') && !$user->hasPermissionTo('manage_customers') && !$user->isSystemAdmin()) {
            return redirect()->route('customers.index')
                ->with('error', 'You do not have permission to create customers.');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'type' => 'required|in:individual,company,government',
            'status' => 'required|in:active,inactive,lead,prospect',
            'branch_id' => 'required|exists:branches,id',
            'user_id' => 'required|exists:users,id',
            'contact_person' => 'nullable|string|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'contact_position' => 'nullable|string|max:100',
            'address' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'tax_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Verify branch belongs to company
        $branch = Branch::findOrFail($request->branch_id);
        if ($company && $branch->company_id != $company->id) {
            return redirect()->back()
                ->with('error', 'Invalid branch selected.')
                ->withInput();
        }

        // Verify user belongs to company
        $assignedUser = User::findOrFail($request->user_id);
        if ($company && $assignedUser->company_id != $company->id) {
            return redirect()->back()
                ->with('error', 'Invalid user selected.')
                ->withInput();
        }

        // Create customer
        $customer = new Customer();
        $customer->company_id = $company ? $company->id : $branch->company_id;
        $customer->branch_id = $request->branch_id;
        $customer->user_id = $request->user_id;
        $customer->name = $request->name;
        $customer->email = $request->email;
        $customer->phone = $request->phone;
        $customer->type = $request->type;
        $customer->status = $request->status ?? 'active';
        $customer->contact_person = $request->contact_person;
        $customer->contact_phone = $request->contact_phone;
        $customer->contact_position = $request->contact_position;
        $customer->address = $request->address;
        $customer->city = $request->city;
        $customer->state = $request->state;
        $customer->country = $request->country;
        $customer->postal_code = $request->postal_code;
        $customer->tax_number = $request->tax_number;
        $customer->notes = $request->notes;
        $customer->is_active = true;
        $customer->save();

        // إنشاء حساب في دليل الحسابات للعميل إذا كان نشطاً
        if ($customer->status === 'active') {
            // استخدام خدمة حسابات العملاء لإنشاء حساب للعميل
            $customerAccount = $this->customerAccountService->createOrUpdateCustomerAccount($customer);
        }

        // إنشاء إشعار للعميل الجديد
        try {
            $notification = $this->customerNotificationService->newCustomerNotification($customer);
            Log::info('تم إنشاء إشعار للعميل الجديد: ' . $customer->name, [
                'customer_id' => $customer->id,
                'notification_id' => $notification ? $notification->id : null
            ]);
        } catch (\Exception $e) {
            Log::error('فشل إنشاء إشعار للعميل الجديد: ' . $e->getMessage(), [
                'customer_id' => $customer->id,
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
        }

        // تحديد الرسالة المناسبة بناءً على نوع العميل
        $successMessage = '';
        switch ($request->status) {
            case 'lead':
                $successMessage = 'تم إضافة العميل المحتمل بنجاح. يمكنك متابعة التفاعلات معه من خلال قسم التفاعلات.';
                break;
            case 'prospect':
                $successMessage = 'تم إضافة العميل المتوقع بنجاح. تابع تحديث بياناته كلما تقدمت المفاوضات.';
                break;
            case 'active':
                $successMessage = 'تم إضافة العميل النشط بنجاح.';
                $successMessage .= ' وتم إنشاء حساب خاص به في دليل الحسابات.';
                break;
            default:
                $successMessage = 'تم إضافة العميل بنجاح.';
        }

        // إذا كان العميل محتمل أو متوقع، قم بإنشاء تفاعل أولي تلقائياً
        if ($request->status == 'lead' || $request->status == 'prospect') {
            // استخدام نوع تفاعل مسموح به من القائمة المحددة في قاعدة البيانات
            $interactionType = $request->status == 'lead' ? 'call' : 'meeting';
            $interactionTitle = $request->status == 'lead' ? 'تواصل أولي' : 'متابعة محتملة';

            // إنشاء تفاعل أولي
            $interaction = new \App\Models\CustomerInteraction();
            $interaction->company_id = $customer->company_id;
            $interaction->customer_id = $customer->id;
            $interaction->user_id = $request->user_id;
            $interaction->type = $interactionType; // استخدام 'call' أو 'meeting' بدلاً من 'initial_contact' أو 'follow_up'
            $interaction->title = $interactionTitle;
            $interaction->description = 'تم إنشاء هذا التفاعل تلقائياً عند إضافة العميل';
            $interaction->interaction_date = now();
            $interaction->follow_up_date = now()->addDays(7); // متابعة بعد أسبوع
            $interaction->status = 'pending';
            $interaction->outcome = 'pending';
            $interaction->notes = $request->notes ?: 'تم إضافة العميل كـ ' . ($request->status == 'lead' ? 'عميل محتمل' : 'عميل متوقع');
            $interaction->save();

            $successMessage .= ' تم إنشاء تفاعل أولي تلقائياً.';
        }

        return redirect()->route('customers.show', $customer->id)
            ->with('success', $successMessage);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $customer = Customer::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            if ($customer->company_id != $user->company_id) {
                return redirect()->route('customers.index')
                    ->with('error', 'ليس لديك صلاحية لعرض هذا العميل.');
            }
        }

        // Get customer statistics
        $statistics = [
            'locations_count' => $customer->locations()->count(),
            'elevators_count' => $customer->elevators()->count(),
            'contracts_count' => $customer->maintenance_contracts()->count(),
            'invoices_count' => $customer->invoices()->count(),
            'total_invoiced' => $customer->invoices()->sum('total_amount'),
            'total_paid' => $customer->invoices()->sum('paid_amount'),
            'total_due' => $customer->invoices()->sum('total_amount') - $customer->invoices()->sum('paid_amount'),
        ];

        return view('customers.show', compact('customer', 'statistics'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $customer = Customer::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('companies.index')
                    ->with('error', 'You need to be associated with a company first.');
            }

            if ($customer->company_id != $company->id) {
                return redirect()->route('customers.index')
                    ->with('error', 'You do not have permission to edit this customer.');
            }

            if (!$user->hasPermissionTo('edit_customer') && !$user->hasPermissionTo('manage_customers') && $customer->user_id != $user->id) {
                return redirect()->route('customers.index')
                    ->with('error', 'You do not have permission to edit this customer.');
            }
        }

        $companyId = $customer->company_id;

        // Get all branches for dropdown
        $branches = Branch::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Get all users for dropdown
        $users = User::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Customer types and statuses for dropdowns
        $types = ['individual', 'company', 'government'];
        $statuses = ['active', 'inactive', 'lead', 'prospect'];

        return view('customers.edit', compact('customer', 'branches', 'users', 'types', 'statuses'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $customer = Customer::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('companies.index')
                    ->with('error', 'You need to be associated with a company first.');
            }

            if ($customer->company_id != $company->id) {
                return redirect()->route('customers.index')
                    ->with('error', 'You do not have permission to update this customer.');
            }

            if (!$user->hasPermissionTo('edit_customer') && !$user->hasPermissionTo('manage_customers') && $customer->user_id != $user->id) {
                return redirect()->route('customers.index')
                    ->with('error', 'You do not have permission to edit this customer.');
            }
        }

        $companyId = $customer->company_id;

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'type' => 'required|in:individual,company,government',
            'status' => 'required|in:active,inactive,lead,prospect',
            'branch_id' => 'required|exists:branches,id',
            'user_id' => 'required|exists:users,id',
            'contact_person' => 'nullable|string|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'contact_position' => 'nullable|string|max:100',
            'address' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'tax_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Verify branch belongs to company
        $branch = Branch::findOrFail($request->branch_id);
        if ($branch->company_id != $companyId) {
            return redirect()->back()
                ->with('error', 'Invalid branch selected.')
                ->withInput();
        }

        // Verify user belongs to company
        $assignedUser = User::findOrFail($request->user_id);
        if ($assignedUser->company_id != $companyId) {
            return redirect()->back()
                ->with('error', 'Invalid user selected.')
                ->withInput();
        }

        // حفظ الحالة القديمة للعميل قبل التحديث
        $oldStatus = $customer->status;

        // Update customer using update method to ensure observers are triggered
        $customer->update([
            'branch_id' => $request->branch_id,
            'user_id' => $request->user_id,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'type' => $request->type,
            'status' => $request->status ?? 'active',
            'contact_person' => $request->contact_person,
            'contact_phone' => $request->contact_phone,
            'contact_position' => $request->contact_position,
            'address' => $request->address,
            'city' => $request->city,
            'state' => $request->state,
            'country' => $request->country,
            'postal_code' => $request->postal_code,
            'tax_number' => $request->tax_number,
            'notes' => $request->notes,
        ]);

        // إنشاء أو تحديث حساب في دليل الحسابات للعميل إذا كان نشطاً
        if ($customer->status === 'active') {
            // استخدام خدمة حسابات العملاء لإنشاء أو تحديث حساب للعميل
            $customerAccount = $this->customerAccountService->createOrUpdateCustomerAccount($customer);
        }

        // إعداد رسالة النجاح
        $successMessage = 'تم تحديث بيانات العميل بنجاح.';

        // إضافة رسالة عن إنشاء الحساب في دليل الحسابات إذا كان العميل نشطاً
        if ($customer->status === 'active') {
            $successMessage .= ' وتم تحديث حسابه في دليل الحسابات.';
        }

        return redirect()->route('customers.show', $customer->id)
            ->with('success', $successMessage);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $customer = Customer::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('companies.index')
                    ->with('error', 'You need to be associated with a company first.');
            }

            if ($customer->company_id != $company->id) {
                return redirect()->route('customers.index')
                    ->with('error', 'You do not have permission to delete this customer.');
            }

            if (!$user->hasPermissionTo('delete_customer') && !$user->hasPermissionTo('manage_customers')) {
                return redirect()->route('customers.index')
                    ->with('error', 'You do not have permission to delete this customer.');
            }
        }

        // Check if customer has related records
        $hasRelatedRecords = false;
        $relatedRecordsMessage = 'Cannot delete customer because it has related: ';

        if ($customer->locations()->count() > 0) {
            $hasRelatedRecords = true;
            $relatedRecordsMessage .= 'locations, ';
        }

        if ($customer->elevators()->count() > 0) {
            $hasRelatedRecords = true;
            $relatedRecordsMessage .= 'elevators, ';
        }

        if ($customer->maintenance_contracts()->count() > 0) {
            $hasRelatedRecords = true;
            $relatedRecordsMessage .= 'maintenance contracts, ';
        }

        if ($customer->invoices()->count() > 0) {
            $hasRelatedRecords = true;
            $relatedRecordsMessage .= 'invoices, ';
        }

        if ($hasRelatedRecords) {
            return redirect()->route('customers.index')
                ->with('error', rtrim($relatedRecordsMessage, ', ') . '.');
        }

        // Delete customer
        $customer->delete();

        return redirect()->route('customers.index')
            ->with('success', 'Customer deleted successfully.');
    }

    /**
     * Show the form for importing customers from Excel.
     *
     * @return \Illuminate\Http\Response
     */
    public function importForm()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        return view('customers.import');
    }

    /**
     * Import customers from Excel file.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls,csv|max:10240',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            \Log::info('Starting customer import for user: ' . Auth::id());
            
            $import = new CustomersImport;
            Excel::import($import, $request->file('file'));

            \Log::info('Customer import completed successfully. Imported: ' . $import->importedCount . ', Duplicates: ' . $import->duplicateCount . ', Skipped: ' . $import->skippedCount);

            $message = "تم استيراد {$import->importedCount} عميل بنجاح.";
            if ($import->duplicateCount > 0) {
                $message .= " تم تجاهل {$import->duplicateCount} عميل مكرر.";
            }
            if ($import->skippedCount > 0) {
                $message .= " تم تخطي {$import->skippedCount} صف لعدم اكتمال البيانات.";
            }

            return redirect()->route('customers.index')
                ->with('success', $message);
        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errorMsgs = [];
            foreach ($failures as $failure) {
                $errorMsgs[] = 'Row ' . $failure->row() . ': ' . implode(', ', $failure->errors());
            }
            \Log::error('Validation error during customer import: ' . implode(' | ', $errorMsgs));
            return redirect()->back()
                ->with('error', 'فشل التحقق من صحة بعض الصفوف في الملف.')
                ->withErrors($errorMsgs)
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Error during customer import: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء استيراد العملاء: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Export customers to Excel file.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function export(Request $request)
    {
        $user = Auth::user();

        if ($user->isSystemAdmin()) {
            $query = Customer::query();
        } else {
            $query = Customer::where('company_id', $user->company_id);
        }

        // Apply filters from request
        if ($request->has('branch_id') && $request->branch_id) {
            $query->where('branch_id', $request->branch_id);
        }

        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('type') && $request->type) {
            $query->where('type', $request->type);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        return Excel::download(new CustomersExport($query), 'customers.xlsx');
    }
}
