<?php

namespace App\Http\Controllers;

use App\Models\CompanySetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CompanySettingController extends Controller
{
    /**
     * عرض صفحة إعدادات الشركة
     */
    public function index()
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->hasPermission('view_company_settings')) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه الصفحة');
        }

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        // الحصول على إعدادات الشركة
        $settings = CompanySetting::where('company_id', $companyId)->first();

        // إذا لم تكن هناك إعدادات، قم بإنشاء إعدادات افتراضية
        if (!$settings) {
            // الحصول على بيانات الشركة
            $company = \App\Models\Company::find($companyId);

            $settings = CompanySetting::create([
                'company_id' => $companyId,
                'company_name' => $company ? $company->name : 'شركة المصاعد',
                'company_address' => $company ? $company->address : '',
                'company_phone' => $company ? $company->phone : '',
                'company_email' => $company ? $company->email : '',
                'company_website' => $company ? $company->website : '',
                'tax_number' => $company ? $company->tax_number : '',
                'commercial_record' => $company ? $company->cr_number : '',
                'logo_path' => $company && $company->logo ? $company->logo : null,
            ]);
        }

        return view('settings.company.index', compact('settings'));
    }

    /**
     * عرض صفحة تعديل إعدادات الشركة
     */
    public function edit()
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->hasPermission('edit_company_settings')) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه الصفحة');
        }

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        // الحصول على إعدادات الشركة
        $settings = CompanySetting::where('company_id', $companyId)->first();

        // إذا لم تكن هناك إعدادات، قم بإنشاء إعدادات افتراضية
        if (!$settings) {
            // الحصول على بيانات الشركة
            $company = \App\Models\Company::find($companyId);

            $settings = CompanySetting::create([
                'company_id' => $companyId,
                'company_name' => $company ? $company->name : 'شركة المصاعد',
                'company_address' => $company ? $company->address : '',
                'company_phone' => $company ? $company->phone : '',
                'company_email' => $company ? $company->email : '',
                'company_website' => $company ? $company->website : '',
                'tax_number' => $company ? $company->tax_number : '',
                'commercial_record' => $company ? $company->cr_number : '',
                'logo_path' => $company && $company->logo ? $company->logo : null,
            ]);
        }

        return view('settings.company.edit', compact('settings'));
    }

    /**
     * تحديث إعدادات الشركة
     */
    public function update(Request $request)
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->hasPermission('edit_company_settings')) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه الصفحة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'company_name' => 'required|string|max:255',
            'company_address' => 'nullable|string|max:500',
            'company_phone' => 'nullable|string|max:20',
            'company_email' => 'nullable|email|max:255',
            'company_website' => 'nullable|url|max:255',
            'tax_number' => 'nullable|string|max:50',
            'commercial_record' => 'nullable|string|max:50',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        // الحصول على إعدادات الشركة
        $settings = CompanySetting::where('company_id', $companyId)->first();

        // إذا لم تكن هناك إعدادات، قم بإنشاء إعدادات جديدة
        if (!$settings) {
            $settings = new CompanySetting();
            $settings->company_id = $companyId;
        }

        // تحديث البيانات
        $settings->company_name = $request->company_name;
        $settings->company_address = $request->company_address;
        $settings->company_phone = $request->company_phone;
        $settings->company_email = $request->company_email;
        $settings->company_website = $request->company_website;
        $settings->tax_number = $request->tax_number;
        $settings->commercial_record = $request->commercial_record;
        $settings->save();

        return redirect()->route('company_settings.index')->with('success', 'تم تحديث إعدادات الشركة بنجاح');
    }

    /**
     * رفع شعار الشركة
     */
    public function uploadLogo(Request $request)
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->hasPermission('edit_company_settings')) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه الصفحة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'logo' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            // الحصول على الشركة الحالية للمستخدم
            $companyId = auth()->user()->company_id;

            // الحصول على الشركة
            $company = \App\Models\Company::find($companyId);
            if (!$company) {
                return redirect()->back()->with('error', 'لم يتم العثور على الشركة');
            }

            // الحصول على إعدادات الشركة
            $settings = CompanySetting::where('company_id', $companyId)->first();

            // إذا لم تكن هناك إعدادات، قم بإنشاء إعدادات جديدة
            if (!$settings) {
                $settings = new CompanySetting();
                $settings->company_id = $companyId;
            }

            // حذف الشعار القديم إذا كان موجودًا
            if ($settings->logo_path) {
                $oldPath = 'public/' . $settings->logo_path;
                if (Storage::exists($oldPath)) {
                    Storage::delete($oldPath);
                }
            }

            // رفع الشعار الجديد
            if ($request->hasFile('logo') && $request->file('logo')->isValid()) {
                $logoFile = $request->file('logo');
                $fileName = time() . '_' . $logoFile->getClientOriginalName();
                $logoPath = $logoFile->storeAs('company_logos', $fileName, 'public');

                // تحديث مسار الشعار في إعدادات الشركة
                $settings->logo_path = $logoPath;
                $settings->save();

                // تحديث مسار الشعار في جدول الشركات أيضًا
                $company->logo = $logoPath;
                $company->save();

                return redirect()->route('company_settings.edit')->with('success', 'تم تحديث شعار الشركة بنجاح');
            } else {
                return redirect()->back()->with('error', 'حدث خطأ أثناء رفع الملف');
            }
        } catch (\Exception $e) {
            // تسجيل الخطأ للتصحيح
            \Log::error('خطأ في رفع شعار الشركة: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء رفع الشعار: ' . $e->getMessage());
        }
    }
}
