<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Company;
use App\Models\Branch;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CompanyManagementController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        // فلترة الشركات بناءً على ارتباط المستخدم بالشركة
        if ($user->isSystemAdmin()) {
            // المستخدمون غير المرتبطين بأي شركة (مدراء النظام) يمكنهم رؤية جميع الشركات
            $companies = Company::orderBy('name')->paginate(10);
        } else {
            // جميع المستخدمين المرتبطين بشركة يرون فقط شركتهم بغض النظر عن رتبتهم
            $companies = Company::where('id', $user->company_id)->orderBy('name')->paginate(10);
        }

        return view('company_management.index', compact('companies'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();

        // فقط المستخدمون بمنصب super_admin أو المستخدمون غير المنتمين لشركة يمكنهم إنشاء شركات جديدة
        if (!$user->isSystemAdmin() && $user->company_id != null) {
            abort(403, 'غير مصرح لك بإنشاء شركات جديدة. فقط مستخدمي سوبر أدمن أو المستخدمين الجدد يمكنهم إنشاء شركات.');
        }

        // الحصول على المستخدمين غير المرتبطين بشركة
        $availableUsers = User::whereNull('company_id')->orWhere('company_id', 0)->get();

        return view('company_management.create', compact('availableUsers'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        // فقط المستخدمون بمنصب super_admin أو المستخدمون غير المنتمين لشركة يمكنهم إنشاء شركات جديدة
        if (!$user->isSystemAdmin() && $user->company_id != null) {
            abort(403, 'غير مصرح لك بإنشاء شركات جديدة. فقط مستخدمي سوبر أدمن أو المستخدمين الجدد يمكنهم إنشاء شركات.');
        }

        // التحقق من عدد الشركات التي أنشأها المستخدم
        // إذا كان المستخدم ليس سوبر ادمن ولديه شركة بالفعل، فلا يمكنه إنشاء شركة أخرى
        if (!$user->isSystemAdmin()) {
            $companiesCount = Company::where('created_by', $user->id)->count();
            if ($companiesCount > 0) {
                return redirect()->route('company_management.index')
                    ->with('error', 'لا يمكنك إنشاء أكثر من شركة واحدة. فقط مستخدمي سوبر أدمن يمكنهم إنشاء شركات متعددة.');
            }
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'mobile' => 'nullable|string|max:20',
            'website' => 'nullable|string|max:255|url',
            'address' => 'nullable|string|max:255',
            'address_en' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'cr_number' => 'nullable|string|max:50',
            'vat_number' => 'nullable|string|max:50',
            'tax_number' => 'nullable|string|max:50',
            'currency' => 'required|string|size:3',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'vat_rate' => 'nullable|numeric|min:0|max:100',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'notes' => 'nullable|string',
            'manager_id' => 'nullable|exists:users,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Generate slug from name
        $slug = Str::slug($request->name);

        // Generate database prefix from name
        $databasePrefix = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $request->name));

        // Handle logo upload
        $logoPath = null;
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('company_logos', 'public');
        }

        // Create company
        $company = Company::create([
            'name' => $request->name,
            'name_en' => $request->name_en,
            'slug' => $slug,
            'domain' => $request->domain ?? $slug . '.example.com',
            'database_prefix' => $databasePrefix,
            'cr_number' => $request->cr_number,
            'vat_number' => $request->vat_number,
            'logo' => $logoPath,
            'email' => $request->email,
            'phone' => $request->phone,
            'mobile' => $request->mobile,
            'website' => $request->website,
            'address' => $request->address,
            'address_en' => $request->address_en,
            'city' => $request->city,
            'state' => $request->state,
            'country' => $request->country,
            'postal_code' => $request->postal_code,
            'tax_number' => $request->tax_number,
            'currency' => $request->currency,
            'tax_rate' => $request->tax_rate,
            'vat_rate' => $request->vat_rate ?? 15.00, // Default VAT rate in Saudi Arabia
            'is_active' => true,
            'notes' => $request->notes,
            'subscription_plan' => 'basic',
            'subscription_ends_at' => now()->addYear(),
            'created_by' => $user->id, // تسجيل المستخدم الذي أنشأ الشركة
        ]);

        // Create main branch for the company
        $branch = Branch::create([
            'company_id' => $company->id,
            'name' => 'الفرع الرئيسي',
            'name_en' => 'Main Branch',
            'code' => 'MAIN-' . $company->id,
            'email' => $company->email,
            'phone' => $company->phone,
            'mobile' => $company->mobile,
            'address' => $company->address,
            'address_en' => $company->address_en,
            'city' => $company->city,
            'state' => $company->state,
            'country' => $company->country,
            'postal_code' => $company->postal_code,
            'is_main_branch' => true,
            'is_active' => true,
        ]);

        // ربط المستخدم المدير بالشركة إذا تم اختياره
        if ($request->filled('manager_id')) {
            $manager = User::find($request->manager_id);
            if ($manager) {
                // التحقق من أن المستخدم غير مرتبط بشركة أخرى
                if (!$manager->company_id) {
                    $manager->company_id = $company->id;
                    $manager->save();
                }
            }
        }

        return redirect()->route('company_management.show', $company->id)
            ->with('success', 'تم إنشاء الشركة بنجاح.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company = Company::findOrFail($id);

        // التحقق من صلاحية الوصول - فقط المستخدمون غير المرتبطين بشركة أو المرتبطين بهذه الشركة يمكنهم الوصول
        if (!$user->isSystemAdmin() && $user->company_id != $company->id) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه الشركة');
        }

        // Get company statistics
        $statistics = [
            'branches_count' => $company->branches()->count(),
            'users_count' => $company->users()->count(),
            'customers_count' => $company->customers()->count(),
            'elevators_count' => $company->elevators()->count(),
            'contracts_count' => $company->maintenance_contracts()->count(),
            'invoices_count' => $company->invoices()->count(),
        ];

        return view('company_management.show', compact('company', 'statistics'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company = Company::findOrFail($id);

        // التحقق من صلاحية الوصول - فقط المستخدمون غير المرتبطين بشركة أو المرتبطين بهذه الشركة يمكنهم الوصول
        if (!$user->isSystemAdmin() && $user->company_id != $company->id) {
            abort(403, 'غير مصرح لك بتعديل هذه الشركة');
        }

        // الحصول على المستخدمين غير المرتبطين بشركة أو المرتبطين بهذه الشركة
        $availableUsers = User::where(function ($query) use ($company) {
            $query->whereNull('company_id')
                ->orWhere('company_id', 0)
                ->orWhere('company_id', $company->id);
        })->get();

        // الحصول على المستخدم المدير الحالي للشركة (إن وجد)
        $currentManager = User::where('company_id', $company->id)->first();

        return view('company_management.edit', compact('company', 'availableUsers', 'currentManager'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $company = Company::findOrFail($id);

        // التحقق من صلاحية الوصول - فقط المستخدمون غير المرتبطين بشركة أو المرتبطين بهذه الشركة يمكنهم الوصول
        if (!$user->isSystemAdmin() && $user->company_id != $company->id) {
            abort(403, 'غير مصرح لك بتعديل هذه الشركة');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'mobile' => 'nullable|string|max:20',
            'website' => 'nullable|string|max:255|url',
            'address' => 'nullable|string|max:255',
            'address_en' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'cr_number' => 'nullable|string|max:50',
            'vat_number' => 'nullable|string|max:50',
            'tax_number' => 'nullable|string|max:50',
            'currency' => 'required|string|size:3',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'vat_rate' => 'nullable|numeric|min:0|max:100',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'notes' => 'nullable|string',
            'manager_id' => 'nullable|exists:users,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($company->logo) {
                Storage::disk('public')->delete($company->logo);
            }

            $logoPath = $request->file('logo')->store('company_logos', 'public');
            $company->logo = $logoPath;
        }

        // Update company
        $company->name = $request->name;
        $company->name_en = $request->name_en;
        $company->email = $request->email;
        $company->phone = $request->phone;
        $company->mobile = $request->mobile;
        $company->website = $request->website;
        $company->address = $request->address;
        $company->address_en = $request->address_en;
        $company->city = $request->city;
        $company->state = $request->state;
        $company->country = $request->country;
        $company->postal_code = $request->postal_code;
        $company->cr_number = $request->cr_number;
        $company->vat_number = $request->vat_number;
        $company->tax_number = $request->tax_number;
        $company->currency = $request->currency;
        $company->tax_rate = $request->tax_rate;
        $company->vat_rate = $request->vat_rate;
        $company->notes = $request->notes;

        // Only super admin can update these fields
        if ($user->isSystemAdmin()) {
            $company->is_active = $request->has('is_active');
        }

        $company->save();

        // تحديث ربط المستخدم المدير بالشركة إذا كان المستخدم الحالي يملك صلاحية super_admin
        if ($user->isSystemAdmin() && $request->filled('manager_id')) {
            // إلغاء ربط أي مستخدمين سابقين مرتبطين بهذه الشركة
            User::where('company_id', $company->id)->update(['company_id' => null]);

            // ربط المستخدم المدير الجديد بالشركة
            $manager = User::find($request->manager_id);
            if ($manager) {
                $manager->company_id = $company->id;
                $manager->save();
            }
        }

        return redirect()->route('company_management.show', $company->id)
            ->with('success', 'تم تحديث الشركة بنجاح.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company = Company::findOrFail($id);

        // التحقق من صلاحية الوصول - فقط المشرفون والمستخدمون بمنصب super_admin يمكنهم حذف الشركات
        if (!$user->isSystemAdmin()) {
            abort(403, 'غير مصرح لك بحذف الشركات');
        }

        // Check if company has related records
        $hasRelatedRecords =
            $company->users()->count() > 0 ||
            $company->branches()->count() > 0 ||
            $company->customers()->count() > 0 ||
            $company->elevators()->count() > 0 ||
            $company->maintenance_contracts()->count() > 0 ||
            $company->invoices()->count() > 0;

        if ($hasRelatedRecords) {
            return redirect()->route('company_management.index')
                ->with('error', 'لا يمكن حذف الشركة لأنها تحتوي على سجلات مرتبطة. يمكنك تعطيلها بدلاً من ذلك.');
        }

        // Delete logo if exists
        if ($company->logo) {
            Storage::disk('public')->delete($company->logo);
        }

        // Delete company
        $company->delete();

        return redirect()->route('company_management.index')
            ->with('success', 'تم حذف الشركة بنجاح.');
    }
}
