<?php

namespace App\Http\Controllers;

use App\Models\ChartOfAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ChartOfAccountController extends Controller
{
    /**
     * عرض قائمة الحسابات
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if (!$companyId && !$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
        }

        // التحقق من وجود حسابات للشركة الحالية (فقط إذا تم تحديد شركة)
        if ($companyId) {
            $accountsCount = ChartOfAccount::where('company_id', $companyId)->count();

            // إذا لم يكن هناك حسابات، قم بإنشاء الحسابات الأساسية
            if ($accountsCount === 0) {
                $this->createDefaultAccounts($companyId);
                
                $accounts = ChartOfAccount::where('company_id', $companyId)
                    ->with('parent')
                    ->orderBy('account_code')
                    ->paginate(20);

                return view('accounting.chart_of_accounts.index', compact('accounts', 'companyId'))
                    ->with('success', 'تم إنشاء الحسابات الأساسية بنجاح! يمكنك الآن إضافة حسابات فرعية.');
            }
        }

        $query = ChartOfAccount::with('parent');

        if ($companyId) {
            $query->where('company_id', $companyId);
        } elseif (!$user->isSystemAdmin()) {
             // This case shouldn't be reached due to the check above, but for safety:
             $query->where('company_id', -1); 
        }

        $accounts = $query->orderBy('account_code')
            ->paginate(20);

        return view('accounting.chart_of_accounts.index', compact('accounts', 'companyId'));
    }

    /**
     * إنشاء الحسابات الأساسية للشركة
     *
     * @param int $companyId
     * @return void
     */
    private function createDefaultAccounts($companyId)
    {
        $defaultAccounts = [
            [
                'account_code' => '1000',
                'name' => 'الأصول',
                'name_en' => 'Assets',
                'account_type' => 'asset',
                'account_nature' => 'parent',
                'is_parent' => true,
                'level' => 1,
                'parent_id' => null,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'is_active' => true,
                'is_cash_account' => false,
                'is_bank_account' => false,
                'description' => 'حساب رئيسي للأصول',
            ],
            [
                'account_code' => '2000',
                'name' => 'الخصوم',
                'name_en' => 'Liabilities',
                'account_type' => 'liability',
                'account_nature' => 'parent',
                'is_parent' => true,
                'level' => 1,
                'parent_id' => null,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'is_active' => true,
                'is_cash_account' => false,
                'is_bank_account' => false,
                'description' => 'حساب رئيسي للخصوم',
            ],
            [
                'account_code' => '3000',
                'name' => 'رأس المال وحقوق الملكية',
                'name_en' => 'Equity',
                'account_type' => 'equity',
                'account_nature' => 'parent',
                'is_parent' => true,
                'level' => 1,
                'parent_id' => null,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'is_active' => true,
                'is_cash_account' => false,
                'is_bank_account' => false,
                'description' => 'حساب رئيسي لرأس المال وحقوق الملكية',
            ],
            [
                'account_code' => '4000',
                'name' => 'الإيرادات',
                'name_en' => 'Revenues',
                'account_type' => 'revenue',
                'account_nature' => 'parent',
                'is_parent' => true,
                'level' => 1,
                'parent_id' => null,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'is_active' => true,
                'is_cash_account' => false,
                'is_bank_account' => false,
                'description' => 'حساب رئيسي للإيرادات',
            ],
            [
                'account_code' => '5000',
                'name' => 'المصروفات',
                'name_en' => 'Expenses',
                'account_type' => 'expense',
                'account_nature' => 'parent',
                'is_parent' => true,
                'level' => 1,
                'parent_id' => null,
                'opening_balance' => 0,
                'current_balance' => 0,
                'currency' => 'SAR',
                'is_active' => true,
                'is_cash_account' => false,
                'is_bank_account' => false,
                'description' => 'حساب رئيسي للمصروفات',
            ],
        ];

        foreach ($defaultAccounts as $accountData) {
            $accountData['company_id'] = $companyId;
            ChartOfAccount::create($accountData);
        }
    }

    /**
     * عرض نموذج إنشاء حساب جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if ($user->isSystemAdmin() && !$companyId) {
             // We can either redirect or allow selecting company in the view
             // For now, let's fetch all companies if it's admin and no companyId
             $companies = \App\Models\Company::all();
             return view('accounting.chart_of_accounts.create', compact('companies'));
        }

        $parentAccounts = ChartOfAccount::where('company_id', $companyId)
            ->orderBy('account_code')
            ->get();

        $accountTypes = [
            'asset' => 'أصول',
            'liability' => 'خصوم',
            'equity' => 'حقوق ملكية',
            'revenue' => 'إيرادات',
            'expense' => 'مصروفات'
        ];

        return view('accounting.chart_of_accounts.create', compact('parentAccounts', 'accountTypes', 'companyId'));
    }

    /**
     * حفظ حساب جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        $validator = Validator::make($request->all(), [
            'company_id' => $user->isSystemAdmin() ? 'required|exists:companies,id' : 'nullable',
            'account_code' => [
                'required',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($companyId) {
                    if (ChartOfAccount::where('company_id', $companyId)->where('account_code', $value)->exists()) {
                        $fail('رمز الحساب مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'name' => 'required|string|max:255',
            'account_type' => 'required|string|in:asset,liability,equity,revenue,expense',
            'sub_type' => 'nullable|string|max:255',
            'account_nature' => 'required|string|in:parent,child',
            'parent_id' => $request->input('account_nature') === 'child' ? 'required|exists:chart_of_accounts,id' : 'nullable|exists:chart_of_accounts,id',
            'description' => 'nullable|string',
            'opening_balance' => 'required|numeric',
            'currency' => 'required|string|max:10',
            'is_active' => 'boolean',
            'is_cash_account' => 'boolean',
            'is_bank_account' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // تحويل القيم المنطقية
        $data = $request->all();
        $data['is_active'] = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_cash_account'] = filter_var($request->input('is_cash_account', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_bank_account'] = filter_var($request->input('is_bank_account', false), FILTER_VALIDATE_BOOLEAN);
        $data['company_id'] = $companyId;
        $data['current_balance'] = $request->opening_balance;

        // تحديد ما إذا كان الحساب رئيسي أو فرعي
        $data['is_parent'] = $request->input('account_nature') === 'parent';

        // إذا كان الحساب رئيسي ولم يتم تحديد حساب أب، اجعله حساب من المستوى الأول
        if ($data['is_parent'] && empty($data['parent_id'])) {
            $data['level'] = 1;
        } else {
            // إذا كان له حساب أب، احسب المستوى بناءً على مستوى الحساب الأب
            if (!empty($data['parent_id'])) {
                $parentAccount = ChartOfAccount::find($data['parent_id']);
                $data['level'] = $parentAccount ? $parentAccount->level + 1 : 1;
            } else {
                $data['level'] = 1;
            }
        }

        $account = ChartOfAccount::create($data);

        return redirect()->route('chart-of-accounts.index')
            ->with('success', 'تم إنشاء الحساب بنجاح');
    }

    /**
     * عرض تفاصيل حساب محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $query = ChartOfAccount::with(['parent', 'children', 'journalItems' => function ($query) {
            $query->with('journalEntry')->latest()->take(15);
        }]);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $account = $query->findOrFail($id);

        return view('accounting.chart_of_accounts.show', compact('account'));
    }

    /**
     * عرض نموذج تعديل حساب
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $query = ChartOfAccount::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $account = $query->findOrFail($id);

        $parentAccounts = ChartOfAccount::where('company_id', $account->company_id)
            ->where('id', '!=', $id)
            ->orderBy('account_code')
            ->get();

        $accountTypes = [
            'asset' => 'أصول',
            'liability' => 'خصوم',
            'equity' => 'حقوق ملكية',
            'revenue' => 'إيرادات',
            'expense' => 'مصروفات'
        ];

        return view('accounting.chart_of_accounts.edit', compact('account', 'parentAccounts', 'accountTypes'));
    }

    /**
     * تحديث حساب محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = ChartOfAccount::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $account = $query->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'account_code' => 'required|string|max:50|unique:chart_of_accounts,account_code,' . $id . ',id,company_id,' . $account->company_id,
            'name' => 'required|string|max:255',
            'account_type' => 'required|string|in:asset,liability,equity,revenue,expense',
            'sub_type' => 'nullable|string|max:255',
            'parent_id' => 'nullable|exists:chart_of_accounts,id',
            'description' => 'nullable|string',
            'currency' => 'required|string|max:10',
            'is_active' => 'boolean',
            'is_cash_account' => 'boolean',
            'is_bank_account' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من عدم اختيار الحساب نفسه كأب
        if ($request->parent_id == $id) {
            return redirect()->back()
                ->withErrors(['parent_id' => 'لا يمكن اختيار الحساب نفسه كحساب أب'])
                ->withInput();
        }

        // تحويل القيم المنطقية
        $data = $request->all();
        $data['is_active'] = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_cash_account'] = filter_var($request->input('is_cash_account', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_bank_account'] = filter_var($request->input('is_bank_account', false), FILTER_VALIDATE_BOOLEAN);

        $account->update($data);

        return redirect()->route('chart-of-accounts.index', ['company_id' => $account->company_id])
            ->with('success', 'تم تحديث الحساب بنجاح');
    }

    /**
     * حذف حساب محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $query = ChartOfAccount::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $account = $query->findOrFail($id);

        // التحقق من عدم وجود حسابات فرعية
        if ($account->children()->count() > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف الحساب لوجود حسابات فرعية مرتبطة به');
        }

        // التحقق من عدم وجود قيود محاسبية مرتبطة بالحساب
        if ($account->journalItems()->count() > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف الحساب لوجود قيود محاسبية مرتبطة به');
        }

        $companyId = $account->company_id;
        $account->delete();

        return redirect()->route('chart-of-accounts.index', ['company_id' => $companyId])
            ->with('success', 'تم حذف الحساب بنجاح');
    }

    /**
     * عرض شجرة الحسابات
     *
     * @return \Illuminate\Http\Response
     */
    public function tree(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if (!$companyId && !$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
        }

        // الحصول على الحسابات الرئيسية مع الحسابات الفرعية
        $query = ChartOfAccount::whereNull('parent_id');

        if ($companyId) {
            $query->where('company_id', $companyId)
                ->with(['children' => function ($query) use ($companyId) {
                    $query->where('company_id', $companyId)
                        ->with(['children' => function ($query) use ($companyId) {
                            $query->where('company_id', $companyId);
                        }]);
                }]);
        } else {
             // Admin viewing global tree (might be messy, but allowed)
             $query->with(['children.children']);
        }

        $accounts = $query->orderBy('account_code')->get();

        // إذا لم يتم العثور على حسابات، قم بإعادة توجيه المستخدم إلى صفحة إنشاء الحسابات
        if ($accounts->isEmpty() && $companyId) {
            return redirect()->route('chart-of-accounts.index', ['company_id' => $companyId])
                ->with('warning', 'لا توجد حسابات في شجرة الحسابات لهذه الشركة. يرجى إنشاء الحسابات أولاً.');
        }

        return view('accounting.chart_of_accounts.tree_modern', compact('accounts', 'companyId'));
    }

    /**
     * عرض تقرير ميزان المراجعة
     *
     * @return \Illuminate\Http\Response
     */
    public function trialBalance(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;
        $date = $request->input('date', now()->format('Y-m-d'));

        $query = ChartOfAccount::query();

        if ($companyId) {
            $query->where('company_id', $companyId);
        } elseif (!$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
        }

        $accounts = $query->where('is_active', true)
            ->orderBy('account_code')
            ->get();

        return view('accounting.chart_of_accounts.trial_balance', compact('accounts', 'date'));
    }
}
