<?php

namespace App\Http\Controllers;

use App\Models\CashRegister;
use App\Models\CashTransaction;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\FiscalYear;
use App\Models\AccountingPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CashDepositsImport;

class CashTransactionController extends Controller
{
    /**
     * عرض قائمة المعاملات
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = CashTransaction::where('company_id', Auth::user()->company_id)
            ->with(['cashRegister', 'relatedCashRegister', 'user']);

        // تصفية حسب الخزينة
        if ($request->has('cash_register_id') && $request->cash_register_id) {
            $query->where('cash_register_id', $request->cash_register_id);
        }

        // تصفية حسب النوع
        if ($request->has('type') && $request->type) {
            $query->where('type', $request->type);
        }

        // تصفية حسب التاريخ
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $transactions = $query->orderBy('created_at', 'desc')
            ->paginate(15);

        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        return view('cash_transactions.index', compact('transactions', 'cashRegisters'));
    }

    /**
     * عرض قائمة الإيداعات
     *
     * @return \Illuminate\Http\Response
     */
    public function deposits(Request $request)
    {
        $query = CashTransaction::where('company_id', Auth::user()->company_id)
            ->where('type', 'deposit')
            ->with(['cashRegister', 'user']);

        // تصفية حسب الخزينة
        if ($request->has('cash_register_id') && $request->cash_register_id) {
            $query->where('cash_register_id', $request->cash_register_id);
        }

        // تصفية حسب التاريخ
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $transactions = $query->orderBy('created_at', 'desc')
            ->paginate(15);

        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        return view('cash_transactions.deposits_index', compact('transactions', 'cashRegisters'));
    }

    /**
     * عرض نموذج استيراد الإيداعات
     *
     * @return \Illuminate\Http\Response
     */
    public function importDepositsForm()
    {
        return view('cash_transactions.import_deposits');
    }

    /**
     * استيراد الإيداعات من ملف إكسيل
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function importDeposits(Request $request)
    {
        $request->validate([
            'import_file' => 'required|mimes:xlsx,xls,csv|max:2048',
        ], [
            'import_file.required' => 'يرجى اختيار ملف للاستيراد',
            'import_file.mimes' => 'يجب أن يكون الملف من نوع إكسيل (xlsx, xls, csv)',
            'import_file.max' => 'حجم الملف يجب ألا يتجاوز 2 ميجابايت',
        ]);

        try {
            Excel::import(new CashDepositsImport, $request->file('import_file'));

            return redirect()->route('cash-deposits.index')
                ->with('success', 'تم استيراد الإيداعات بنجاح');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء استيراد الملف: ' . $e->getMessage());
        }
    }

    /**
     * عرض قائمة السحوبات
     *
     * @return \Illuminate\Http\Response
     */
    public function withdrawals(Request $request)
    {
        $query = CashTransaction::where('company_id', Auth::user()->company_id)
            ->where('type', 'withdrawal')
            ->with(['cashRegister', 'user']);

        // تصفية حسب الخزينة
        if ($request->has('cash_register_id') && $request->cash_register_id) {
            $query->where('cash_register_id', $request->cash_register_id);
        }

        // تصفية حسب التاريخ
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $transactions = $query->orderBy('created_at', 'desc')
            ->paginate(15);

        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        return view('cash_transactions.withdrawals_index', compact('transactions', 'cashRegisters'));
    }

    /**
     * عرض نموذج إنشاء معاملة جديدة
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('cash_transactions.create', compact('cashRegisters'));
    }

    /**
     * عرض نموذج إنشاء إيداع جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function createDepositForm()
    {
        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // الحصول على الحسابات الفرعية من دليل الحسابات
        $accounts = \App\Models\ChartOfAccount::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->where('is_parent', false)  // فقط الحسابات الفرعية
            ->orderBy('account_code')
            ->get();

        // الحصول على قائمة المصاعد
        $elevators = \App\Models\Elevator::where('company_id', Auth::user()->company_id)
            ->orderBy('serial_number')
            ->get();

        return view('cash_transactions.create_deposit', compact('cashRegisters', 'accounts', 'elevators'));
    }

    /**
     * عرض نموذج إنشاء سحب جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function createWithdrawalForm()
    {
        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // الحصول على الحسابات الفرعية من دليل الحسابات
        $accounts = \App\Models\ChartOfAccount::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->where('is_parent', false)  // فقط الحسابات الفرعية
            ->orderBy('account_code')
            ->get();

        return view('cash_transactions.create_withdrawal', compact('cashRegisters', 'accounts'));
    }

    /**
     * حفظ معاملة جديدة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $rules = [
            'cash_register_id' => 'required|exists:cash_registers,id',
            'type' => 'required|in:deposit,withdrawal,transfer',
            'amount' => 'required|numeric|min:0.01',
            'notes' => 'nullable|string',
            'attachment' => 'nullable|file|mimes:jpeg,png,jpg,pdf|max:2048',
        ];

        // إضافة قواعد التحقق حسب نوع المعاملة
        if ($request->type == 'transfer') {
            $rules['related_cash_register_id'] = 'required|exists:cash_registers,id|different:cash_register_id';
        } elseif ($request->type == 'deposit' || $request->type == 'withdrawal') {
            $rules['party_name'] = 'required|string|max:255';
            $rules['party_type'] = 'required|string|max:255';

            // إضافة قاعدة تحقق لتاريخ الاستلام للإيداعات فقط
            if ($request->type == 'deposit') {
                $rules['receipt_date'] = 'required|date';
            }

            // إضافة قاعدة تحقق لتاريخ الصرف للسحوبات فقط
            if ($request->type == 'withdrawal') {
                $rules['withdrawal_date'] = 'required|date';
            }
        }

        $messages = [
            'cash_register_id.required' => 'يرجى اختيار الخزينة',
            'cash_register_id.exists' => 'الخزينة المحددة غير موجودة',
            'type.required' => 'يرجى اختيار نوع المعاملة',
            'type.in' => 'نوع المعاملة غير صالح',
            'amount.required' => 'يرجى إدخال المبلغ',
            'amount.numeric' => 'المبلغ يجب أن يكون رقمًا',
            'amount.min' => 'المبلغ يجب أن يكون أكبر من صفر',
            'related_cash_register_id.required' => 'يرجى اختيار الخزينة المستلمة',
            'related_cash_register_id.exists' => 'الخزينة المستلمة غير موجودة',
            'related_cash_register_id.different' => 'لا يمكن التحويل إلى نفس الخزينة',
            'party_name.required' => 'يرجى إدخال اسم الطرف',
            'party_type.required' => 'يرجى اختيار نوع الطرف',
            'receipt_date.required' => 'يرجى إدخال تاريخ الاستلام',
            'receipt_date.date' => 'تاريخ الاستلام غير صالح',
            'withdrawal_date.required' => 'يرجى إدخال تاريخ الصرف',
            'withdrawal_date.date' => 'تاريخ الصرف غير صالح',
        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // الحصول على الخزينة
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($request->cash_register_id);

        // التحقق من الرصيد في حالة السحب أو التحويل
        if (($request->type == 'withdrawal' || $request->type == 'transfer') && $cashRegister->current_balance < $request->amount) {
            return redirect()->back()
                ->with('error', 'رصيد الخزينة غير كافٍ لإجراء هذه العملية')
                ->withInput();
        }

        // معالجة المرفق إذا وجد
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('cash_transactions', 'public');
        }

        DB::beginTransaction();

        try {
            // إنشاء المعاملة
            $transaction = new CashTransaction([
                'cash_register_id' => $request->cash_register_id,
                'company_id' => Auth::user()->company_id,
                'user_id' => Auth::id(),
                'type' => $request->type,
                'amount' => $request->amount,
                'receipt_date' => $request->type == 'deposit' ? $request->receipt_date : null,
                'withdrawal_date' => $request->type == 'withdrawal' ? $request->withdrawal_date : null,
                'related_cash_register_id' => $request->type == 'transfer' ? $request->related_cash_register_id : null,
                'party_name' => ($request->type == 'deposit' || $request->type == 'withdrawal') ? $request->party_name : null,
                'party_type' => ($request->type == 'deposit' || $request->type == 'withdrawal') ? $request->party_type : null,
                'elevator_id' => $request->elevator_id, // إضافة رقم المصعد
                'notes' => $request->notes,
                'attachment' => $attachmentPath,
            ]);

            $transaction->save();

            // تحديث رصيد الخزينة
            switch ($request->type) {
                case 'deposit':
                    $cashRegister->current_balance += $request->amount;
                    break;
                case 'withdrawal':
                    $cashRegister->current_balance -= $request->amount;
                    break;
                case 'transfer':
                    $cashRegister->current_balance -= $request->amount;

                    // تحديث رصيد الخزينة المستلمة
                    $relatedCashRegister = CashRegister::where('company_id', Auth::user()->company_id)
                        ->findOrFail($request->related_cash_register_id);
                    $relatedCashRegister->current_balance += $request->amount;
                    $relatedCashRegister->save();
                    break;
            }

            $cashRegister->save();

            // إنشاء قيد يومية تلقائي حسب نوع المعاملة
            $journalEntry = null;
            if ($request->type == 'deposit') {
                $journalEntry = $this->createJournalEntryForDeposit($transaction);
            } elseif ($request->type == 'withdrawal') {
                $journalEntry = $this->createJournalEntryForWithdrawal($transaction);
            }

            // التحقق من إنشاء قيد اليومية بنجاح
            if (($request->type == 'deposit' || $request->type == 'withdrawal') && !$journalEntry) {
                // تسجيل معلومات تصحيح الأخطاء
                \Log::error('فشل في إنشاء قيد اليومية التلقائي للمعاملة رقم: ' . $transaction->id);

                // التحقق من وجود سنة مالية نشطة
                $fiscalYear = FiscalYear::where('company_id', Auth::user()->company_id)
                    ->where('is_active', true)
                    ->where('is_closed', false)
                    ->first();

                if (!$fiscalYear) {
                    throw new \Exception('فشل في إنشاء قيد اليومية التلقائي: لا توجد سنة مالية نشطة');
                }

                // التحقق من وجود فترة محاسبية نشطة
                $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
                    ->where('is_closed', false)
                    ->where('start_date', '<=', now())
                    ->where('end_date', '>=', now())
                    ->first();

                if (!$accountingPeriod) {
                    throw new \Exception('فشل في إنشاء قيد اليومية التلقائي: لا توجد فترة محاسبية نشطة للتاريخ الحالي');
                }

                // التحقق من وجود حساب مرتبط بالخزينة
                $cashRegister = CashRegister::findOrFail($request->cash_register_id);
                if (!$cashRegister->account) {
                    throw new \Exception('فشل في إنشاء قيد اليومية التلقائي: الخزينة غير مرتبطة بحساب في دليل الحسابات');
                }

                // إذا وصلنا إلى هنا، فهناك مشكلة أخرى غير معروفة
                throw new \Exception('فشل في إنشاء قيد اليومية التلقائي: يرجى مراجعة سجلات النظام للمزيد من التفاصيل');
            }

            DB::commit();

            // توجيه المستخدم إلى الصفحة المناسبة مع رسالة نجاح
            if ($request->type == 'deposit') {
                return redirect()->route('cash-deposits.index')
                    ->with('success', 'تم إنشاء الإيداع بنجاح وتم توليد قيد يومية تلقائي');
            } elseif ($request->type == 'withdrawal') {
                return redirect()->route('cash-withdrawals.index')
                    ->with('success', 'تم إنشاء السحب بنجاح وتم توليد قيد يومية تلقائي');
            } else {
                // حالة التحويل
                return redirect()->route('cash-transactions.index')
                    ->with('success', 'تم إنشاء التحويل بنجاح');
            }
        } catch (\Exception $e) {
            DB::rollBack();

            // حذف المرفق في حالة حدوث خطأ
            if ($attachmentPath) {
                Storage::disk('public')->delete($attachmentPath);
            }

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء المعاملة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * عرض تفاصيل معاملة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $transaction = CashTransaction::where('company_id', Auth::user()->company_id)
            ->with(['cashRegister', 'relatedCashRegister', 'user'])
            ->findOrFail($id);

        return view('cash_transactions.show', compact('transaction'));
    }

    /**
     * حذف معاملة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $transaction = CashTransaction::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // لا يمكن حذف المعاملات القديمة (أكثر من يوم مثلاً)
        if ($transaction->created_at->diffInDays(now()) > 1) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف المعاملات القديمة');
        }

        DB::beginTransaction();

        try {
            // استرجاع التغييرات في الرصيد
            $cashRegister = $transaction->cashRegister;

            switch ($transaction->type) {
                case 'deposit':
                    $cashRegister->current_balance -= $transaction->amount;
                    break;
                case 'withdrawal':
                    $cashRegister->current_balance += $transaction->amount;
                    break;
                case 'transfer':
                    $cashRegister->current_balance += $transaction->amount;

                    // استرجاع رصيد الخزينة المستلمة
                    $relatedCashRegister = $transaction->relatedCashRegister;
                    if ($relatedCashRegister) {
                        $relatedCashRegister->current_balance -= $transaction->amount;
                        $relatedCashRegister->save();
                    }
                    break;
            }

            $cashRegister->save();

            // حذف المرفق إذا وجد
            if ($transaction->attachment) {
                Storage::disk('public')->delete($transaction->attachment);
            }

            // حذف المعاملة
            $transaction->delete();

            DB::commit();

            return redirect()->route('cash-transactions.index')
                ->with('success', 'تم حذف المعاملة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف المعاملة: ' . $e->getMessage());
        }
    }

    /**
     * عرض نموذج إيداع
     *
     * @param  int  $cashRegisterId
     * @return \Illuminate\Http\Response
     */
    public function createDeposit($cashRegisterId)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($cashRegisterId);

        return view('cash_transactions.deposit', compact('cashRegister'));
    }

    /**
     * عرض نموذج سحب
     *
     * @param  int  $cashRegisterId
     * @return \Illuminate\Http\Response
     */
    public function createWithdrawal($cashRegisterId)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($cashRegisterId);

        return view('cash_transactions.withdrawal', compact('cashRegister'));
    }

    /**
     * إنشاء قيد يومية تلقائي للإيداع
     *
     * @param  \App\Models\CashTransaction  $transaction
     * @return \App\Models\JournalEntry|null
     */
    public function createJournalEntryForDeposit(CashTransaction $transaction)
    {
        try {
            // تسجيل بداية محاولة إنشاء قيد اليومية
            \Log::info('بدء محاولة إنشاء قيد يومية للإيداع رقم: ' . $transaction->id);

            // البحث عن السنة المالية والفترة المحاسبية النشطة
            $fiscalYear = FiscalYear::where('company_id', $transaction->company_id)
                ->where('is_active', true)
                ->where('is_closed', false)
                ->first();

            if (!$fiscalYear) {
                \Log::error('فشل إنشاء قيد يومية للإيداع: لا توجد سنة مالية نشطة للشركة رقم ' . $transaction->company_id);
                throw new \Exception('لا توجد سنة مالية نشطة');
            }

            // استخدام تاريخ الاستلام إذا كان موجودًا، وإلا استخدام التاريخ الحالي
            $entryDate = $transaction->receipt_date ?? now();

            $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
                ->where('is_closed', false)
                ->where('start_date', '<=', $entryDate)
                ->where('end_date', '>=', $entryDate)
                ->first();

            if (!$accountingPeriod) {
                \Log::error('فشل إنشاء قيد يومية للإيداع: لا توجد فترة محاسبية نشطة للتاريخ ' . $entryDate . ' في السنة المالية رقم ' . $fiscalYear->id);
                throw new \Exception('لا توجد فترة محاسبية نشطة للتاريخ المحدد');
            }

            // الحصول على حساب الخزينة المرتبط بالخزينة المحددة
            $cashAccount = $transaction->cashRegister->account;

            // إذا لم يكن هناك حساب مرتبط بالخزينة، نبحث عن حساب خزينة عام
            if (!$cashAccount) {
                \Log::warning('لا يوجد حساب مرتبط بالخزينة: ' . $transaction->cashRegister->name . ' (رقم ' . $transaction->cash_register_id . '). محاولة استخدام حساب خزينة عام.');

                $cashAccount = ChartOfAccount::where('company_id', $transaction->company_id)
                    ->where('is_active', true)
                    ->cash() // استخدام النطاق المعرف في النموذج
                    ->first();

                if (!$cashAccount) {
                    \Log::error('فشل إنشاء قيد يومية للإيداع: لا يوجد حساب خزينة معرف في دليل الحسابات للشركة رقم ' . $transaction->company_id);
                    throw new \Exception('لا يوجد حساب خزينة معرف في دليل الحسابات');
                }

                // تسجيل تحذير بأنه لا يوجد حساب مرتبط بالخزينة
                \Log::warning('لا يوجد حساب مرتبط بالخزينة: ' . $transaction->cashRegister->name . '. تم استخدام حساب خزينة عام رقم ' . $cashAccount->id);
            }

            // تسجيل معلومات عن الطرف الدائن المطلوب البحث عنه
            \Log::info('البحث عن حساب الطرف الدائن: ' . $transaction->party_name . ' (نوع الطرف: ' . $transaction->party_type . ')');

            // البحث عن حساب الطرف الدائن (الذي تم اختياره في النموذج)
            $creditAccount = ChartOfAccount::where('company_id', $transaction->company_id)
                ->where('is_active', true)
                ->where(function ($query) use ($transaction) {
                    $query->where('name', $transaction->party_name)
                        ->orWhere('name_en', $transaction->party_name)
                        ->orWhere('id', $transaction->party_name); // في حالة تم اختيار الحساب بالمعرف
                })
                ->first();

            if (!$creditAccount) {
                // تسجيل معلومات تصحيح الأخطاء
                \Log::error('فشل إنشاء قيد يومية للإيداع: لم يتم العثور على حساب للطرف الدائن: ' . $transaction->party_name);

                throw new \Exception('لا يوجد حساب للطرف الدائن في دليل الحسابات: ' . $transaction->party_name);
            }

            // تسجيل معلومات الحساب الذي تم العثور عليه
            \Log::info('تم العثور على حساب الطرف الدائن: ' . $creditAccount->name . ' (رقم ' . $creditAccount->id . ')');


            // إنشاء القيد المحاسبي
            $entry = new JournalEntry();
            $entry->company_id = $transaction->company_id;
            $entry->fiscal_year_id = $fiscalYear->id;
            $entry->accounting_period_id = $accountingPeriod->id;
            $entry->entry_number = JournalEntry::generateEntryNumber($transaction->company_id);
            $entry->entry_date = $entryDate;
            $entry->reference_type = 'App\\Models\\CashTransaction';
            $entry->reference_id = $transaction->id;
            $entry->description = 'قيد إيداع خزينة تلقائي - ' . $transaction->cashRegister->name;
            $entry->is_posted = true; // ترحيل القيد مباشرة
            $entry->is_approved = false; // لا يتم اعتماده تلقائيًا
            $entry->created_by = $transaction->user_id;
            $entry->save();

            // إنشاء بنود القيد المحاسبي

            // البند المدين (حساب الخزينة)
            $debitItem = new JournalEntryItem();
            $debitItem->journal_entry_id = $entry->id;
            $debitItem->account_id = $cashAccount->id;
            $debitItem->debit = $transaction->amount;
            $debitItem->credit = 0;
            $debitItem->description = 'إيداع في خزينة ' . $transaction->cashRegister->name;
            $debitItem->save();

            // البند الدائن (حساب الطرف الدائن)
            $creditItem = new JournalEntryItem();
            $creditItem->journal_entry_id = $entry->id;
            $creditItem->account_id = $creditAccount->id;
            $creditItem->debit = 0;
            $creditItem->credit = $transaction->amount;
            $creditItem->description = 'إيداع من ' . $transaction->party_name;
            $creditItem->save();

            // تحديث أرصدة الحسابات
            $cashAccount->updateBalance();
            $creditAccount->updateBalance();

            // إنشاء سجل في جدول الربط بين المعاملة والقيد المحاسبي
            DB::table('cash_transaction_entries')->insert([
                'cash_transaction_id' => $transaction->id,
                'journal_entry_id' => $entry->id,
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return $entry;
        } catch (\Exception $e) {
            // تسجيل الخطأ ولكن لا نوقف العملية
            \Log::error('خطأ في إنشاء قيد يومية تلقائي للإيداع: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * إنشاء قيد يومية تلقائي للسحب
     *
     * @param CashTransaction $transaction
     * @return JournalEntry|null
     */
    public function createJournalEntryForWithdrawal(CashTransaction $transaction)
    {
        try {
            // البحث عن السنة المالية والفترة المحاسبية النشطة
            $fiscalYear = FiscalYear::where('company_id', $transaction->company_id)
                ->where('is_active', true)
                ->where('is_closed', false)
                ->first();

            if (!$fiscalYear) {
                throw new \Exception('لا توجد سنة مالية نشطة');
            }

            $entryDate = now();

            $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
                ->where('is_closed', false)
                ->where('start_date', '<=', $entryDate)
                ->where('end_date', '>=', $entryDate)
                ->first();

            if (!$accountingPeriod) {
                throw new \Exception('لا توجد فترة محاسبية نشطة للتاريخ المحدد');
            }

            // الحصول على حساب الخزينة المرتبط بالخزينة المحددة
            $cashAccount = $transaction->cashRegister->account;

            // إذا لم يكن هناك حساب مرتبط بالخزينة، نبحث عن حساب خزينة عام
            if (!$cashAccount) {
                $cashAccount = ChartOfAccount::where('company_id', $transaction->company_id)
                    ->where('is_active', true)
                    ->where('is_cash_account', true)
                    ->first();

                if (!$cashAccount) {
                    // إذا لم يتم العثور على حساب بالخاصية is_cash_account، نبحث عن حساب النقدية والبنوك بالاسم
                    $cashAccount = ChartOfAccount::where('company_id', $transaction->company_id)
                        ->where('is_active', true)
                        ->where('name', 'النقدية والبنوك')
                        ->orWhere('name_en', 'Cash and Banks')
                        ->first();
                }

                if (!$cashAccount) {
                    throw new \Exception('لا يوجد حساب خزينة معرف في دليل الحسابات');
                }

                // تسجيل تحذير بأنه لا يوجد حساب مرتبط بالخزينة
                \Log::warning('لا يوجد حساب مرتبط بالخزينة: ' . $transaction->cashRegister->name . '. تم استخدام حساب خزينة عام.');
            }

            // البحث عن حساب الطرف المدين (الذي تم اختياره في النموذج)
            $debitAccount = ChartOfAccount::where('company_id', $transaction->company_id)
                ->where('is_active', true)
                ->where(function ($query) use ($transaction) {
                    $query->where('name', $transaction->party_name)
                        ->orWhere('name_en', $transaction->party_name)
                        ->orWhere('id', $transaction->party_name); // في حالة تم اختيار الحساب بالمعرف
                })
                ->first();

            if (!$debitAccount) {
                // تسجيل معلومات تصحيح الأخطاء
                \Log::warning('لم يتم العثور على حساب للطرف المدين: ' . $transaction->party_name);

                throw new \Exception('لا يوجد حساب للطرف المدين في دليل الحسابات: ' . $transaction->party_name);
            }

            // إنشاء القيد المحاسبي
            $entry = new JournalEntry();
            $entry->company_id = $transaction->company_id;
            $entry->fiscal_year_id = $fiscalYear->id;
            $entry->accounting_period_id = $accountingPeriod->id;
            $entry->entry_number = JournalEntry::generateEntryNumber($transaction->company_id);
            // استخدام تاريخ الصرف إذا كان موجودًا، وإلا استخدام التاريخ الحالي
            $entry->entry_date = $transaction->withdrawal_date ?? $entryDate;
            $entry->reference_type = 'App\\Models\\CashTransaction';
            $entry->reference_id = $transaction->id;
            $entry->description = 'قيد سحب خزينة تلقائي - ' . $transaction->cashRegister->name;
            $entry->is_posted = true; // ترحيل القيد مباشرة
            $entry->is_approved = false; // لا يتم اعتماده تلقائيًا
            $entry->created_by = $transaction->user_id;
            $entry->save();

            // إنشاء بنود القيد المحاسبي

            // البند المدين (حساب الطرف المدين)
            $debitItem = new JournalEntryItem();
            $debitItem->journal_entry_id = $entry->id;
            $debitItem->account_id = $debitAccount->id;
            $debitItem->debit = $transaction->amount;
            $debitItem->credit = 0;
            $debitItem->description = 'سحب لصالح ' . $transaction->party_name;
            $debitItem->save();

            // البند الدائن (حساب الخزينة)
            $creditItem = new JournalEntryItem();
            $creditItem->journal_entry_id = $entry->id;
            $creditItem->account_id = $cashAccount->id;
            $creditItem->debit = 0;
            $creditItem->credit = $transaction->amount;
            $creditItem->description = 'سحب من خزينة ' . $transaction->cashRegister->name;
            $creditItem->save();

            // تحديث أرصدة الحسابات
            $debitAccount->updateBalance();
            $cashAccount->updateBalance();

            // إنشاء سجل في جدول الربط بين المعاملة والقيد المحاسبي
            DB::table('cash_transaction_entries')->insert([
                'cash_transaction_id' => $transaction->id,
                'journal_entry_id' => $entry->id,
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return $entry;
        } catch (\Exception $e) {
            // تسجيل الخطأ ولكن لا نوقف العملية
            \Log::error('خطأ في إنشاء قيد يومية تلقائي للسحب: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * عرض نموذج تحويل
     *
     * @param  int  $cashRegisterId
     * @return \Illuminate\Http\Response
     */
    public function createTransfer($cashRegisterId)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($cashRegisterId);

        $targetCashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->where('id', '!=', $cashRegisterId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('cash_transactions.transfer', compact('cashRegister', 'targetCashRegisters'));
    }
}
