<?php

namespace App\Http\Controllers;

use App\Models\Branch;
use App\Models\CashRegister;
use App\Models\User;
use App\Services\CashRegisterService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class CashRegisterController extends Controller
{
    /**
     * خدمة إدارة الخزائن
     *
     * @var CashRegisterService
     */
    protected $cashRegisterService;

    /**
     * إنشاء مثيل جديد من وحدة التحكم
     *
     * @param CashRegisterService $cashRegisterService
     * @return void
     */
    public function __construct(CashRegisterService $cashRegisterService)
    {
        $this->cashRegisterService = $cashRegisterService;
    }
    /**
     * عرض قائمة الخزائن
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $cashRegisters = CashRegister::where('company_id', Auth::user()->company_id)
            ->with(['branch', 'responsibleUser'])
            ->orderBy('name')
            ->paginate(15);

        return view('cash_registers.index', compact('cashRegisters'));
    }

    /**
     * عرض نموذج إنشاء خزينة جديدة
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $branches = Branch::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        $users = User::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        return view('cash_registers.create', compact('branches', 'users'));
    }

    /**
     * حفظ خزينة جديدة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => [
                'required',
                'string',
                'max:50',
                function ($attribute, $value, $fail) {
                    // التحقق من فرادة الكود على مستوى الشركة فقط
                    $exists = CashRegister::where('code', $value)
                        ->where('company_id', Auth::user()->company_id)
                        ->exists();

                    if ($exists) {
                        $fail('قيمة code مُستخدمة من قبل في نفس الشركة.');
                    }
                }
            ],
            'description' => 'nullable|string',
            'branch_id' => 'nullable|exists:branches,id',
            'responsible_user_id' => 'nullable|exists:users,id',
            'initial_balance' => 'required|numeric|min:0',
            'currency' => 'required|string|max:10',
            'is_active' => 'boolean',
            'is_main' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // تحويل القيم المنطقية
        $data = $request->all();
        $data['is_active'] = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_main'] = filter_var($request->input('is_main', false), FILTER_VALIDATE_BOOLEAN);
        $data['company_id'] = Auth::user()->company_id;
        $data['current_balance'] = $request->initial_balance;

        // إذا كانت هذه الخزينة الرئيسية، نقوم بإلغاء تعيين الخزينة الرئيسية السابقة
        if ($data['is_main']) {
            CashRegister::where('company_id', Auth::user()->company_id)
                ->where('is_main', true)
                ->update(['is_main' => false]);
        }

        $cashRegister = CashRegister::create($data);

        try {
            // إنشاء حساب في شجرة الحسابات للخزينة
            $this->cashRegisterService->createOrUpdateCashRegisterAccount($cashRegister);

            return redirect()->route('cash-registers.index')
                ->with('success', 'تم إنشاء الخزينة وحسابها في شجرة الحسابات بنجاح');
        } catch (\Exception $e) {
            Log::error('خطأ في إنشاء حساب الخزينة: ' . $e->getMessage());

            return redirect()->route('cash-registers.index')
                ->with('warning', 'تم إنشاء الخزينة بنجاح ولكن حدث خطأ في إنشاء الحساب المرتبط: ' . $e->getMessage());
        }
    }

    /**
     * عرض تفاصيل خزينة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->with(['branch', 'responsibleUser', 'transactions' => function ($query) {
                $query->latest()->take(15);
            }])
            ->findOrFail($id);

        return view('cash_registers.show', compact('cashRegister'));
    }

    /**
     * عرض نموذج تعديل خزينة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $branches = Branch::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        $users = User::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        return view('cash_registers.edit', compact('cashRegister', 'branches', 'users'));
    }

    /**
     * تحديث خزينة محددة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => [
                'required',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($id, $cashRegister) {
                    // التحقق من فرادة الكود على مستوى الشركة فقط
                    $exists = CashRegister::where('code', $value)
                        ->where('company_id', Auth::user()->company_id)
                        ->where('id', '!=', $id)
                        ->exists();

                    if ($exists) {
                        $fail('قيمة code مُستخدمة من قبل في نفس الشركة.');
                    }
                }
            ],
            'description' => 'nullable|string',
            'branch_id' => 'nullable|exists:branches,id',
            'responsible_user_id' => 'nullable|exists:users,id',
            'currency' => 'required|string|max:10',
            'is_active' => 'boolean',
            'is_main' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // تحويل القيم المنطقية
        $data = $request->all();
        $data['is_active'] = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
        $data['is_main'] = filter_var($request->input('is_main', false), FILTER_VALIDATE_BOOLEAN);

        // إذا كانت هذه الخزينة الرئيسية، نقوم بإلغاء تعيين الخزينة الرئيسية السابقة
        if ($data['is_main'] && !$cashRegister->is_main) {
            CashRegister::where('company_id', Auth::user()->company_id)
                ->where('is_main', true)
                ->update(['is_main' => false]);
        }

        $cashRegister->update($data);

        try {
            // تحديث حساب الخزينة في شجرة الحسابات
            $this->cashRegisterService->createOrUpdateCashRegisterAccount($cashRegister);

            return redirect()->route('cash-registers.index')
                ->with('success', 'تم تحديث الخزينة وحسابها في شجرة الحسابات بنجاح');
        } catch (\Exception $e) {
            Log::error('خطأ في تحديث حساب الخزينة: ' . $e->getMessage());

            return redirect()->route('cash-registers.index')
                ->with('warning', 'تم تحديث الخزينة بنجاح ولكن حدث خطأ في تحديث الحساب المرتبط: ' . $e->getMessage());
        }
    }

    /**
     * حذف خزينة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $cashRegister = CashRegister::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من عدم وجود معاملات مرتبطة بالخزينة
        if ($cashRegister->transactions()->count() > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف الخزينة لوجود معاملات مرتبطة بها');
        }

        $cashRegister->delete();

        return redirect()->route('cash-registers.index')
            ->with('success', 'تم حذف الخزينة بنجاح');
    }
}
