<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Branch;
use App\Models\Company;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;

class BranchController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();

        // السماح للمستخدم بعرض جميع الفروع إذا كان مسؤول نظام
        if ($user->isSystemAdmin()) {
            $branches = Branch::orderBy('name')->paginate(10);
            return view('branches.index', compact('branches'));
        }

        $company = $user->company;
        if (!$company) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك شركة مرتبطة بحسابك');
        }

        $branches = Branch::where('company_id', $company->id)
            ->orderBy('name')
            ->paginate(10);

        return view('branches.index', compact('branches', 'company'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();

        // السماح للمستخدم بإنشاء فرع حتى لو لم يكن مرتبطاً بشركة (مثل السوبر أدمن)
        if ($user->isSystemAdmin()) {
            $companies = Company::all();
            return view('branches.create_alt', compact('companies'));
        }

        $company = $user->company;
        if (!$company) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك شركة مرتبطة بحسابك');
        }

        return view('branches.create_alt', compact('company'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        // تحقق من وجود شركة في الطلب (في حالة السوبر أدمن)
        $companyId = $request->company_id ?? ($company ? $company->id : null);

        if (!$companyId) {
            return redirect()->route('companies.index')
                ->with('error', 'يجب اختيار شركة أولاً.');
        }

        $validator = Validator::make($request->all(), [
            'company_id' => 'sometimes|required|exists:companies,id',
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'code' => [
                'nullable',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($companyId) {
                    // التحقق من فرادة الكود على مستوى الشركة
                    if (!empty($value)) {
                        $exists = Branch::where('company_id', $companyId)
                            ->where('code', $value)
                            ->exists();

                        if ($exists) {
                            $fail('رمز الفرع مستخدم بالفعل في هذه الشركة. الرجاء استخدام رمز آخر.');
                        }
                    }
                },
            ],
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'mobile' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'address_en' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',
            'manager_name' => 'nullable|string|max:255',
            'is_main_branch' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            // Create branch
            $branch = new Branch();
            $branch->company_id = $companyId;
            $branch->name = $request->name;
            $branch->name_en = $request->name_en;

            // تعيين رمز الفرع إذا كان موجودًا
            if ($request->has('code') && !empty($request->code)) {
                $branch->code = $request->code;
            } else {
                // إنشاء رمز افتراضي إذا كان الحقل موجودًا في الجدول
                if (Schema::hasColumn('branches', 'code')) {
                    $branch->code = 'BR-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
                }
            }

            $branch->email = $request->email;
            $branch->phone = $request->phone;
            $branch->mobile = $request->mobile;
            $branch->address = $request->address;
            $branch->address_en = $request->address_en;
            $branch->city = $request->city;
            $branch->state = $request->state;
            $branch->country = $request->country;
            $branch->postal_code = $request->postal_code;
            $branch->location_lat = $request->location_lat;
            $branch->location_lng = $request->location_lng;

            // الآن يمكننا استخدام الحقول مباشرة بعد إضافتها إلى قاعدة البيانات
            $branch->manager_name = $request->manager_name;
            $branch->is_main_branch = $request->has('is_main_branch');
            $branch->is_active = true;
            $branch->notes = $request->notes;

            $branch->save();

            return redirect()->route('branches.show', $branch->id)
                ->with('success', 'تم إنشاء الفرع بنجاح.');
        } catch (\Exception $e) {
            // تسجيل الخطأ للمطورين
            \Log::error('خطأ في إنشاء فرع جديد: ' . $e->getMessage());

            // التحقق من نوع الخطأ وإرجاع رسالة مناسبة
            if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                if (strpos($e->getMessage(), 'branches_company_id_code_unique') !== false) {
                    return redirect()->back()
                        ->with('error', 'رمز الفرع مستخدم بالفعل في هذه الشركة. الرجاء استخدام رمز آخر.')
                        ->withInput();
                } elseif (strpos($e->getMessage(), 'branches_code_unique') !== false) {
                    return redirect()->back()
                        ->with('error', 'رمز الفرع مستخدم بالفعل. الرجاء استخدام رمز آخر.')
                        ->withInput();
                }
            }

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء الفرع. الرجاء التحقق من البيانات المدخلة وحاول مرة أخرى.')
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $branch = Branch::findOrFail($id);

        // إذا كان المستخدم مرتبط بشركة، تحقق من أن الفرع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company || $branch->company_id != $company->id) {
                return redirect()->route('branches.index')
                    ->with('error', 'ليس لديك صلاحية لعرض هذا الفرع.');
            }
        }

        // Get branch statistics
        $statistics = [
            'users_count' => $branch->users()->count(),
            // El método customers() no existe, usamos customer_statements() en su lugar o 0 si no hay relación
            'customers_count' => method_exists($branch, 'customer_statements') ? $branch->customer_statements()->count() : 0,
            'elevators_count' => $branch->elevators()->count(),
            // El método maintenanceContracts() no existe, usamos maintenance_contracts() en su lugar
            'contracts_count' => $branch->maintenance_contracts()->count(),
            'invoices_count' => $branch->invoices()->count(),
        ];

        return view('branches.show', compact('branch', 'statistics'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $branch = Branch::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company || $branch->company_id != $company->id) {
                return redirect()->route('branches.index')
                    ->with('error', 'ليس لديك صلاحية لتعديل هذا الفرع.');
            }
            return view('branches.edit', compact('branch'));
        }

        // إذا كان مسؤول نظام، قم بتمرير قائمة الشركات
        $companies = Company::all();
        return view('branches.edit', compact('branch', 'companies'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $branch = Branch::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company || $branch->company_id != $company->id) {
                return redirect()->route('branches.index')
                    ->with('error', 'ليس لديك صلاحية لتحديث هذا الفرع.');
            }
        }

        // تحديث معرف الشركة إذا تم توفيره وكان المستخدم مسؤول نظام
        if ($user->isSystemAdmin() && isset($request->company_id)) {
            $branch->company_id = $request->company_id;
        }

        $validator = Validator::make($request->all(), [
            'company_id' => 'sometimes|required|exists:companies,id',
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'code' => [
                'required',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($branch) {
                    // التحقق من فرادة الكود على مستوى الشركة
                    if (!empty($value)) {
                        $exists = Branch::where('company_id', $branch->company_id)
                            ->where('code', $value)
                            ->where('id', '!=', $branch->id)
                            ->exists();

                        if ($exists) {
                            $fail('رمز الفرع مستخدم بالفعل في هذه الشركة. الرجاء استخدام رمز آخر.');
                        }
                    }
                },
            ],
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'mobile' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'address_en' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',
            'manager_name' => 'nullable|string|max:255',
            'is_main_branch' => 'boolean',
            'is_active' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Update branch
        $branch->name = $request->name;
        $branch->name_en = $request->name_en;
        $branch->code = $request->code;
        $branch->email = $request->email;
        $branch->phone = $request->phone;
        $branch->mobile = $request->mobile;
        $branch->address = $request->address;
        $branch->address_en = $request->address_en;
        $branch->city = $request->city;
        $branch->state = $request->state;
        $branch->country = $request->country;
        $branch->postal_code = $request->postal_code;
        $branch->location_lat = $request->location_lat;
        $branch->location_lng = $request->location_lng;
        $branch->manager_name = $request->manager_name;
        $branch->is_main_branch = $request->has('is_main_branch');
        $branch->is_active = $request->has('is_active');
        $branch->notes = $request->notes;
        $branch->save();

        return redirect()->route('branches.show', $branch->id)
            ->with('success', 'تم تحديث الفرع بنجاح.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $branch = Branch::findOrFail($id);

        // التحقق من الصلاحيات
        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company || $branch->company_id != $company->id) {
                return redirect()->route('branches.index')
                    ->with('error', 'ليس لديك صلاحية لحذف هذا الفرع.');
            }
        }

        // Check if this is the main branch
        if ($branch->is_main_branch) {
            return redirect()->route('branches.index')
                ->with('error', 'لا يمكن حذف الفرع الرئيسي.');
        }

        // Check if branch has related records
        $hasRelatedRecords =
            $branch->users()->count() > 0 ||
            // El método customers() no existe, usamos customer_statements() en su lugar o 0 si no hay relación
            (method_exists($branch, 'customer_statements') ? $branch->customer_statements()->count() > 0 : false) ||
            $branch->elevators()->count() > 0 ||
            // El método maintenanceContracts() no existe, usamos maintenance_contracts() en su lugar
            $branch->maintenance_contracts()->count() > 0 ||
            $branch->invoices()->count() > 0;

        if ($hasRelatedRecords) {
            return redirect()->route('branches.index')
                ->with('error', 'لا يمكن حذف هذا الفرع لأنه يحتوي على سجلات مرتبطة به. يمكنك تعطيله بدلاً من ذلك.');
        }

        // Delete branch
        $branch->delete();

        return redirect()->route('branches.index')
            ->with('success', 'تم حذف الفرع بنجاح.');
    }
}
