<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem;
use App\Models\InventorySerial;
use App\Models\InventoryBatch;
use App\Models\InventoryTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Milon\Barcode\DNS1D;
use Milon\Barcode\DNS2D;

class BarcodeController extends Controller
{
    /**
     * عرض صفحة إدارة الباركود
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        return view('inventory.barcodes.index');
    }

    /**
     * طباعة باركود/QR Code لعنصر مخزون
     */
    public function printItemCode(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $item = InventoryItem::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        $quantity = $request->input('quantity', 1);
        $type = $request->input('type', 'qrcode'); // qrcode, barcode
        $size = $request->input('size', 'medium'); // small, medium, large

        return view('inventory.barcodes.print_item', compact('item', 'quantity', 'type', 'size'));
    }

    /**
     * طباعة باركود/QR Code لمجموعة من العناصر
     */
    public function printMultipleItems(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $itemIds = $request->input('items', []);
        $items = InventoryItem::where('company_id', $company->id)
            ->whereIn('id', $itemIds)
            ->get();

        $type = $request->input('type', 'qrcode');
        $size = $request->input('size', 'medium');

        return view('inventory.barcodes.print_multiple', compact('items', 'type', 'size'));
    }

    /**
     * طباعة باركود/QR Code للأرقام التسلسلية
     */
    public function printSerialCodes(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $serialIds = $request->input('serials', []);
        $serials = InventorySerial::whereIn('id', $serialIds)
            ->whereHas('item', function ($query) use ($company) {
                $query->where('company_id', $company->id);
            })
            ->with('item')
            ->get();

        return view('inventory.barcodes.print_serials', compact('serials'));
    }

    /**
     * مسح الباركود/QR Code
     */
    public function scan()
    {
        return view('inventory.barcodes.scan');
    }

    /**
     * معالجة نتيجة المسح
     */
    public function processScan(Request $request)
    {
        $code = $request->input('code');

        // محاولة فك تشفير البيانات
        try {
            $data = json_decode($code);

            if (isset($data->id)) {
                // إذا كان QR Code لعنصر مخزون
                return redirect()->route('inventory.items.show', $data->id);
            } elseif (isset($data->serial_id)) {
                // إذا كان QR Code لرقم تسلسلي
                return redirect()->route('inventory.serials.show', $data->serial_id);
            } elseif (isset($data->batch_id)) {
                // إذا كان QR Code لدفعة
                return redirect()->route('inventory.batches.show', $data->batch_id);
            }
        } catch (\Exception $e) {
            // إذا كان باركود عادي، ابحث عنه في قاعدة البيانات
            $item = InventoryItem::where('barcode', $code)->first();

            if ($item) {
                return redirect()->route('inventory.items.show', $item->id);
            }

            $serial = InventorySerial::where('serial_number', $code)->first();

            if ($serial) {
                return redirect()->route('inventory.serials.show', $serial->id);
            }
        }

        return redirect()->back()->with('error', 'لم يتم العثور على عنصر مطابق للرمز المسحوب');
    }

    /**
     * إنشاء باركود/QR Code للمخزون الجديد
     */
    public function generateForNewStock(Request $request, $transactionId)
    {
        $user = Auth::user();
        $company = $user->company;

        $transaction = InventoryTransaction::where('id', $transactionId)
            ->where(function ($query) use ($company) {
                $query->whereHas('item', function ($q) use ($company) {
                    $q->where('company_id', $company->id);
                });
            })
            ->firstOrFail();

        // الحصول على العناصر والأرقام التسلسلية المرتبطة بالمعاملة
        $serials = InventorySerial::where('transaction_id', $transaction->id)->get();
        $batches = InventoryBatch::where('transaction_id', $transaction->id)->get();

        return view('inventory.barcodes.new_stock', compact('transaction', 'serials', 'batches'));
    }
}
