<?php

namespace App\Http\Controllers;

use App\Models\Bank;
use App\Models\BankAccount;
use App\Models\BankTransaction;
use App\Models\CashRegister;
use App\Models\ChartOfAccount;
use App\Models\JournalEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class BankTransactionController extends Controller
{
    /**
     * Display a listing of the bank transactions.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->authorize('view_bank_transactions');

        // Filter by current company
        $query = BankTransaction::with(['bankAccount', 'bankAccount.bank'])
            ->whereHas('bankAccount', function ($q) {
                $q->where('company_id', auth()->user()->company_id);
            });

        // Apply filters if provided
        if ($request->has('account_id') && $request->account_id) {
            $query->where('bank_account_id', $request->account_id);
        }

        if ($request->has('type') && $request->type) {
            $query->where('transaction_type', $request->type);
        }

        if ($request->has('start_date') && $request->start_date) {
            $query->where('transaction_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->where('transaction_date', '<=', $request->end_date);
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(15);

        // Filter bank accounts by current company
        $bankAccounts = BankAccount::where('company_id', auth()->user()->company_id)
            ->where('is_active', true)
            ->get();

        return view('banking.transactions.index', compact('transactions', 'bankAccounts'));
    }

    /**
     * Show the form for creating a new deposit transaction.
     *
     * @return \Illuminate\Http\Response
     */
    public function createDeposit()
    {
        $this->authorize('create_bank_transactions');

        $companyId = auth()->user()->company_id;

        // Filter by current company
        $bankAccounts = BankAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $cashRegisters = CashRegister::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        // تحميل جميع الحسابات من شجرة الحسابات باستثناء حسابات البنوك
        $chartAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('is_bank_account', false)
            ->where('is_active', true)
            ->orderBy('account_code')
            ->get();

        $customers = \App\Models\Customer::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $invoices = \App\Models\Invoice::where('company_id', $companyId)
            ->where('status', 'unpaid')
            ->get();
        $elevators = \App\Models\Elevator::where('company_id', $companyId)->get();

        return view('banking.transactions.create_deposit', compact('bankAccounts', 'cashRegisters', 'chartAccounts', 'customers', 'invoices', 'elevators'));
    }

    /**
     * Show the form for creating a new withdrawal transaction.
     *
     * @return \Illuminate\Http\Response
     */
    public function createWithdrawal()
    {
        $this->authorize('create_bank_transactions');

        $companyId = auth()->user()->company_id;

        // Filter by current company
        $bankAccounts = BankAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $chartAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', '!=', 'bank')
            ->get();

        return view('banking.transactions.create_withdrawal', compact('bankAccounts', 'chartAccounts'));
    }

    /**
     * Show the form for creating a new transfer transaction.
     *
     * @return \Illuminate\Http\Response
     */
    public function createTransfer()
    {
        $this->authorize('create_bank_transactions');

        // Filter by current company
        $bankAccounts = BankAccount::where('company_id', auth()->user()->company_id)
            ->where('is_active', true)
            ->get();

        return view('banking.transactions.create_transfer', compact('bankAccounts'));
    }

    /**
     * Store a newly created deposit transaction in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeDeposit(Request $request)
    {
        $this->authorize('create_bank_transactions');

        $validated = $request->validate([
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'transaction_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'reference_number' => 'nullable|string|max:50',
            'description' => 'required|string',
            'deposit_method' => 'required|in:cash,cheque,transfer,other',
            'source' => 'required|in:chart_account',
            'customer_id' => 'nullable|exists:customers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'cash_register_id' => 'nullable|exists:cash_registers,id',
            'chart_of_account_id' => 'required|exists:chart_of_accounts,id',
            'elevator_id' => 'nullable|exists:elevators,id',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        try {
            DB::beginTransaction();

            $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);

            // Security check: Ensure the bank account belongs to the current company
            if ($bankAccount->company_id !== auth()->user()->company_id) {
                abort(403, 'Unauthorized action.');
            }

            // Create the bank transaction
            $transaction = new BankTransaction();
            $transaction->bank_account_id = $validated['bank_account_id'];
            $transaction->transaction_type = 'deposit';
            $transaction->transaction_date = $validated['transaction_date'];
            $transaction->amount = $validated['amount'];
            $transaction->reference_number = $validated['reference_number'] ?? null;
            $transaction->description = $validated['description'];
            $transaction->deposit_method = $validated['deposit_method'];
            $transaction->source = $validated['source'];
            $transaction->created_by = auth()->id();

            // حفظ معلومات إضافية حسب المصدر
            $additionalData = [];

            if ($validated['source'] == 'customer' && isset($validated['customer_id'])) {
                $additionalData['customer_id'] = $validated['customer_id'];
                if (isset($validated['invoice_id'])) {
                    $additionalData['invoice_id'] = $validated['invoice_id'];
                }
            }

            // حفظ رقم المصعد إذا تم تحديده
            if (isset($validated['elevator_id'])) {
                $additionalData['elevator_id'] = $validated['elevator_id'];
            }

            if (!empty($additionalData)) {
                $transaction->additional_data = json_encode($additionalData);
            }

            // Handle attachment if provided
            if ($request->hasFile('attachment')) {
                $path = $request->file('attachment')->store('bank_transactions', 'public');
                $transaction->attachment = $path;
            }

            $transaction->save();

            // Update bank account balance
            $bankAccount->current_balance += $validated['amount'];
            $bankAccount->save();

            // الحصول على الحساب المصدر (الدائن)
            $sourceAccount = ChartOfAccount::findOrFail($validated['chart_of_account_id']);

            // حفظ معلومات الحساب في البيانات الإضافية
            $additionalData = json_decode($transaction->additional_data ?? '{}', true);
            $additionalData['chart_of_account_id'] = $sourceAccount->id;
            $transaction->additional_data = json_encode($additionalData);
            $transaction->save();

            // إنشاء قيد محاسبي للإيداع البنكي
            $this->createJournalEntryForDeposit(
                $transaction,
                $bankAccount,
                $sourceAccount,
                $validated['transaction_date'],
                $validated['amount'],
                $validated['description'],
                $validated['reference_number'] ?? null
            );

            // Log activity
            Log::info('Bank deposit created', [
                'transaction_id' => $transaction->id,
                'user_id' => auth()->id(),
                'amount' => $validated['amount'],
                'performed_by' => auth()->user()->name
            ]);

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم إنشاء الإيداع البنكي بنجاح وتسجيل القيد المحاسبي.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating bank deposit: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء الإيداع البنكي: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء قيد محاسبي للإيداع البنكي
     *
     * @param BankTransaction $transaction معاملة الإيداع البنكي
     * @param BankAccount $bankAccount الحساب البنكي (المدين)
     * @param ChartOfAccount $sourceAccount حساب المصدر (الدائن)
     * @param string $transactionDate تاريخ المعاملة
     * @param float $amount المبلغ
     * @param string $description الوصف
     * @param string|null $referenceNumber رقم المرجع
     * @return JournalEntry
     */
    private function createJournalEntryForDeposit(
        BankTransaction $transaction,
        BankAccount $bankAccount,
        ChartOfAccount $sourceAccount,
        $transactionDate,
        $amount,
        $description,
        $referenceNumber = null
    ) {
        // الحصول على الشركة الحالية
        $companyId = auth()->user()->company_id;
        $company = \App\Models\Company::findOrFail($companyId);

        // البحث عن السنة المالية النشطة
        $fiscalYear = \App\Models\FiscalYear::where('company_id', $companyId)
            ->where('is_active', true)
            ->first();

        if (!$fiscalYear) {
            throw new \Exception('لا توجد سنة مالية نشطة للشركة');
        }

        // البحث عن الفترة المحاسبية المناسبة للتاريخ
        $accountingPeriod = \App\Models\AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
            ->where('start_date', '<=', $transactionDate)
            ->where('end_date', '>=', $transactionDate)
            ->first();

        if (!$accountingPeriod) {
            throw new \Exception('لا توجد فترة محاسبية نشطة للتاريخ المحدد: ' . $transactionDate);
        }

        // إنشاء القيد المحاسبي
        $journalEntry = new JournalEntry();
        $journalEntry->company_id = $companyId;
        $journalEntry->fiscal_year_id = $fiscalYear->id;
        $journalEntry->accounting_period_id = $accountingPeriod->id;
        $journalEntry->entry_number = 'BD-' . $transaction->id; // رقم الإيداع البنكي
        $journalEntry->entry_date = $transactionDate;
        $journalEntry->description = 'إيداع بنكي: ' . $description;
        $journalEntry->reference_type = 'App\\Models\\BankTransaction';
        $journalEntry->reference_id = $transaction->id;
        $journalEntry->is_posted = true;
        $journalEntry->posted_by = auth()->id();
        $journalEntry->posted_at = now();
        $journalEntry->created_by = auth()->id();
        $journalEntry->save();

        // إضافة بنود القيد المحاسبي

        // الطرف المدين: الحساب البنكي (زيادة في الأصول)
        $journalEntry->items()->create([
            'account_id' => $bankAccount->chart_account_id,
            'description' => 'إيداع بنكي إلى ' . $bankAccount->account_name,
            'debit' => $amount,
            'credit' => 0,
        ]);

        // الطرف الدائن: حساب المصدر
        $journalEntry->items()->create([
            'account_id' => $sourceAccount->id,
            'description' => 'إيداع بنكي من حساب: ' . $sourceAccount->name,
            'debit' => 0,
            'credit' => $amount,
        ]);

        // ربط المعاملة بالقيد المحاسبي
        $transaction->journal_entry_id = $journalEntry->id;
        $transaction->save();

        // تسجيل معلومات إنشاء القيد المحاسبي
        Log::info('تم إنشاء قيد محاسبي للإيداع البنكي', [
            'transaction_id' => $transaction->id,
            'journal_entry_id' => $journalEntry->id,
            'amount' => $amount,
            'bank_account' => $bankAccount->account_name,
            'source_account' => $sourceAccount->name
        ]);

        return $journalEntry;
    }

    /**
     * إنشاء قيد محاسبي للسحب البنكي
     *
     * @param BankTransaction $transaction معاملة السحب البنكي
     * @param BankAccount $bankAccount الحساب البنكي (الدائن)
     * @param ChartOfAccount $debitAccount الحساب المدين
     * @param string $transactionDate تاريخ المعاملة
     * @param float $amount المبلغ
     * @param string $description الوصف
     * @param string|null $referenceNumber رقم المرجع
     * @return JournalEntry
     */
    private function createJournalEntryForWithdrawal(
        BankTransaction $transaction,
        BankAccount $bankAccount,
        ChartOfAccount $debitAccount,
        $transactionDate,
        $amount,
        $description,
        $referenceNumber = null
    ) {
        // الحصول على الشركة الحالية
        $companyId = auth()->user()->company_id;
        $company = \App\Models\Company::findOrFail($companyId);

        // البحث عن السنة المالية النشطة
        $fiscalYear = \App\Models\FiscalYear::where('company_id', $companyId)
            ->where('is_active', true)
            ->first();

        if (!$fiscalYear) {
            throw new \Exception('لا توجد سنة مالية نشطة للشركة');
        }

        // البحث عن الفترة المحاسبية المناسبة للتاريخ
        $accountingPeriod = \App\Models\AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
            ->where('start_date', '<=', $transactionDate)
            ->where('end_date', '>=', $transactionDate)
            ->first();

        if (!$accountingPeriod) {
            throw new \Exception('لا توجد فترة محاسبية نشطة للتاريخ المحدد: ' . $transactionDate);
        }

        // إنشاء القيد المحاسبي
        $journalEntry = new JournalEntry();
        $journalEntry->company_id = $companyId;
        $journalEntry->fiscal_year_id = $fiscalYear->id;
        $journalEntry->accounting_period_id = $accountingPeriod->id;
        $journalEntry->entry_number = 'BW-' . $transaction->id; // رقم السحب البنكي
        $journalEntry->entry_date = $transactionDate;
        $journalEntry->description = 'سحب بنكي: ' . $description;
        $journalEntry->reference_type = 'App\\Models\\BankTransaction';
        $journalEntry->reference_id = $transaction->id;
        $journalEntry->is_posted = true;
        $journalEntry->posted_by = auth()->id();
        $journalEntry->posted_at = now();
        $journalEntry->created_by = auth()->id();
        $journalEntry->save();

        // إضافة بنود القيد المحاسبي

        // الطرف الدائن: الحساب البنكي (نقص في الأصول)
        $journalEntry->items()->create([
            'account_id' => $bankAccount->chart_account_id,
            'description' => 'سحب بنكي من ' . $bankAccount->account_name,
            'debit' => 0,
            'credit' => $amount,
        ]);

        // الطرف المدين: الحساب المحدد من المستخدم
        $journalEntry->items()->create([
            'account_id' => $debitAccount->id,
            'description' => 'سحب بنكي إلى حساب: ' . $debitAccount->name,
            'debit' => $amount,
            'credit' => 0,
        ]);

        // ربط المعاملة بالقيد المحاسبي
        $transaction->journal_entry_id = $journalEntry->id;
        $transaction->save();

        // تسجيل معلومات إنشاء القيد المحاسبي
        Log::info('تم إنشاء قيد محاسبي للسحب البنكي', [
            'transaction_id' => $transaction->id,
            'journal_entry_id' => $journalEntry->id,
            'amount' => $amount,
            'bank_account' => $bankAccount->account_name,
            'debit_account' => $debitAccount->name
        ]);

        return $journalEntry;
    }

    /**
     * Store a newly created withdrawal transaction in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeWithdrawal(Request $request)
    {
        $this->authorize('create_bank_transactions');

        $validated = $request->validate([
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'transaction_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'reference_number' => 'nullable|string|max:50',
            'description' => 'required|string',
            'debit_account_id' => 'required|exists:chart_of_accounts,id',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        try {
            DB::beginTransaction();

            $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);

            // التحقق من أن الحساب البنكي ينتمي للشركة الحالية
            if ($bankAccount->company_id !== auth()->user()->company_id) {
                abort(403, 'عملية غير مصرح بها.');
            }

            // التحقق من وجود رصيد كافي في الحساب البنكي
            if ($bankAccount->current_balance < $validated['amount']) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'رصيد الحساب البنكي غير كافي لإجراء عملية السحب.');
            }

            // إنشاء معاملة السحب البنكي
            $transaction = new BankTransaction();
            $transaction->bank_account_id = $validated['bank_account_id'];
            $transaction->transaction_type = 'withdrawal';
            $transaction->transaction_date = $validated['transaction_date'];
            $transaction->amount = $validated['amount'];
            $transaction->reference_number = $validated['reference_number'] ?? null;
            $transaction->description = $validated['description'];
            $transaction->created_by = auth()->id();

            // حفظ معلومات الحساب المدين في البيانات الإضافية
            $additionalData = [
                'debit_account_id' => $validated['debit_account_id']
            ];
            $transaction->additional_data = json_encode($additionalData);

            // معالجة المرفق إذا تم تقديمه
            if ($request->hasFile('attachment')) {
                $path = $request->file('attachment')->store('bank_transactions', 'public');
                $transaction->attachment = $path;
            }

            $transaction->save();

            // تحديث رصيد الحساب البنكي
            $bankAccount->current_balance -= $validated['amount'];
            $bankAccount->save();

            // الحصول على الحساب المدين
            $debitAccount = ChartOfAccount::findOrFail($validated['debit_account_id']);

            // إنشاء قيد محاسبي للسحب البنكي
            $this->createJournalEntryForWithdrawal(
                $transaction,
                $bankAccount,
                $debitAccount,
                $validated['transaction_date'],
                $validated['amount'],
                $validated['description'],
                $validated['reference_number'] ?? null
            );

            // تسجيل النشاط
            Log::info('تم إنشاء سحب بنكي', [
                'transaction_id' => $transaction->id,
                'user_id' => auth()->id(),
                'amount' => $validated['amount'],
                'performed_by' => auth()->user()->name
            ]);

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم إنشاء السحب البنكي بنجاح وتسجيل القيد المحاسبي.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('خطأ في إنشاء السحب البنكي: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء السحب البنكي: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing a deposit transaction.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function editDeposit($id)
    {
        $this->authorize('edit_bank_transactions');

        $transaction = BankTransaction::findOrFail($id);

        // Security check: Ensure the transaction's bank account belongs to the current company
        if ($transaction->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // التحقق من أن المعاملة هي إيداع
        if ($transaction->transaction_type !== 'deposit') {
            return redirect()->route('bank-transactions.index')
                ->with('error', 'المعاملة المحددة ليست إيداعاً بنكياً.');
        }

        $companyId = auth()->user()->company_id;

        // Filter by current company
        $bankAccounts = BankAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $chartAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', '!=', 'bank')
            ->get();
        $elevators = \App\Models\Elevator::where('company_id', $companyId)
            ->get();

        // تحديد معرف الحساب من البيانات الإضافية
        $sourceType = 'chart_of_account';
        $sourceId = null;

        // استخراج معرف الحساب من البيانات الإضافية
        if ($transaction->additional_data) {
            $additionalData = json_decode($transaction->additional_data, true);
            if (isset($additionalData['chart_of_account_id'])) {
                $sourceId = $additionalData['chart_of_account_id'];
            }
        }

        return view('banking.transactions.edit_deposit', compact(
            'transaction',
            'bankAccounts',
            'chartAccounts',
            'elevators',
            'sourceType',
            'sourceId'
        ));
    }

    /**
     * Show the form for editing a withdrawal transaction.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function editWithdrawal($id)
    {
        $this->authorize('edit_bank_transactions');

        $transaction = BankTransaction::findOrFail($id);

        // Security check: Ensure the transaction's bank account belongs to the current company
        if ($transaction->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // التحقق من أن المعاملة هي سحب
        if ($transaction->transaction_type !== 'withdrawal') {
            return redirect()->route('bank-transactions.index')
                ->with('error', 'المعاملة المحددة ليست سحباً بنكياً.');
        }

        $companyId = auth()->user()->company_id;

        // Filter by current company
        $bankAccounts = BankAccount::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $cashRegisters = CashRegister::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $chartAccounts = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', '!=', 'bank')
            ->get();
        $suppliers = \App\Models\Supplier::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();
        $expenseCategories = ChartOfAccount::where('company_id', $companyId)
            ->where('account_type', 'expense')
            ->where('is_active', true)
            ->get();

        // تحديد الحساب المدين من القيد المحاسبي
        $debitAccountId = null;
        if ($transaction->journal_entry) {
            $debitItem = $transaction->journal_entry->items()->where('debit', '>', 0)->first();
            if ($debitItem) {
                $debitAccountId = $debitItem->account_id;
            }
        }

        return view('banking.transactions.edit_withdrawal', compact(
            'transaction',
            'bankAccounts',
            'cashRegisters',
            'chartAccounts',
            'suppliers',
            'expenseCategories',
            'debitAccountId'
        ));
    }

    /**
     * Update the specified withdrawal transaction in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateWithdrawal(Request $request, $id)
    {
        $this->authorize('edit_bank_transactions');

        $transaction = BankTransaction::findOrFail($id);

        // Security check: Ensure transaction belongs to current company
        if ($transaction->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // التحقق من أن المعاملة هي سحب
        if ($transaction->transaction_type !== 'withdrawal') {
            return redirect()->route('bank-transactions.index')
                ->with('error', 'المعاملة المحددة ليست سحباً بنكياً.');
        }

        try {
            DB::beginTransaction();

            // التحقق من صحة البيانات
            $validated = $request->validate([
                'bank_account_id' => 'required|exists:bank_accounts,id',
                'amount' => 'required|numeric|min:0.01',
                'date' => 'required|date',
                'reference' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'debit_account_id' => 'required|exists:chart_of_accounts,id',
                'destination_type' => 'required|in:cash_register,supplier,expense,other',
                'cash_register_id' => 'required_if:destination_type,cash_register|exists:cash_registers,id',
                'supplier_id' => 'required_if:destination_type,supplier|exists:suppliers,id',
                'expense_category_id' => 'required_if:destination_type,expense|exists:chart_of_accounts,id',
                'attachment' => 'nullable|file|mimes:jpeg,png,jpg,pdf|max:2048',
            ]);

            // Security check: Ensure new bank account belongs to current company
            $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);
            if ($bankAccount->company_id !== auth()->user()->company_id) {
                abort(403, 'Unauthorized action.');
            }

            // استرجاع المبلغ القديم إلى الحساب البنكي القديم
            $oldBankAccount = BankAccount::findOrFail($transaction->bank_account_id);
            $oldBankAccount->current_balance += $transaction->amount;
            $oldBankAccount->save();

            // خصم المبلغ الجديد من الحساب البنكي الجديد
            $bankAccount->current_balance -= $validated['amount'];
            $bankAccount->save();

            // حذف القيد المحاسبي القديم إذا وجد
            if ($transaction->journal_entry_id) {
                $oldJournalEntry = JournalEntry::find($transaction->journal_entry_id);
                if ($oldJournalEntry) {
                    $oldJournalEntry->items()->delete();
                    $oldJournalEntry->delete();
                }
            }

            // إنشاء قيد محاسبي جديد
            $journalEntry = JournalEntry::create([
                'date' => $validated['date'],
                'reference' => $validated['reference'] ?? 'Bank Withdrawal #' . $transaction->id,
                'description' => $validated['description'] ?? 'Bank Withdrawal',
                'created_by' => auth()->id(),
            ]);

            // إضافة بنود القيد المحاسبي
            // الائتمان للحساب البنكي
            $journalEntry->items()->create([
                'account_id' => $bankAccount->chart_of_account_id,
                'description' => 'سحب من الحساب البنكي: ' . $bankAccount->account_name,
                'debit' => 0,
                'credit' => $validated['amount'],
            ]);

            // الخصم للحساب المدين
            $debitAccount = ChartOfAccount::findOrFail($validated['debit_account_id']);
            $journalEntry->items()->create([
                'account_id' => $debitAccount->id,
                'description' => 'سحب بنكي - ' . $debitAccount->name,
                'debit' => $validated['amount'],
                'credit' => 0,
            ]);

            // تحديث بيانات المعاملة
            $transaction->bank_account_id = $validated['bank_account_id'];
            $transaction->amount = $validated['amount'];
            $transaction->date = $validated['date'];
            $transaction->reference = $validated['reference'];
            $transaction->description = $validated['description'];
            $transaction->destination = $validated['destination_type'];
            $transaction->journal_entry_id = $journalEntry->id;

            // تحديث البيانات الإضافية حسب نوع الوجهة
            $additionalData = [];
            if ($validated['destination_type'] == 'cash_register') {
                $transaction->cash_register_id = $validated['cash_register_id'];
                $additionalData['cash_register_id'] = $validated['cash_register_id'];

                // استرجاع المبلغ القديم إذا كانت الوجهة القديمة هي خزينة
                $oldDestination = $transaction->destination;
                $oldCashRegisterId = $transaction->cash_register_id;

                if ($oldDestination == 'cash_register' && $oldCashRegisterId) {
                    $oldCashRegister = CashRegister::find($oldCashRegisterId);
                    if ($oldCashRegister) {
                        $oldCashRegister->current_balance -= $transaction->amount;
                        $oldCashRegister->save();
                    }
                }

                // إضافة المبلغ الجديد إلى الخزينة الجديدة
                $cashRegister = CashRegister::findOrFail($validated['cash_register_id']);
                $cashRegister->current_balance += $validated['amount'];
                $cashRegister->save();
            } elseif ($validated['destination_type'] == 'supplier') {
                $transaction->supplier_id = $validated['supplier_id'];
                $additionalData['supplier_id'] = $validated['supplier_id'];
            } elseif ($validated['destination_type'] == 'expense') {
                $additionalData['expense_category_id'] = $validated['expense_category_id'];
            }

            // تحديث المرفق إذا تم تقديمه
            if ($request->hasFile('attachment')) {
                // حذف المرفق القديم إذا وجد
                if ($transaction->attachment) {
                    Storage::delete('public/attachments/' . $transaction->attachment);
                }

                $attachmentPath = $request->file('attachment')->store('public/attachments');
                $transaction->attachment = basename($attachmentPath);
            }

            $transaction->additional_data = json_encode($additionalData);
            $transaction->save();

            // تسجيل النشاط
            Log::info('تم تحديث السحب البنكي', [
                'transaction_id' => $transaction->id,
                'user_id' => auth()->id(),
                'amount' => $validated['amount'],
                'performed_by' => auth()->user()->name
            ]);

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم تحديث السحب البنكي بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating bank withdrawal: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث السحب البنكي. يرجى المحاولة مرة أخرى.');
        }
    }

    /**
     * Update the specified deposit transaction in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateDeposit(Request $request, $id)
    {
        $this->authorize('edit_bank_transactions');

        $transaction = BankTransaction::findOrFail($id);

        // Security check: Ensure transaction belongs to current company
        if ($transaction->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // التحقق من أن المعاملة هي إيداع
        if ($transaction->transaction_type !== 'deposit') {
            return redirect()->route('bank-transactions.index')
                ->with('error', 'المعاملة المحددة ليست إيداعاً بنكياً.');
        }

        $validated = $request->validate([
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'transaction_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'reference_number' => 'nullable|string|max:50',
            'description' => 'required|string',
            'deposit_method' => 'required|in:cash,cheque,transfer,other',
            'source' => 'required|in:chart_account',
            'chart_of_account_id' => 'required|exists:chart_of_accounts,id',
            'elevator_id' => 'nullable|exists:elevators,id',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        try {
            DB::beginTransaction();

            // Security check: Ensure new bank account belongs to current company
            $newBankAccount = BankAccount::findOrFail($validated['bank_account_id']);
            if ($newBankAccount->company_id !== auth()->user()->company_id) {
                abort(403, 'Unauthorized action.');
            }

            // Update the transaction
            $transaction->bank_account_id = $validated['bank_account_id'];
            $transaction->transaction_date = $validated['transaction_date'];
            $transaction->amount = $validated['amount'];
            $transaction->reference_number = $validated['reference_number'] ?? null;
            $transaction->description = $validated['description'];
            $transaction->deposit_method = $validated['deposit_method'];
            $transaction->source = $validated['source'];
            $transaction->updated_by = auth()->id();

            // Update additional data
            $additionalData = [];
            // إذا كان المصدر حساب من شجرة الحسابات
            $additionalData['chart_of_account_id'] = $validated['chart_of_account_id'];

            // إضافة معرف المصعد إذا كان موجودًا
            if (isset($validated['elevator_id'])) {
                $additionalData['elevator_id'] = $validated['elevator_id'];
            }

            $transaction->additional_data = json_encode($additionalData);

            // Handle attachment if provided
            if ($request->hasFile('attachment')) {
                // Delete old attachment if exists
                if ($transaction->attachment) {
                    Storage::disk('public')->delete($transaction->attachment);
                }
                $path = $request->file('attachment')->store('bank_transactions', 'public');
                $transaction->attachment = $path;
            }

            $transaction->save();

            // Log activity
            Log::info('Updated bank deposit', [
                'transaction_id' => $transaction->id,
                'user_id' => auth()->id(),
                'amount' => $validated['amount'],
                'performed_by' => auth()->user()->name
            ]);

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم تحديث الإيداع البنكي بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating bank deposit: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث الإيداع البنكي. يرجى المحاولة مرة أخرى.');
        }
    }

    // هذا القسم تم حذفه لتجنب تكرار تعريف دالة updateWithdrawal

    /**
     * Store a newly created transfer transaction in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeTransfer(Request $request)
    {
        $this->authorize('create_bank_transactions');

        $validated = $request->validate([
            'from_account_id' => 'required|exists:bank_accounts,id',
            'to_account_id' => 'required|exists:bank_accounts,id|different:from_account_id',
            'transaction_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'reference_number' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        try {
            DB::beginTransaction();

            $fromAccount = BankAccount::findOrFail($validated['from_account_id']);
            $toAccount = BankAccount::findOrFail($validated['to_account_id']);

            // Security check: Ensure both accounts belong to the current company
            $companyId = auth()->user()->company_id;
            if ($fromAccount->company_id !== $companyId || $toAccount->company_id !== $companyId) {
                abort(403, 'Unauthorized action.');
            }

            // Check if source account has sufficient balance
            if ($fromAccount->current_balance < $validated['amount']) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Insufficient balance in source account.');
            }

            // Create the withdrawal transaction
            $withdrawalTransaction = new BankTransaction();
            $withdrawalTransaction->bank_account_id = $validated['from_account_id'];
            $withdrawalTransaction->transaction_type = 'transfer_out';
            $withdrawalTransaction->transaction_date = $validated['transaction_date'];
            $withdrawalTransaction->amount = $validated['amount'];
            $withdrawalTransaction->reference_number = $validated['reference_number'] ?? null;
            $withdrawalTransaction->description = 'Transfer to ' . $toAccount->account_name . ': ' . ($validated['description'] ?? '');
            $withdrawalTransaction->created_by = auth()->id();

            // Handle attachment if provided
            if ($request->hasFile('attachment')) {
                $path = $request->file('attachment')->store('bank_transactions', 'public');
                $withdrawalTransaction->attachment = $path;
            }

            $withdrawalTransaction->save();

            // Create the deposit transaction
            $depositTransaction = new BankTransaction();
            $depositTransaction->bank_account_id = $validated['to_account_id'];
            $depositTransaction->transaction_type = 'transfer_in';
            $depositTransaction->transaction_date = $validated['transaction_date'];
            $depositTransaction->amount = $validated['amount'];
            $depositTransaction->reference_number = $validated['reference_number'] ?? null;
            $depositTransaction->description = 'Transfer from ' . $fromAccount->account_name . ': ' . ($validated['description'] ?? '');
            $depositTransaction->related_transaction_id = $withdrawalTransaction->id;
            $depositTransaction->created_by = auth()->id();

            if ($request->hasFile('attachment')) {
                $depositTransaction->attachment = $path;
            }

            $depositTransaction->save();

            // Update the related transaction ID for the withdrawal
            $withdrawalTransaction->related_transaction_id = $depositTransaction->id;
            $withdrawalTransaction->save();

            // Update account balances
            $fromAccount->current_balance -= $validated['amount'];
            $fromAccount->save();

            $toAccount->current_balance += $validated['amount'];
            $toAccount->save();

            // Create journal entry
            $journalEntry = new JournalEntry();
            $journalEntry->entry_date = $validated['transaction_date'];
            $journalEntry->reference_number = $validated['reference_number'] ?? 'BT-' . $withdrawalTransaction->id;
            $journalEntry->description = 'Bank Transfer: ' . ($validated['description'] ?? '');
            $journalEntry->status = 'posted';
            $journalEntry->created_by = auth()->id();
            $journalEntry->save();

            // Add journal entry items
            // Credit source account
            $journalEntry->items()->create([
                'chart_of_account_id' => $fromAccount->chart_of_account_id,
                'description' => 'Transfer from ' . $fromAccount->account_name,
                'debit' => 0,
                'credit' => $validated['amount'],
            ]);

            // Debit destination account
            $journalEntry->items()->create([
                'chart_of_account_id' => $toAccount->chart_of_account_id,
                'description' => 'Transfer to ' . $toAccount->account_name,
                'debit' => $validated['amount'],
                'credit' => 0,
            ]);

            // Link transactions to journal entry
            $withdrawalTransaction->journal_entry_id = $journalEntry->id;
            $withdrawalTransaction->save();

            $depositTransaction->journal_entry_id = $journalEntry->id;
            $depositTransaction->save();

            // Log activity
            Log::info('Bank transfer created', [
                'withdrawal_transaction_id' => $withdrawalTransaction->id,
                'deposit_transaction_id' => $depositTransaction->id,
                'user_id' => auth()->id(),
                'from_account' => $fromAccount->account_name,
                'to_account' => $toAccount->account_name,
                'amount' => $validated['amount'],
                'performed_by' => auth()->user()->name
            ]);

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'Bank transfer created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating bank transfer: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error creating bank transfer. Please try again.');
        }
    }

    /**
     * Display the specified bank transaction.
     *
     * @param  \App\Models\BankTransaction  $transaction
     * @return \Illuminate\Http\Response
     */
    public function show(BankTransaction $transaction)
    {
        $this->authorize('view_bank_transactions');

        // Security check: Ensure the transaction's bank account belongs to the current company
        if ($transaction->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Load related journal entry if exists
        if ($transaction->journal_entry_id) {
            $transaction->load('journalEntry.items.chartOfAccount');
        }

        // Load related transaction if it's a transfer
        if ($transaction->related_transaction_id) {
            $transaction->load('relatedTransaction.bankAccount');
        }

        return view('banking.transactions.show', compact('transaction'));
    }

    /**
     * Remove the specified bank transaction from storage.
     *
     * @param  \App\Models\BankTransaction  $transaction
     * @return \Illuminate\Http\Response
     */
    public function destroy(BankTransaction $transaction)
    {
        $this->authorize('delete_bank_transactions');

        // Security check: Ensure the transaction's bank account belongs to the current company
        if ($transaction->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $bankAccount = $transaction->bankAccount;

            // Check if transaction is part of a reconciliation
            if ($transaction->reconciliationItems()->count() > 0) {
                return redirect()->back()
                    ->with('error', 'Cannot delete transaction that is part of a reconciliation.');
            }

            // Reverse the effect on bank account balance
            if (in_array($transaction->transaction_type, ['deposit', 'transfer_in'])) {
                $bankAccount->current_balance -= $transaction->amount;
            } else {
                $bankAccount->current_balance += $transaction->amount;
            }

            $bankAccount->save();

            // If it's a transfer, handle the related transaction
            if ($transaction->related_transaction_id) {
                $relatedTransaction = BankTransaction::find($transaction->related_transaction_id);

                if ($relatedTransaction) {
                    $relatedAccount = $relatedTransaction->bankAccount;

                    // Reverse the effect on related account balance
                    if (in_array($relatedTransaction->transaction_type, ['deposit', 'transfer_in'])) {
                        $relatedAccount->current_balance -= $relatedTransaction->amount;
                    } else {
                        $relatedAccount->current_balance += $relatedTransaction->amount;
                    }

                    $relatedAccount->save();

                    // Delete related transaction
                    $relatedTransaction->delete();
                }
            }

            // Delete journal entry if exists
            if ($transaction->journal_entry_id) {
                $journalEntry = JournalEntry::find($transaction->journal_entry_id);

                if ($journalEntry) {
                    // Delete journal entry items
                    $journalEntry->items()->delete();

                    // Delete journal entry
                    $journalEntry->delete();
                }
            }

            // Log activity before deletion
            Log::info('Bank transaction deleted', [
                'transaction_id' => $transaction->id,
                'user_id' => auth()->id(),
                'transaction_type' => $transaction->transaction_type,
                'amount' => $transaction->amount,
                'performed_by' => auth()->user()->name
            ]);

            // Delete transaction
            $transaction->delete();

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'Bank transaction deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting bank transaction: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Error deleting bank transaction. Please try again.');
        }
    }
}
