<?php

namespace App\Http\Controllers;

use App\Models\BankAccount;
use App\Models\BankReconciliation;
use App\Models\BankReconciliationItem;
use App\Models\BankTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BankReconciliationController extends Controller
{
    /**
     * Display a listing of the bank reconciliations.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->authorize('view_bank_reconciliations');

        // Filter by current company
        $query = BankReconciliation::with(['bankAccount', 'bankAccount.bank'])
            ->whereHas('bankAccount', function ($q) {
                $q->where('company_id', auth()->user()->company_id);
            });

        // Apply filters if provided
        if ($request->has('account_id') && $request->account_id) {
            $query->where('bank_account_id', $request->account_id);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('start_date') && $request->start_date) {
            $query->where('statement_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->where('statement_date', '<=', $request->end_date);
        }

        $reconciliations = $query->orderBy('statement_date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(15);

        $accounts = BankAccount::where('is_active', true)
            ->where('company_id', auth()->user()->company_id)
            ->get();

        return view('banking.reconciliations.index', compact('reconciliations', 'accounts'));
    }

    /**
     * Show the form for creating a new bank reconciliation.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->authorize('create_bank_reconciliations');

        $bankAccounts = BankAccount::where('is_active', true)
            ->where('company_id', auth()->user()->company_id)
            ->get();

        return view('banking.reconciliations.create', compact('bankAccounts'));
    }

    /**
     * Store a newly created bank reconciliation in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->authorize('create_bank_reconciliations');

        $validated = $request->validate([
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'statement_date' => 'required|date',
            'statement_balance' => 'required|numeric',
            'statement_closing_date' => 'required|date|after_or_equal:statement_date',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $bankAccount = BankAccount::findOrFail($validated['bank_account_id']);

            // Security check: Ensure bank account belongs to current company
            if ($bankAccount->company_id !== auth()->user()->company_id) {
                abort(403, 'Unauthorized action.');
            }

            // Check if there's already an in-progress reconciliation for this account
            $existingReconciliation = BankReconciliation::where('bank_account_id', $validated['bank_account_id'])
                ->where('status', 'in_progress')
                ->first();

            if ($existingReconciliation) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'There is already an in-progress reconciliation for this account.');
            }

            // Create the reconciliation
            $reconciliation = new BankReconciliation();
            $reconciliation->bank_account_id = $validated['bank_account_id'];
            $reconciliation->statement_date = $validated['statement_date'];
            $reconciliation->statement_balance = $validated['statement_balance'];
            $reconciliation->statement_closing_date = $validated['statement_closing_date'];
            $reconciliation->notes = $validated['notes'] ?? null;
            $reconciliation->status = 'in_progress';
            $reconciliation->created_by = auth()->id();
            $reconciliation->save();

            // Log activity
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties(['reconciliation_id' => $reconciliation->id])
                ->log('Created bank reconciliation for account: ' . $bankAccount->account_name);

            DB::commit();

            return redirect()->route('bank-reconciliations.edit', $reconciliation)
                ->with('success', 'Bank reconciliation created successfully. You can now reconcile transactions.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating bank reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error creating bank reconciliation. Please try again.');
        }
    }

    /**
     * Display the specified bank reconciliation.
     *
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function show(BankReconciliation $reconciliation)
    {
        $this->authorize('view_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Load reconciliation items with their transactions
        $reconciliation->load('items.bankTransaction');

        return view('banking.reconciliations.show', compact('reconciliation'));
    }

    /**
     * Show the form for editing the specified bank reconciliation.
     *
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function edit(BankReconciliation $reconciliation)
    {
        $this->authorize('edit_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow editing of in-progress reconciliations
        if ($reconciliation->status != 'in_progress') {
            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('error', 'Cannot edit a completed or cancelled reconciliation.');
        }

        // Get all transactions for this account up to the statement closing date
        $bankAccount = $reconciliation->bankAccount;

        // Get transactions that are already reconciled in this reconciliation
        $reconciledTransactionIds = $reconciliation->items()->pluck('bank_transaction_id')->toArray();

        // Get transactions that are not yet reconciled
        $unReconciledTransactions = BankTransaction::where('bank_account_id', $bankAccount->id)
            ->where('transaction_date', '<=', $reconciliation->statement_closing_date)
            ->whereNotIn('id', $reconciledTransactionIds)
            ->whereNull('reconciled_at')
            ->orderBy('transaction_date')
            ->get();

        // Get reconciled transactions for this reconciliation
        $reconciledTransactions = BankTransaction::whereIn('id', $reconciledTransactionIds)
            ->orderBy('transaction_date')
            ->get();

        // Calculate current difference
        $reconciledAmount = $reconciliation->items()->sum('amount');
        $difference = $reconciliation->statement_balance - ($bankAccount->current_balance + $reconciledAmount);

        return view('banking.reconciliations.edit', compact(
            'reconciliation',
            'bankAccount',
            'unReconciledTransactions',
            'reconciledTransactions',
            'difference'
        ));
    }

    /**
     * Update the specified bank reconciliation in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, BankReconciliation $reconciliation)
    {
        $this->authorize('edit_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow editing of in-progress reconciliations
        if ($reconciliation->status != 'in_progress') {
            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('error', 'Cannot edit a completed or cancelled reconciliation.');
        }

        $validated = $request->validate([
            'statement_date' => 'required|date',
            'statement_balance' => 'required|numeric',
            'statement_closing_date' => 'required|date|after_or_equal:statement_date',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $reconciliation->statement_date = $validated['statement_date'];
            $reconciliation->statement_balance = $validated['statement_balance'];
            $reconciliation->statement_closing_date = $validated['statement_closing_date'];
            $reconciliation->notes = $validated['notes'] ?? $reconciliation->notes;
            $reconciliation->save();

            // Log activity
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties(['reconciliation_id' => $reconciliation->id])
                ->log('Updated bank reconciliation details');

            DB::commit();

            return redirect()->route('bank-reconciliations.edit', $reconciliation)
                ->with('success', 'Bank reconciliation updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating bank reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error updating bank reconciliation. Please try again.');
        }
    }

    /**
     * Add a transaction to the reconciliation.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function addTransaction(Request $request, BankReconciliation $reconciliation)
    {
        $this->authorize('edit_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow editing of in-progress reconciliations
        if ($reconciliation->status != 'in_progress') {
            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('error', 'Cannot edit a completed or cancelled reconciliation.');
        }

        $validated = $request->validate([
            'transaction_id' => 'required|exists:bank_transactions,id',
        ]);

        try {
            DB::beginTransaction();

            $transaction = BankTransaction::findOrFail($validated['transaction_id']);

            // Check if transaction belongs to the same bank account
            if ($transaction->bank_account_id != $reconciliation->bank_account_id) {
                return redirect()->back()
                    ->with('error', 'Transaction does not belong to the same bank account.');
            }

            // Check if transaction is already reconciled
            if ($transaction->reconciled_at) {
                return redirect()->back()
                    ->with('error', 'Transaction is already reconciled.');
            }

            // Check if transaction is already in this reconciliation
            $existingItem = BankReconciliationItem::where('bank_reconciliation_id', $reconciliation->id)
                ->where('bank_transaction_id', $transaction->id)
                ->first();

            if ($existingItem) {
                return redirect()->back()
                    ->with('error', 'Transaction is already in this reconciliation.');
            }

            // Add transaction to reconciliation
            $item = new BankReconciliationItem();
            $item->bank_reconciliation_id = $reconciliation->id;
            $item->bank_transaction_id = $transaction->id;
            $item->amount = $transaction->amount;
            $item->save();

            // Log activity
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties([
                    'reconciliation_id' => $reconciliation->id,
                    'transaction_id' => $transaction->id
                ])
                ->log('Added transaction to bank reconciliation');

            DB::commit();

            return redirect()->route('bank-reconciliations.edit', $reconciliation)
                ->with('success', 'Transaction added to reconciliation successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error adding transaction to reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Error adding transaction to reconciliation. Please try again.');
        }
    }

    /**
     * Remove a transaction from the reconciliation.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function removeTransaction(Request $request, BankReconciliation $reconciliation)
    {
        $this->authorize('edit_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow editing of in-progress reconciliations
        if ($reconciliation->status != 'in_progress') {
            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('error', 'Cannot edit a completed or cancelled reconciliation.');
        }

        $validated = $request->validate([
            'item_id' => 'required|exists:bank_reconciliation_items,id',
        ]);

        try {
            DB::beginTransaction();

            $item = BankReconciliationItem::findOrFail($validated['item_id']);

            // Check if item belongs to this reconciliation
            if ($item->bank_reconciliation_id != $reconciliation->id) {
                return redirect()->back()
                    ->with('error', 'Item does not belong to this reconciliation.');
            }

            // Log activity before deletion
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties([
                    'reconciliation_id' => $reconciliation->id,
                    'transaction_id' => $item->bank_transaction_id
                ])
                ->log('Removed transaction from bank reconciliation');

            // Delete item
            $item->delete();

            DB::commit();

            return redirect()->route('bank-reconciliations.edit', $reconciliation)
                ->with('success', 'Transaction removed from reconciliation successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error removing transaction from reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Error removing transaction from reconciliation. Please try again.');
        }
    }

    /**
     * Complete the bank reconciliation.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function complete(Request $request, BankReconciliation $reconciliation)
    {
        $this->authorize('edit_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow completion of in-progress reconciliations
        if ($reconciliation->status != 'in_progress') {
            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('error', 'Cannot complete a reconciliation that is not in progress.');
        }

        try {
            DB::beginTransaction();

            // Get all reconciliation items
            $items = $reconciliation->items;

            // Mark all transactions as reconciled
            foreach ($items as $item) {
                $transaction = BankTransaction::find($item->bank_transaction_id);

                if ($transaction) {
                    $transaction->reconciled_at = now();
                    $transaction->save();
                }
            }

            // Update reconciliation status
            $reconciliation->status = 'completed';
            $reconciliation->completed_at = now();
            $reconciliation->completed_by = auth()->id();
            $reconciliation->save();

            // Log activity
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties(['reconciliation_id' => $reconciliation->id])
                ->log('Completed bank reconciliation');

            DB::commit();

            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('success', 'Bank reconciliation completed successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error completing bank reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Error completing bank reconciliation. Please try again.');
        }
    }

    /**
     * Cancel the bank reconciliation.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function cancel(Request $request, BankReconciliation $reconciliation)
    {
        $this->authorize('edit_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow cancellation of in-progress reconciliations
        if ($reconciliation->status != 'in_progress') {
            return redirect()->route('bank-reconciliations.show', $reconciliation)
                ->with('error', 'Cannot cancel a reconciliation that is not in progress.');
        }

        try {
            DB::beginTransaction();

            // Update reconciliation status
            $reconciliation->status = 'cancelled';
            $reconciliation->save();

            // Log activity
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties(['reconciliation_id' => $reconciliation->id])
                ->log('Cancelled bank reconciliation');

            DB::commit();

            return redirect()->route('bank-reconciliations.index')
                ->with('success', 'Bank reconciliation cancelled successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error cancelling bank reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Error cancelling bank reconciliation. Please try again.');
        }
    }

    /**
     * Remove the specified bank reconciliation from storage.
     *
     * @param  \App\Models\BankReconciliation  $reconciliation
     * @return \Illuminate\Http\Response
     */
    public function destroy(BankReconciliation $reconciliation)
    {
        $this->authorize('delete_bank_reconciliations');

        // Security check: Ensure reconciliation belongs to current company
        if ($reconciliation->bankAccount->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Only allow deletion of in-progress or cancelled reconciliations
        if ($reconciliation->status == 'completed') {
            return redirect()->back()
                ->with('error', 'Cannot delete a completed reconciliation.');
        }

        try {
            DB::beginTransaction();

            // Delete all reconciliation items
            $reconciliation->items()->delete();

            // Log activity before deletion
            activity()
                ->performedOn($reconciliation)
                ->causedBy(auth()->user())
                ->withProperties([
                    'reconciliation_id' => $reconciliation->id,
                    'bank_account' => $reconciliation->bankAccount->account_name
                ])
                ->log('Deleted bank reconciliation');

            // Delete reconciliation
            $reconciliation->delete();

            DB::commit();

            return redirect()->route('bank-reconciliations.index')
                ->with('success', 'Bank reconciliation deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting bank reconciliation: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Error deleting bank reconciliation. Please try again.');
        }
    }
}
