<?php

namespace App\Http\Controllers;

use App\Models\Bank;
use App\Models\ChartOfAccount;
use App\Services\BankAccountService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BankController extends Controller
{
    /**
     * @var BankAccountService
     */
    protected $bankAccountService;

    /**
     * Create a new controller instance.
     *
     * @param BankAccountService $bankAccountService
     */
    public function __construct(BankAccountService $bankAccountService)
    {
        $this->bankAccountService = $bankAccountService;
    }

    /**
     * Display a listing of the banks.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->authorize('view_banks');

        $user = auth()->user();
        $query = Bank::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $banks = $query->get();
        return view('banking.banks.index', compact('banks'));
    }

    /**
     * Show the form for creating a new bank.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->authorize('create_banks');

        return view('banking.banks.create');
    }

    /**
     * Store a newly created bank in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->authorize('create_banks');

        $user = auth()->user();
        $rules = [
            'name' => 'required|string|max:255',
            'branch' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:500',
            'contact_person' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'swift_code' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
            'status' => 'required|boolean',
        ];

        if ($user->isSystemAdmin()) {
            $rules['company_id'] = 'required|exists:companies,id';
        }

        $validated = $request->validate($rules);

        // Add company_id to validated data if not already present (for regular users)
        if (!isset($validated['company_id'])) {
            $validated['company_id'] = $user->company_id;
        }

        // تحويل أسماء الحقول من النموذج إلى أسماء الحقول في قاعدة البيانات
        if (isset($validated['branch'])) {
            $validated['code'] = $validated['branch'];
            unset($validated['branch']);
        }

        if (isset($validated['phone'])) {
            $validated['contact_phone'] = $validated['phone'];
            unset($validated['phone']);
        }

        if (isset($validated['email'])) {
            $validated['contact_email'] = $validated['email'];
            unset($validated['email']);
        }

        if (isset($validated['status'])) {
            $validated['is_active'] = $validated['status'];
            unset($validated['status']);
        }

        try {
            DB::beginTransaction();

            $bank = Bank::create($validated);

            // إنشاء حساب للبنك في شجرة الحسابات
            $account = $this->bankAccountService->createBankAccount($bank);

            if (!$account) {
                throw new \Exception('فشل في إنشاء حساب البنك في شجرة الحسابات');
            }

            // Log activity
            Log::info('Created new bank: ' . $bank->name, [
                'bank_id' => $bank->id,
                'user_id' => auth()->id(),
                'action' => 'create',
                'account_id' => $account->id
            ]);

            DB::commit();

            return redirect()->route('banks.index')
                ->with('success', 'تم إنشاء البنك وحسابه في شجرة الحسابات بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating bank: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'validated_data' => $validated
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error creating bank. Please try again. Error: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified bank.
     *
     * @param  \App\Models\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function show(Bank $bank)
    {
        $this->authorize('view_banks');

        $user = auth()->user();
        // Verify bank belongs to current company
        if (!$user->isSystemAdmin() && $bank->company_id !== $user->company_id) {
            abort(403, 'Unauthorized access to this bank.');
        }

        // Get bank accounts associated with this bank
        $bankAccounts = $bank->bankAccounts;

        return view('banking.banks.show', compact('bank', 'bankAccounts'));
    }

    /**
     * Show the form for editing the specified bank.
     *
     * @param  \App\Models\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function edit(Bank $bank)
    {
        $this->authorize('edit_banks');

        $user = auth()->user();
        // Verify bank belongs to current company
        if (!$user->isSystemAdmin() && $bank->company_id !== $user->company_id) {
            abort(403, 'Unauthorized access to this bank.');
        }

        return view('banking.banks.edit', compact('bank'));
    }

    /**
     * Update the specified bank in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Bank $bank)
    {
        $this->authorize('edit_banks');

        $user = auth()->user();
        // Verify bank belongs to current company
        if (!$user->isSystemAdmin() && $bank->company_id !== $user->company_id) {
            abort(403, 'Unauthorized access to this bank.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'branch' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:500',
            'contact_person' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'swift_code' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
            'status' => 'required|boolean',
        ]);

        // تحويل أسماء الحقول من النموذج إلى أسماء الحقول في قاعدة البيانات
        if (isset($validated['branch'])) {
            $validated['code'] = $validated['branch'];
            unset($validated['branch']);
        }

        if (isset($validated['phone'])) {
            $validated['contact_phone'] = $validated['phone'];
            unset($validated['phone']);
        }

        if (isset($validated['email'])) {
            $validated['contact_email'] = $validated['email'];
            unset($validated['email']);
        }

        if (isset($validated['status'])) {
            $validated['is_active'] = $validated['status'];
            unset($validated['status']);
        }

        try {
            DB::beginTransaction();

            // حفظ الاسم القديم للبنك قبل التحديث
            $oldName = $bank->name;

            // تحديث البنك
            $bank->update($validated);

            // البحث عن حساب البنك في شجرة الحسابات أو إنشائه إذا لم يكن موجوداً
            $account = $this->bankAccountService->findBankAccount($bank);

            if ($account) {
                // تحديث الحساب إذا كان موجوداً
                $this->bankAccountService->updateBankAccount($bank, $account);
            } else {
                // إنشاء حساب جديد إذا لم يكن موجوداً
                $account = $this->bankAccountService->createBankAccount($bank);

                if (!$account) {
                    throw new \Exception('فشل في إنشاء حساب البنك في شجرة الحسابات');
                }
            }

            // Log activity
            Log::info('Updated bank: ' . $bank->name, [
                'bank_id' => $bank->id,
                'user_id' => auth()->id(),
                'action' => 'update',
                'old_name' => $oldName,
                'account_id' => $account->id
            ]);

            DB::commit();

            return redirect()->route('banks.index')
                ->with('success', 'تم تحديث البنك وحسابه في شجرة الحسابات بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating bank: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'bank_id' => $bank->id
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error updating bank. Please try again. Error: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified bank from storage.
     *
     * @param  \App\Models\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function destroy(Bank $bank)
    {
        $this->authorize('delete_banks');

        $user = auth()->user();
        // Verify bank belongs to current company
        if (!$user->isSystemAdmin() && $bank->company_id !== $user->company_id) {
            abort(403, 'Unauthorized access to this bank.');
        }

        try {
            DB::beginTransaction();

            // Check if bank has any accounts
            if ($bank->bankAccounts()->count() > 0) {
                return redirect()->back()
                    ->with('error', 'لا يمكن حذف البنك لوجود حسابات بنكية مرتبطة به.');
            }

            // البحث عن حساب البنك في شجرة الحسابات
            $account = $this->bankAccountService->findBankAccount($bank);

            // حذف حساب البنك من شجرة الحسابات إذا كان موجوداً
            if ($account) {
                $accountId = $account->id;
                $account->delete();

                Log::info('تم حذف حساب البنك من شجرة الحسابات', [
                    'bank_id' => $bank->id,
                    'bank_name' => $bank->name,
                    'account_id' => $accountId,
                    'user_id' => auth()->id()
                ]);
            }

            // Log activity before deletion
            Log::info('Deleted bank: ' . $bank->name, [
                'bank_id' => $bank->id,
                'bank_name' => $bank->name,
                'user_id' => auth()->id(),
                'action' => 'delete'
            ]);

            $bank->delete();

            DB::commit();

            return redirect()->route('banks.index')
                ->with('success', 'تم حذف البنك وحسابه في شجرة الحسابات بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting bank: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'bank_id' => $bank->id
            ]);

            return redirect()->back()
                ->with('error', 'Error deleting bank. Please try again. Error: ' . $e->getMessage());
        }
    }
}
