<?php

namespace App\Http\Controllers;

use App\Models\AutoPurchaseRule;
use App\Models\InventoryItem;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class AutoPurchaseController extends Controller
{
    /**
     * عرض قائمة قواعد الشراء التلقائي
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        $rules = AutoPurchaseRule::where('company_id', $company->id)
            ->with(['item', 'supplier', 'warehouse'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('inventory.auto_purchase.index', compact('rules'));
    }

    /**
     * عرض نموذج إنشاء قاعدة جديدة
     */
    public function create()
    {
        $user = Auth::user();
        $company = $user->company;

        $items = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $suppliers = Supplier::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.auto_purchase.create', compact('items', 'suppliers', 'warehouses'));
    }

    /**
     * حفظ قاعدة جديدة
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $validator = Validator::make($request->all(), [
            'item_id' => 'required|exists:inventory_items,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:1',
            'frequency_days' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من أن العنصر والمورد والمستودع ينتمون للشركة
        $item = InventoryItem::where('id', $request->item_id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $supplier = Supplier::where('id', $request->supplier_id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $warehouse = Warehouse::where('id', $request->warehouse_id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $rule = new AutoPurchaseRule();
        $rule->company_id = $company->id;
        $rule->item_id = $request->item_id;
        $rule->supplier_id = $request->supplier_id;
        $rule->warehouse_id = $request->warehouse_id;
        $rule->min_quantity = $request->min_quantity;
        $rule->order_quantity = $request->order_quantity;
        $rule->frequency_days = $request->frequency_days ?? 0;
        $rule->is_active = $request->has('is_active');
        $rule->next_check_date = Carbon::today();
        $rule->save();

        return redirect()->route('inventory.auto-purchase.index')
            ->with('success', 'تم إنشاء قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * عرض نموذج تعديل قاعدة
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $rule = AutoPurchaseRule::where('id', $id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $items = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $suppliers = Supplier::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.auto_purchase.edit', compact('rule', 'items', 'suppliers', 'warehouses'));
    }

    /**
     * تحديث قاعدة
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $rule = AutoPurchaseRule::where('id', $id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'item_id' => 'required|exists:inventory_items,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:1',
            'frequency_days' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من أن العنصر والمورد والمستودع ينتمون للشركة
        $item = InventoryItem::where('id', $request->item_id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $supplier = Supplier::where('id', $request->supplier_id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $warehouse = Warehouse::where('id', $request->warehouse_id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $rule->item_id = $request->item_id;
        $rule->supplier_id = $request->supplier_id;
        $rule->warehouse_id = $request->warehouse_id;
        $rule->min_quantity = $request->min_quantity;
        $rule->order_quantity = $request->order_quantity;
        $rule->frequency_days = $request->frequency_days ?? 0;
        $rule->is_active = $request->has('is_active');
        $rule->save();

        return redirect()->route('inventory.auto-purchase.index')
            ->with('success', 'تم تحديث قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * حذف قاعدة
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $rule = AutoPurchaseRule::where('id', $id)
            ->where('company_id', $company->id)
            ->firstOrFail();

        $rule->delete();

        return redirect()->route('inventory.auto-purchase.index')
            ->with('success', 'تم حذف قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * تشغيل فحص المخزون وإنشاء طلبات الشراء التلقائية
     */
    public function runCheck()
    {
        $user = Auth::user();
        $company = $user->company;

        // الحصول على القواعد النشطة التي يجب فحصها اليوم
        $rules = AutoPurchaseRule::where('company_id', $company->id)
            ->where('is_active', true)
            ->where('next_check_date', '<=', Carbon::today())
            ->with(['item', 'supplier', 'warehouse'])
            ->get();

        $createdOrders = 0;
        $processedItems = 0;

        // تجميع القواعد حسب المورد والمستودع
        $groupedRules = $rules->groupBy(function ($rule) {
            return $rule->supplier_id . '-' . $rule->warehouse_id;
        });

        foreach ($groupedRules as $group => $groupRules) {
            $firstRule = $groupRules->first();
            $supplier = $firstRule->supplier;
            $warehouse = $firstRule->warehouse;

            $itemsToOrder = [];

            foreach ($groupRules as $rule) {
                $processedItems++;

                // فحص المخزون الحالي
                $stock = $rule->item->stocks()
                    ->where('warehouse_id', $rule->warehouse_id)
                    ->first();

                $currentQuantity = $stock ? $stock->quantity - $stock->reserved_quantity : 0;

                // إذا كان المخزون أقل من الحد الأدنى، أضف العنصر إلى قائمة الطلب
                if ($currentQuantity <= $rule->min_quantity) {
                    $itemsToOrder[] = [
                        'rule' => $rule,
                        'item' => $rule->item,
                        'quantity' => $rule->order_quantity,
                    ];
                }

                // تحديث تاريخ الفحص التالي
                if ($rule->frequency_days > 0) {
                    $rule->next_check_date = Carbon::today()->addDays($rule->frequency_days);
                } else {
                    $rule->next_check_date = Carbon::today()->addDay();
                }
                $rule->save();
            }

            // إذا كان هناك عناصر للطلب، أنشئ طلب شراء
            if (count($itemsToOrder) > 0) {
                // إنشاء طلب شراء جديد
                $purchaseOrder = new PurchaseOrder();
                $purchaseOrder->company_id = $company->id;
                $purchaseOrder->supplier_id = $supplier->id;
                $purchaseOrder->warehouse_id = $warehouse->id;
                $purchaseOrder->created_by = $user->id;
                $purchaseOrder->po_number = 'AUTO-' . time();
                $purchaseOrder->order_date = Carbon::today();
                $purchaseOrder->expected_delivery_date = Carbon::today()->addDays(7);
                $purchaseOrder->status = 'draft';
                $purchaseOrder->notes = 'تم إنشاء هذا الطلب تلقائيًا بواسطة نظام الطلب التلقائي';

                // حساب المجاميع
                $subtotal = 0;

                foreach ($itemsToOrder as $orderItem) {
                    $item = $orderItem['item'];
                    $quantity = $orderItem['quantity'];
                    $subtotal += $item->purchase_price * $quantity;
                }

                $purchaseOrder->subtotal = $subtotal;
                $purchaseOrder->tax_amount = 0;
                $purchaseOrder->discount_amount = 0;
                $purchaseOrder->total_amount = $subtotal;

                $purchaseOrder->save();

                // إضافة عناصر الطلب
                foreach ($itemsToOrder as $orderItem) {
                    $item = $orderItem['item'];
                    $rule = $orderItem['rule'];
                    $quantity = $orderItem['quantity'];

                    $poItem = new PurchaseOrderItem();
                    $poItem->purchase_order_id = $purchaseOrder->id;
                    $poItem->item_id = $item->id;
                    $poItem->quantity = $quantity;
                    $poItem->received_quantity = 0;
                    $poItem->unit_price = $item->purchase_price;
                    $poItem->tax_rate = 0;
                    $poItem->discount_amount = 0;
                    $poItem->total_amount = $item->purchase_price * $quantity;
                    $poItem->notes = 'تم إنشاؤه تلقائيًا - الحد الأدنى: ' . $rule->min_quantity;
                    $poItem->save();

                    // تحديث وقت آخر تشغيل للقاعدة
                    $rule->last_triggered_at = now();
                    $rule->save();
                }

                $createdOrders++;
            }
        }

        return redirect()->route('inventory.auto-purchase.index')
            ->with('success', "تم فحص {$processedItems} عنصر وإنشاء {$createdOrders} طلب شراء تلقائي");
    }

    /**
     * عرض سجل الطلبات التلقائية
     */
    public function history()
    {
        $user = Auth::user();
        $company = $user->company;

        $purchaseOrders = PurchaseOrder::where('company_id', $company->id)
            ->where('po_number', 'like', 'AUTO-%')
            ->with(['supplier', 'warehouse', 'items.item'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('inventory.auto_purchase.history', compact('purchaseOrders'));
    }

    /**
     * عرض لوحة معلومات الطلبات التلقائية
     */
    public function dashboard()
    {
        $user = Auth::user();
        $company = $user->company;

        // إحصائيات القواعد
        $totalRules = AutoPurchaseRule::where('company_id', $company->id)->count();
        $activeRules = AutoPurchaseRule::where('company_id', $company->id)
            ->where('is_active', true)
            ->count();

        // إحصائيات الطلبات التلقائية
        $totalAutoOrders = PurchaseOrder::where('company_id', $company->id)
            ->where('po_number', 'like', 'AUTO-%')
            ->count();

        $recentAutoOrders = PurchaseOrder::where('company_id', $company->id)
            ->where('po_number', 'like', 'AUTO-%')
            ->with(['supplier', 'warehouse'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // العناصر التي تحتاج إلى طلب
        $lowStockItems = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['category', 'stocks.warehouse'])
            ->get()
            ->filter(function ($item) {
                return $item->is_low_stock;
            })
            ->take(10);

        // القواعد التي سيتم فحصها قريبًا
        $upcomingChecks = AutoPurchaseRule::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['item', 'supplier', 'warehouse'])
            ->orderBy('next_check_date')
            ->limit(10)
            ->get();

        return view('inventory.auto_purchase.dashboard', compact(
            'totalRules',
            'activeRules',
            'totalAutoOrders',
            'recentAutoOrders',
            'lowStockItems',
            'upcomingChecks'
        ));
    }
}
