<?php

namespace App\Http\Controllers;

use App\Services\AutoInvoiceService;
use Illuminate\Http\Request;
use App\Models\MaintenanceContract;
use App\Models\MaintenanceSchedule;
use Illuminate\Support\Facades\Auth;

class AutoInvoiceController extends Controller
{
    protected $autoInvoiceService;

    /**
     * Create a new controller instance.
     *
     * @param  AutoInvoiceService  $autoInvoiceService
     * @return void
     */
    public function __construct(AutoInvoiceService $autoInvoiceService)
    {
        $this->middleware('auth');
        $this->middleware('permission:generate-auto-invoices');
        $this->autoInvoiceService = $autoInvoiceService;
    }

    /**
     * Show the auto invoice dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        // Get contracts eligible for auto-invoicing
        $contractsQuery = MaintenanceContract::query();
        if (!$user->isSystemAdmin()) {
            $contractsQuery->where('company_id', $companyId);
        }
        $contracts = $contractsQuery->where('status', 'active')
            ->with(['customer', 'elevators'])
            ->get();

        // Get completed maintenance schedules without invoices
        $schedulesQuery = MaintenanceSchedule::query();
        if (!$user->isSystemAdmin()) {
            $schedulesQuery->where('company_id', $companyId);
        }
        $schedules = $schedulesQuery->where('status', 'completed')
            ->whereNull('invoice_id')
            ->with(['elevator', 'maintenanceContract'])
            ->get();

        return view('invoices.auto.index', compact('contracts', 'schedules'));
    }

    /**
     * Generate invoices from contracts.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generateFromContracts(Request $request)
    {
        $request->validate([
            'contract_ids' => 'required|array',
            'contract_ids.*' => 'exists:maintenance_contracts,id',
        ]);

        $contractIds = $request->input('contract_ids');
        $user = Auth::user();

        // Get selected contracts
        $query = MaintenanceContract::query();
        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }
        
        $contracts = $query->whereIn('id', $contractIds)
            ->with(['customer', 'company', 'elevators'])
            ->get();

        $success = 0;
        $failed = 0;

        foreach ($contracts as $contract) {
            try {
                $this->autoInvoiceService->createInvoiceFromContract($contract);
                $success++;
            } catch (\Exception $e) {
                $failed++;
            }
        }

        return redirect()->route('auto-invoices.index')
            ->with('success', "تم إنشاء {$success} فاتورة بنجاح. فشل إنشاء {$failed} فاتورة.");
    }

    /**
     * Generate invoices from maintenance schedules.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generateFromSchedules(Request $request)
    {
        $request->validate([
            'schedule_ids' => 'required|array',
            'schedule_ids.*' => 'exists:maintenance_schedules,id',
        ]);

        $scheduleIds = $request->input('schedule_ids');
        $user = Auth::user();

        // Get selected schedules
        $query = MaintenanceSchedule::query();
        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $schedules = $query->whereIn('id', $scheduleIds)
            ->with(['elevator', 'maintenanceContract', 'maintenanceLogs', 'company'])
            ->get();

        // Group schedules by contract
        $groupedSchedules = $schedules->groupBy('maintenance_contract_id');

        $success = 0;
        $failed = 0;

        foreach ($groupedSchedules as $contractId => $contractSchedules) {
            try {
                $contract = MaintenanceContract::find($contractId);
                if (!$contract) continue;

                $invoice = $this->autoInvoiceService->createInvoiceFromMaintenanceSchedules($contractSchedules, $contract);

                // Update schedules to link them to the invoice
                foreach ($contractSchedules as $schedule) {
                    $schedule->invoice_id = $invoice->id;
                    $schedule->save();
                }

                $success++;
            } catch (\Exception $e) {
                $failed++;
            }
        }

        return redirect()->route('auto-invoices.index')
            ->with('success', "تم إنشاء {$success} فاتورة بنجاح. فشل إنشاء {$failed} فاتورة.");
    }

    /**
     * Run the auto invoice generation process.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function runAutoGeneration(Request $request)
    {
        $request->validate([
            'type' => 'required|in:all,contracts,maintenance',
        ]);

        $type = $request->input('type');
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? null : $user->company_id;
        $stats = [];

        if ($type === 'all' || $type === 'contracts') {
            $contractStats = $this->autoInvoiceService->generateContractInvoices($companyId);
            $stats['contracts'] = $contractStats;
        }

        if ($type === 'all' || $type === 'maintenance') {
            $maintenanceStats = $this->autoInvoiceService->generateMaintenanceInvoices($companyId);
            $stats['maintenance'] = $maintenanceStats;
        }

        $totalSuccess = 0;
        $totalFailed = 0;

        if (isset($stats['contracts'])) {
            $totalSuccess += $stats['contracts']['success'];
            $totalFailed += $stats['contracts']['failed'];
        }

        if (isset($stats['maintenance'])) {
            $totalSuccess += $stats['maintenance']['success'];
            $totalFailed += $stats['maintenance']['failed'];
        }

        return redirect()->route('auto-invoices.index')
            ->with('success', "تم إنشاء {$totalSuccess} فاتورة بنجاح. فشل إنشاء {$totalFailed} فاتورة.");
    }
}
