<?php

namespace App\Http\Controllers;

use App\Models\ProjectTask;
use App\Models\ProjectPhase;
use App\Models\InstallationProject;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AllProjectTasksController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view-project-tasks', ['only' => ['index', 'show']]);
        $this->middleware('permission:create-project-tasks', ['only' => ['create', 'store']]);
        $this->middleware('permission:edit-project-tasks', ['only' => ['edit', 'update']]);
        $this->middleware('permission:delete-project-tasks', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of all project tasks across all projects.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = ProjectTask::with(['phase.installationProject', 'assignedTo']);

        // Filter by status if provided
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        // Filter by project if provided
        if ($request->has('project_id') && $request->project_id) {
            $query->whereHas('phase', function ($q) use ($request) {
                $q->where('installation_project_id', $request->project_id);
            });
        }

        // Filter by phase if provided
        if ($request->has('phase_id') && $request->phase_id) {
            $query->where('project_phase_id', $request->phase_id);
        }

        // Filter by assigned user if provided
        if ($request->has('assigned_to') && $request->assigned_to) {
            $query->where('assigned_to', $request->assigned_to);
        }

        $tasks = $query->orderBy('due_date', 'asc')->paginate(15);

        // Get all active projects for the filter dropdown
        $projects = InstallationProject::where('status', '!=', 'cancelled')->get();

        // Get all phases for the filter dropdown
        $phases = ProjectPhase::all();

        return view('installation-projects.tasks.all', compact('tasks', 'projects', 'phases'));
    }

    /**
     * Show the form for creating a new task.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // الحصول على المراحل من قاعدة البيانات مع معلومات المشروع
        $phases = ProjectPhase::with('installationProject')->get();

        // إذا كانت القائمة فارغة، نضيف بعض المراحل الافتراضية للاختبار
        if ($phases->isEmpty()) {
            $phases = collect([
                (object)[
                    'id' => 1,
                    'name' => 'مرحلة 1 للمشروع 1',
                    'installation_project_id' => 1
                ],
                (object)[
                    'id' => 2,
                    'name' => 'مرحلة 2 للمشروع 1',
                    'installation_project_id' => 1
                ],
                (object)[
                    'id' => 3,
                    'name' => 'مرحلة 1 للمشروع 2',
                    'installation_project_id' => 2
                ],
                (object)[
                    'id' => 4,
                    'name' => 'مرحلة 1 للمشروع 3',
                    'installation_project_id' => 3
                ]
            ]);
        }

        // التأكد من عدم وجود تكرار في المراحل عن طريق معرف المرحلة
        $phases = $phases->unique('id');

        return view('installation-projects.tasks.create_from_all', compact('phases'));
    }

    /**
     * Store a newly created task in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'project_id' => 'required',
            'project_phase_id' => 'required',
            'status' => 'required|in:not_started,in_progress,completed,delayed,cancelled',
            'priority' => 'required|in:low,medium,high,urgent',
            'due_date' => 'required|date',
            'estimated_hours' => 'nullable|numeric',
            'assigned_to' => 'nullable|exists:users,id',
        ]);

        // Verify that the selected phase belongs to the selected project
        $phase = ProjectPhase::findOrFail($request->project_phase_id);
        if ($phase->installation_project_id != $request->project_id) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['project_phase_id' => 'المرحلة المختارة لا تنتمي إلى المشروع المحدد']);
        }

        $installationProject = $phase->installationProject;

        $task = new ProjectTask();
        $task->name = $request->title;
        $task->description = $request->description;
        $task->project_phase_id = $request->project_phase_id;
        $task->status = $request->status;
        $task->priority = $request->priority;
        $task->start_date = now();
        $task->due_date = $request->due_date;
        $task->order_number = ProjectTask::where('project_phase_id', $request->project_phase_id)->max('order_number') + 1;
        $task->completion_percentage = 0;
        $task->assigned_to = $request->assigned_to;
        $task->save();

        return redirect()->route('all-project-tasks.index')
            ->with('success', 'تم إنشاء المهمة بنجاح');
    }

    /**
     * Display the specified task.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $task = ProjectTask::with(['phase.installationProject', 'assignedTo', 'creator'])->findOrFail($id);
        return view('installation-projects.tasks.show', compact('task'));
    }

    /**
     * Show the form for editing the specified task.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $task = ProjectTask::findOrFail($id);
        $phases = ProjectPhase::with('installationProject')->get();
        return view('installation-projects.tasks.edit', compact('task', 'phases'));
    }

    /**
     * Update the specified task in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'project_phase_id' => 'required|exists:project_phases,id',
            'status' => 'required|in:not_started,in_progress,completed,delayed,cancelled',
            'priority' => 'required|in:low,medium,high,urgent',
            'due_date' => 'required|date',
            'estimated_hours' => 'nullable|numeric',
            'assigned_to' => 'nullable|exists:users,id',
        ]);

        $task = ProjectTask::findOrFail($id);
        $task->name = $request->title;
        $task->description = $request->description;
        $task->project_phase_id = $request->project_phase_id;
        $task->status = $request->status;
        $task->priority = $request->priority;
        $task->due_date = $request->due_date;
        $task->assigned_to = $request->assigned_to;
        $task->save();

        return redirect()->route('all-project-tasks.index')
            ->with('success', 'تم تحديث المهمة بنجاح');
    }

    /**
     * Remove the specified task from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $task = ProjectTask::findOrFail($id);
        $task->delete();

        return redirect()->route('all-project-tasks.index')
            ->with('success', 'تم حذف المهمة بنجاح');
    }

    /**
     * Update the progress of a task.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateProgress(Request $request, $id)
    {
        $request->validate([
            'progress' => 'required|integer|min:0|max:100',
        ]);

        $task = ProjectTask::findOrFail($id);
        $task->progress = $request->progress;

        // If progress is 100%, mark task as completed
        if ($request->progress == 100) {
            $task->status = 'completed';
            $task->completed_at = now();
        } else {
            // If task was completed but progress is now less than 100%, revert to in_progress
            if ($task->status == 'completed') {
                $task->status = 'in_progress';
                $task->completed_at = null;
            }
        }

        $task->save();

        return redirect()->back()->with('success', 'تم تحديث تقدم المهمة بنجاح');
    }

    /**
     * Assign a task to a user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function assignTask(Request $request, $id)
    {
        $request->validate([
            'assigned_to' => 'required|exists:users,id',
        ]);

        $task = ProjectTask::findOrFail($id);
        $task->assigned_to = $request->assigned_to;
        $task->save();

        return redirect()->back()->with('success', 'تم تعيين المهمة للمستخدم بنجاح');
    }
}
