<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ActivityLogController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_activity_logs');
    }

    /**
     * عرض قائمة سجلات النشاطات.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = ActivityLog::query();

        // تطبيق الفلاتر
        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('action') && $request->action) {
            $query->where('action', $request->action);
        }

        if ($request->has('level') && $request->level) {
            $query->where('level', $request->level);
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->has('subject_type') && $request->subject_type) {
            $query->where('subject_type', $request->subject_type);
        }

        // الترتيب
        $query->orderBy('created_at', 'desc');

        // الحصول على البيانات
        $logs = $query->with('user')->paginate(20);

        // الحصول على قائمة المستخدمين للفلتر
        $users = User::orderBy('name')->get();

        // الحصول على قائمة الإجراءات للفلتر
        $actions = ActivityLog::distinct()->pluck('action');

        // الحصول على قائمة المستويات للفلتر
        $levels = ActivityLog::distinct()->pluck('level');

        // الحصول على قائمة أنواع الكائنات للفلتر
        $subjectTypes = ActivityLog::distinct()->pluck('subject_type');

        return view('admin.activity-logs.index', compact(
            'logs',
            'users',
            'actions',
            'levels',
            'subjectTypes'
        ));
    }

    /**
     * عرض تفاصيل سجل النشاط.
     *
     * @param \App\Models\ActivityLog $activityLog
     * @return \Illuminate\View\View
     */
    public function show(ActivityLog $activityLog)
    {
        return view('admin.activity-logs.show', compact('activityLog'));
    }

    /**
     * حذف سجل النشاط.
     *
     * @param \App\Models\ActivityLog $activityLog
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(ActivityLog $activityLog)
    {
        // التحقق من الصلاحية
        if (!Auth::user()->hasPermissionTo('delete_activity_logs')) {
            return back()->withErrors([
                'error' => 'ليس لديك صلاحية لحذف سجلات النشاطات.',
            ]);
        }

        $activityLog->delete();

        return redirect()->route('activity-logs.index')
            ->with('success', 'تم حذف سجل النشاط بنجاح.');
    }

    /**
     * حذف سجلات النشاطات القديمة.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deleteOld(Request $request)
    {
        // التحقق من الصلاحية
        if (!Auth::user()->hasPermissionTo('delete_activity_logs')) {
            return back()->withErrors([
                'error' => 'ليس لديك صلاحية لحذف سجلات النشاطات.',
            ]);
        }

        $request->validate([
            'days' => 'required|integer|min:1',
        ]);

        $date = now()->subDays($request->days);

        $count = ActivityLog::where('created_at', '<', $date)->delete();

        return redirect()->route('activity-logs.index')
            ->with('success', "تم حذف {$count} من سجلات النشاطات القديمة بنجاح.");
    }

    /**
     * تصدير سجلات النشاطات.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        // التحقق من الصلاحية
        if (!Auth::user()->hasPermissionTo('export_activity_logs')) {
            return back()->withErrors([
                'error' => 'ليس لديك صلاحية لتصدير سجلات النشاطات.',
            ]);
        }

        $query = ActivityLog::query();

        // تطبيق الفلاتر
        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('action') && $request->action) {
            $query->where('action', $request->action);
        }

        if ($request->has('level') && $request->level) {
            $query->where('level', $request->level);
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // الترتيب
        $query->orderBy('created_at', 'desc');

        // الحصول على البيانات
        $logs = $query->with('user')->get();

        // تصدير البيانات (يمكن استخدام مكتبة مثل Maatwebsite/Laravel-Excel)
        // return Excel::download(new ActivityLogsExport($logs), 'activity_logs.xlsx');

        // بديل مؤقت: تحويل البيانات إلى CSV
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="activity_logs.csv"',
        ];

        $columns = ['ID', 'المستخدم', 'الإجراء', 'الوصف', 'المستوى', 'عنوان IP', 'التاريخ'];

        $callback = function () use ($logs, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($logs as $log) {
                fputcsv($file, [
                    $log->id,
                    $log->user ? $log->user->name : 'غير مسجل',
                    $log->action,
                    $log->description,
                    $log->level,
                    $log->ip_address,
                    $log->created_at->format('Y-m-d H:i:s'),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
