<?php

namespace App\Http\Controllers;

use App\Models\AccountingPeriod;
use App\Models\FiscalYear;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class AccountingPeriodController extends Controller
{
    /**
     * عرض قائمة الفترات المحاسبية
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $query = AccountingPeriod::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->whereHas('fiscalYear', function ($q) {
                $q->where('company_id', Auth::user()->company_id);
            });
        }

        $periods = $query->with('fiscalYear')
            ->orderBy('start_date', 'desc')
            ->paginate(15);

        return view('accounting.accounting_periods.index', compact('periods'));
    }

    /**
     * عرض نموذج إنشاء فترة محاسبية جديدة
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $query = FiscalYear::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->where('company_id', Auth::user()->company_id);
        }

        $fiscalYears = $query->where('is_closed', false)
            ->orderBy('start_date', 'desc')
            ->get();

        if ($fiscalYears->isEmpty()) {
            return redirect()->route('fiscal-years.index')
                ->with('error', 'لا توجد سنوات مالية مفتوحة. يرجى إنشاء سنة مالية أولاً.');
        }

        return view('accounting.accounting_periods.create', compact('fiscalYears'));
    }

    /**
     * حفظ فترة محاسبية جديدة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'fiscal_year_id' => 'required|exists:fiscal_years,id',
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من أن السنة المالية تنتمي للشركة الحالية وغير مغلقة
        $fyQuery = FiscalYear::where('id', $request->fiscal_year_id);
        if (!Auth::user()->isSystemAdmin()) {
            $fyQuery->where('company_id', Auth::user()->company_id);
        }
        $fiscalYear = $fyQuery->where('is_closed', false)
            ->firstOrFail();

        // التحقق من أن الفترة المحاسبية ضمن نطاق السنة المالية
        if ($request->start_date < $fiscalYear->start_date || $request->end_date > $fiscalYear->end_date) {
            return redirect()->back()
                ->with('error', 'الفترة المحاسبية يجب أن تكون ضمن نطاق السنة المالية')
                ->withInput();
        }

        // التحقق من عدم تداخل الفترات المحاسبية
        $overlapping = AccountingPeriod::where('fiscal_year_id', $request->fiscal_year_id)
            ->where(function ($query) use ($request) {
                $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                    ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                    ->orWhere(function ($q) use ($request) {
                        $q->where('start_date', '<=', $request->start_date)
                            ->where('end_date', '>=', $request->end_date);
                    });
            })
            ->exists();

        if ($overlapping) {
            return redirect()->back()
                ->with('error', 'هناك تداخل مع فترة محاسبية أخرى في نفس النطاق')
                ->withInput();
        }

        $period = new AccountingPeriod();
        $period->fiscal_year_id = $request->fiscal_year_id;
        $period->company_id = $fiscalYear->company_id;
        $period->name = $request->name;
        $period->start_date = $request->start_date;
        $period->end_date = $request->end_date;
        $period->is_closed = false;
        $period->save();

        return redirect()->route('accounting-periods.index')
            ->with('success', 'تم إنشاء الفترة المحاسبية بنجاح');
    }

    /**
     * عرض تفاصيل فترة محاسبية محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $query = AccountingPeriod::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->whereHas('fiscalYear', function ($q) {
                $q->where('company_id', Auth::user()->company_id);
            });
        }

        $period = $query->with(['fiscalYear', 'journalEntries' => function ($q) {
            $q->orderBy('entry_date', 'desc')->take(15);
        }])
            ->findOrFail($id);

        return view('accounting.accounting_periods.show', compact('period'));
    }

    /**
     * عرض نموذج تعديل فترة محاسبية
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $query = AccountingPeriod::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->whereHas('fiscalYear', function ($q) {
                $q->where('company_id', Auth::user()->company_id);
            });
        }

        $period = $query->with('fiscalYear')
            ->findOrFail($id);

        // التحقق من أن الفترة المحاسبية غير مغلقة
        if ($period->is_closed) {
            return redirect()->route('accounting-periods.show', $period->id)
                ->with('error', 'لا يمكن تعديل الفترة المحاسبية لأنها مغلقة');
        }

        $fyQuery = FiscalYear::query();
        if (!Auth::user()->isSystemAdmin()) {
            $fyQuery->where('company_id', Auth::user()->company_id);
        }

        $fiscalYears = $fyQuery->where('is_closed', false)
            ->orderBy('start_date', 'desc')
            ->get();

        return view('accounting.accounting_periods.edit', compact('period', 'fiscalYears'));
    }

    /**
     * تحديث فترة محاسبية محددة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $query = AccountingPeriod::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->whereHas('fiscalYear', function ($q) {
                $q->where('company_id', Auth::user()->company_id);
            });
        }

        $period = $query->findOrFail($id);

        // التحقق من أن الفترة المحاسبية غير مغلقة
        if ($period->is_closed) {
            return redirect()->route('accounting-periods.show', $period->id)
                ->with('error', 'لا يمكن تعديل الفترة المحاسبية لأنها مغلقة');
        }

        $validator = Validator::make($request->all(), [
            'fiscal_year_id' => 'required|exists:fiscal_years,id',
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من أن السنة المالية تنتمي للشركة الحالية وغير مغلقة
        $fyQuery = FiscalYear::where('id', $request->fiscal_year_id);
        if (!Auth::user()->isSystemAdmin()) {
            $fyQuery->where('company_id', Auth::user()->company_id);
        }
        $fiscalYear = $fyQuery->where('is_closed', false)
            ->firstOrFail();

        // التحقق من أن الفترة المحاسبية ضمن نطاق السنة المالية
        if ($request->start_date < $fiscalYear->start_date || $request->end_date > $fiscalYear->end_date) {
            return redirect()->back()
                ->with('error', 'الفترة المحاسبية يجب أن تكون ضمن نطاق السنة المالية')
                ->withInput();
        }

        // التحقق من عدم تداخل الفترات المحاسبية
        $overlapping = AccountingPeriod::where('fiscal_year_id', $request->fiscal_year_id)
            ->where('id', '!=', $id)
            ->where(function ($query) use ($request) {
                $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                    ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                    ->orWhere(function ($q) use ($request) {
                        $q->where('start_date', '<=', $request->start_date)
                            ->where('end_date', '>=', $request->end_date);
                    });
            })
            ->exists();

        if ($overlapping) {
            return redirect()->back()
                ->with('error', 'هناك تداخل مع فترة محاسبية أخرى في نفس النطاق')
                ->withInput();
        }

        $period->fiscal_year_id = $request->fiscal_year_id;
        $period->name = $request->name;
        $period->start_date = $request->start_date;
        $period->end_date = $request->end_date;
        $period->save();

        return redirect()->route('accounting-periods.index')
            ->with('success', 'تم تحديث الفترة المحاسبية بنجاح');
    }

    /**
     * إغلاق فترة محاسبية محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function close($id)
    {
        $query = AccountingPeriod::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->whereHas('fiscalYear', function ($q) {
                $q->where('company_id', Auth::user()->company_id);
            });
        }

        $period = $query->findOrFail($id);

        // التحقق من أن الفترة المحاسبية غير مغلقة بالفعل
        if ($period->is_closed) {
            return redirect()->route('accounting-periods.show', $period->id)
                ->with('error', 'الفترة المحاسبية مغلقة بالفعل');
        }

        try {
            $period->close();
            return redirect()->route('accounting-periods.show', $period->id)
                ->with('success', 'تم إغلاق الفترة المحاسبية بنجاح');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إغلاق الفترة المحاسبية: ' . $e->getMessage());
        }
    }

    /**
     * إعادة فتح فترة محاسبية محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reopen($id)
    {
        $query = AccountingPeriod::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->whereHas('fiscalYear', function ($q) {
                $q->where('company_id', Auth::user()->company_id);
            });
        }

        $period = $query->findOrFail($id);

        // التحقق من أن الفترة المحاسبية مغلقة
        if (!$period->is_closed) {
            return redirect()->route('accounting-periods.show', $period->id)
                ->with('error', 'الفترة المحاسبية مفتوحة بالفعل');
        }

        // التحقق من أن السنة المالية غير مغلقة
        if ($period->fiscalYear->is_closed) {
            return redirect()->route('accounting-periods.show', $period->id)
                ->with('error', 'لا يمكن إعادة فتح الفترة المحاسبية لأن السنة المالية مغلقة');
        }

        $period->is_closed = false;
        $period->save();

        return redirect()->route('accounting-periods.show', $period->id)
            ->with('success', 'تم إعادة فتح الفترة المحاسبية بنجاح');
    }
}
