<?php

namespace App\Http\Controllers;

use App\Models\ChartOfAccount;
use App\Models\JournalEntryItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AccountLedgerController extends Controller
{
    /**
     * عرض صفحة البحث في دفتر الأستاذ
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $query = ChartOfAccount::query();

        if (!Auth::user()->isSystemAdmin()) {
            $query->where('company_id', Auth::user()->company_id);
        }

        $accounts = $query->where('is_active', true)
            ->orderBy('account_code')
            ->get();

        return view('accounting.ledger.index', compact('accounts'));
    }

    /**
     * عرض كشف حساب (دفتر الأستاذ) لحساب محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:chart_of_accounts,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $accountId = $request->account_id;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        // الحصول على معلومات الحساب
        $accountQuery = ChartOfAccount::query();
        if (!Auth::user()->isSystemAdmin()) {
            $accountQuery->where('company_id', Auth::user()->company_id);
        }
        $account = $accountQuery->findOrFail($accountId);

        // حساب الرصيد الافتتاحي (الرصيد قبل تاريخ البداية)
        $openingBalance = $this->calculateOpeningBalance($account, $startDate);

        // الحصول على حركات الحساب خلال الفترة المحددة
        $transactions = JournalEntryItem::where('account_id', $accountId)
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                if (!Auth::user()->isSystemAdmin()) {
                    $query->where('company_id', Auth::user()->company_id);
                }
                $query->whereBetween('entry_date', [$startDate, $endDate])
                    ->orderBy('entry_date');
            })
            ->with(['journalEntry' => function ($query) {
                $query->with(['cashTransactions']);
            }])
            ->get();

        // ترتيب الحركات حسب التاريخ
        $transactions = $transactions->sortBy(function ($transaction) {
            return $transaction->journalEntry->entry_date->format('Y-m-d') . $transaction->journalEntry->id;
        });

        // حساب الرصيد التراكمي
        $runningBalance = $openingBalance;
        foreach ($transactions as $transaction) {
            if (in_array($account->account_type, ['asset', 'expense'])) {
                // الأصول والمصروفات تزيد بالمدين وتنقص بالدائن
                $runningBalance += $transaction->debit - $transaction->credit;
            } else {
                // الخصوم والإيرادات وحقوق الملكية تزيد بالدائن وتنقص بالمدين
                $runningBalance += $transaction->credit - $transaction->debit;
            }
            $transaction->running_balance = $runningBalance;
        }

        // حساب إجماليات الفترة
        $periodTotals = [
            'debit' => $transactions->sum('debit'),
            'credit' => $transactions->sum('credit'),
            'opening_balance' => $openingBalance,
            'closing_balance' => $runningBalance
        ];

        return view('accounting.ledger.show', compact('account', 'transactions', 'periodTotals', 'startDate', 'endDate', 'openingBalance'));
    }

    /**
     * حساب الرصيد الافتتاحي للحساب قبل تاريخ البداية
     *
     * @param  \App\Models\ChartOfAccount  $account
     * @param  string  $startDate
     * @return float
     */
    private function calculateOpeningBalance(ChartOfAccount $account, $startDate)
    {
        // الرصيد الافتتاحي للحساب
        $openingBalance = $account->opening_balance;

        // حساب الحركات قبل تاريخ البداية
        $previousTransactions = JournalEntryItem::where('account_id', $account->id)
            ->whereHas('journalEntry', function ($query) use ($startDate) {
                if (!Auth::user()->isSystemAdmin()) {
                    $query->where('company_id', Auth::user()->company_id);
                }
                $query->where('entry_date', '<', $startDate);
            })
            ->get();

        // حساب الرصيد بناءً على نوع الحساب
        if (in_array($account->account_type, ['asset', 'expense'])) {
            // الأصول والمصروفات تزيد بالمدين وتنقص بالدائن
            $openingBalance += $previousTransactions->sum('debit') - $previousTransactions->sum('credit');
        } else {
            // الخصوم والإيرادات وحقوق الملكية تزيد بالدائن وتنقص بالمدين
            $openingBalance += $previousTransactions->sum('credit') - $previousTransactions->sum('debit');
        }

        return $openingBalance;
    }

    /**
     * طباعة كشف حساب
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function print(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:chart_of_accounts,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $accountId = $request->account_id;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        // الحصول على معلومات الحساب
        $accountQuery = ChartOfAccount::query();
        if (!Auth::user()->isSystemAdmin()) {
            $accountQuery->where('company_id', Auth::user()->company_id);
        }
        $account = $accountQuery->findOrFail($accountId);

        // حساب الرصيد الافتتاحي (الرصيد قبل تاريخ البداية)
        $openingBalance = $this->calculateOpeningBalance($account, $startDate);

        // الحصول على حركات الحساب خلال الفترة المحددة
        $transactions = JournalEntryItem::where('account_id', $accountId)
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                if (!Auth::user()->isSystemAdmin()) {
                    $query->where('company_id', Auth::user()->company_id);
                }
                $query->whereBetween('entry_date', [$startDate, $endDate])
                    ->orderBy('entry_date');
            })
            ->with(['journalEntry' => function ($query) {
                $query->with(['cashTransactions']);
            }])
            ->get();

        // ترتيب الحركات حسب التاريخ
        $transactions = $transactions->sortBy(function ($transaction) {
            return $transaction->journalEntry->entry_date->format('Y-m-d') . $transaction->journalEntry->id;
        });

        // حساب الرصيد التراكمي
        $runningBalance = $openingBalance;
        foreach ($transactions as $transaction) {
            if (in_array($account->account_type, ['asset', 'expense'])) {
                // الأصول والمصروفات تزيد بالمدين وتنقص بالدائن
                $runningBalance += $transaction->debit - $transaction->credit;
            } else {
                // الخصوم والإيرادات وحقوق الملكية تزيد بالدائن وتنقص بالمدين
                $runningBalance += $transaction->credit - $transaction->debit;
            }
            $transaction->running_balance = $runningBalance;
        }

        // حساب إجماليات الفترة
        $periodTotals = [
            'debit' => $transactions->sum('debit'),
            'credit' => $transactions->sum('credit'),
            'opening_balance' => $openingBalance,
            'closing_balance' => $runningBalance
        ];

        return view('accounting.ledger.print', compact('account', 'transactions', 'periodTotals', 'startDate', 'endDate', 'openingBalance'));
    }
}
