<?php

namespace App\Http\Controllers;

use App\Models\ChartOfAccount;
use App\Models\JournalEntryItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\AccountBalanceExport;

class AccountBalanceReportController extends Controller
{
    /**
     * عرض تقرير أرصدة الحسابات
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        if (!$companyId && !$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
        }

        // الحصول على الحسابات الفرعية فقط (التي ليس لها حسابات فرعية)
        $query = ChartOfAccount::query();
        
        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $companyId);
        }

        $accounts = $query->where('is_active', true)
            ->where('is_parent', false)
            ->orderBy('account_code')
            ->get();

        // الحصول على تاريخ آخر حركة لكل حساب
        foreach ($accounts as $account) {
            // الحصول على آخر حركة للحساب
            $lastMovement = JournalEntryItem::where('account_id', $account->id)
                ->join('journal_entries', 'journal_entries.id', '=', 'journal_entry_items.journal_entry_id')
                ->where('journal_entries.is_posted', true)
                ->orderBy('journal_entries.entry_date', 'desc')
                ->select('journal_entries.entry_date')
                ->first();

            $account->last_movement_date = $lastMovement ? $lastMovement->entry_date : null;

            // تحديد نوع الرصيد (مدين/دائن) بناءً على نوع الحساب
            $balance = $account->current_balance;
            $account->balance_type = '';

            if (in_array($account->account_type, ['asset', 'expense'])) {
                // الأصول والمصروفات: الرصيد الموجب يكون مدين، والسالب يكون دائن
                if ($balance > 0) {
                    $account->balance_type = 'مدين';
                } elseif ($balance < 0) {
                    $account->balance_type = 'دائن';
                    $account->display_balance = abs($balance);
                } else {
                    $account->display_balance = 0;
                }
            } else {
                // الخصوم والإيرادات وحقوق الملكية: الرصيد الموجب يكون دائن، والسالب يكون مدين
                if ($balance > 0) {
                    $account->balance_type = 'دائن';
                } elseif ($balance < 0) {
                    $account->balance_type = 'مدين';
                    $account->display_balance = abs($balance);
                } else {
                    $account->display_balance = 0;
                }
            }

            // إذا لم يتم تعيين display_balance، استخدم القيمة الأصلية
            if (!isset($account->display_balance)) {
                $account->display_balance = $balance;
            }
        }

        // حساب إجمالي الأرصدة المدينة والدائنة
        $totalDebit = 0;
        $totalCredit = 0;

        foreach ($accounts as $account) {
            if ($account->balance_type == 'مدين' && $account->display_balance != 0) {
                $totalDebit += $account->display_balance;
            } elseif ($account->balance_type == 'دائن' && $account->display_balance != 0) {
                $totalCredit += $account->display_balance;
            }
        }

        return view('accounting.reports.account_balance', compact('accounts', 'totalDebit', 'totalCredit'));
    }

    /**
     * تصدير تقرير أرصدة الحسابات إلى ملف Excel
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function export(Request $request)
    {
        return Excel::download(new AccountBalanceExport, 'account_balances.xlsx');
    }
}
