<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\MaintenanceSchedule;
use App\Models\MaintenanceContract;
use App\Models\Elevator;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class MaintenanceScheduleController extends Controller
{
    /**
     * خدمة الإشعارات
     *
     * @var NotificationService
     */
    protected $notificationService;

    /**
     * إنشاء مثيل جديد من وحدة التحكم
     *
     * @param NotificationService $notificationService
     */
    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * عرض قائمة جدولة الصيانة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $query = MaintenanceSchedule::with(['elevator', 'maintenanceContract', 'user']);

        // تصفية حسب المصعد
        if ($request->has('elevator_id')) {
            $query->where('elevator_id', $request->elevator_id);
        }

        // تصفية حسب العقد
        if ($request->has('maintenance_contract_id')) {
            $query->where('maintenance_contract_id', $request->maintenance_contract_id);
        }

        // تصفية حسب الحالة
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // تصفية حسب التاريخ
        if ($request->has('from_date') && $request->has('to_date')) {
            $query->whereBetween('scheduled_date', [$request->from_date, $request->to_date]);
        } elseif ($request->has('from_date')) {
            $query->where('scheduled_date', '>=', $request->from_date);
        } elseif ($request->has('to_date')) {
            $query->where('scheduled_date', '<=', $request->to_date);
        }

        // ترتيب النتائج
        $query->orderBy('scheduled_date', 'asc');

        // تقسيم الصفحات
        $perPage = $request->has('per_page') ? (int) $request->per_page : 15;
        $maintenanceSchedules = $query->paginate($perPage);

        return response()->json([
            'status' => 'success',
            'data' => $maintenanceSchedules,
        ]);
    }

    /**
     * إنشاء جدولة صيانة جديدة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'maintenance_contract_id' => 'required|exists:maintenance_contracts,id',
            'elevator_id' => 'required|exists:elevators,id',
            'scheduled_date' => 'required|date',
            'status' => 'required|in:scheduled,completed,missed,cancelled',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        // التحقق من وجود العقد والمصعد
        $contract = MaintenanceContract::findOrFail($request->maintenance_contract_id);
        $elevator = Elevator::findOrFail($request->elevator_id);

        // التحقق من أن المصعد مرتبط بالعقد
        $isElevatorInContract = $contract->elevators()->where('elevators.id', $elevator->id)->exists();
        if (!$isElevatorInContract) {
            return response()->json([
                'status' => 'error',
                'message' => 'المصعد غير مرتبط بهذا العقد',
            ], 422);
        }

        // إنشاء جدولة الصيانة
        $maintenanceSchedule = new MaintenanceSchedule();
        $maintenanceSchedule->maintenance_contract_id = $request->maintenance_contract_id;
        $maintenanceSchedule->elevator_id = $request->elevator_id;
        $maintenanceSchedule->scheduled_date = $request->scheduled_date;
        $maintenanceSchedule->status = $request->status;
        $maintenanceSchedule->notes = $request->notes;
        $maintenanceSchedule->company_id = Auth::user()->company_id;
        $maintenanceSchedule->user_id = Auth::id();
        $maintenanceSchedule->save();

        // إنشاء إشعارات للصيانة المستحقة
        $this->notificationService->checkUpcomingMaintenanceSchedules();

        return response()->json([
            'status' => 'success',
            'message' => 'تم إنشاء جدولة الصيانة بنجاح',
            'data' => $maintenanceSchedule->load(['elevator', 'maintenanceContract', 'user']),
        ], 201);
    }

    /**
     * عرض جدولة صيانة محددة
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $maintenanceSchedule = MaintenanceSchedule::with(['elevator', 'maintenanceContract', 'user', 'maintenanceLogs'])->findOrFail($id);

        return response()->json([
            'status' => 'success',
            'data' => $maintenanceSchedule,
        ]);
    }

    /**
     * تحديث جدولة صيانة محددة
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'scheduled_date' => 'nullable|date',
            'status' => 'nullable|in:scheduled,completed,missed,cancelled',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $maintenanceSchedule = MaintenanceSchedule::findOrFail($id);

        // تحديث البيانات
        if ($request->has('scheduled_date')) {
            $maintenanceSchedule->scheduled_date = $request->scheduled_date;
        }
        if ($request->has('status')) {
            $maintenanceSchedule->status = $request->status;
        }
        if ($request->has('notes')) {
            $maintenanceSchedule->notes = $request->notes;
        }

        $maintenanceSchedule->save();

        // تحديث الإشعارات إذا تم تغيير الحالة إلى مكتمل
        if ($request->has('status') && $request->status === 'completed') {
            $this->notificationService->checkUpcomingMaintenanceSchedules();
        }

        return response()->json([
            'status' => 'success',
            'message' => 'تم تحديث جدولة الصيانة بنجاح',
            'data' => $maintenanceSchedule->load(['elevator', 'maintenanceContract', 'user']),
        ]);
    }

    /**
     * حذف جدولة صيانة محددة
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $maintenanceSchedule = MaintenanceSchedule::findOrFail($id);

        // التحقق من الصلاحيات
        if (!Auth::user()->hasRole(['Company Admin', 'Branch Manager'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'غير مصرح لك بحذف هذه الجدولة',
            ], 403);
        }

        $maintenanceSchedule->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'تم حذف جدولة الصيانة بنجاح',
        ]);
    }

    /**
     * إنشاء جدولة صيانة دورية لعقد
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function generateScheduleForContract(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'maintenance_contract_id' => 'required|exists:maintenance_contracts,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $contract = MaintenanceContract::with('elevators')->findOrFail($request->maintenance_contract_id);
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $visitsPerYear = $contract->visits_per_year;

        if (!$visitsPerYear || $visitsPerYear <= 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'عدد الزيارات السنوية غير محدد في العقد',
            ], 422);
        }

        // حساب الفاصل الزمني بين الزيارات بالأيام
        $intervalDays = floor(365 / $visitsPerYear);
        $createdCount = 0;
        $currentDate = $startDate->copy();

        // إنشاء جدولة لكل مصعد في العقد
        foreach ($contract->elevators as $elevator) {
            $visitDate = $currentDate->copy();

            while ($visitDate->lte($endDate)) {
                // التحقق من عدم وجود جدولة مسبقة لنفس المصعد في نفس اليوم
                $existingSchedule = MaintenanceSchedule::where('maintenance_contract_id', $contract->id)
                    ->where('elevator_id', $elevator->id)
                    ->whereDate('scheduled_date', $visitDate->toDateString())
                    ->first();

                if (!$existingSchedule) {
                    $schedule = new MaintenanceSchedule();
                    $schedule->maintenance_contract_id = $contract->id;
                    $schedule->elevator_id = $elevator->id;
                    $schedule->scheduled_date = $visitDate->toDateString();
                    $schedule->status = 'scheduled';
                    $schedule->company_id = Auth::user()->company_id;
                    $schedule->user_id = Auth::id();
                    $schedule->save();
                    $createdCount++;
                }

                // الانتقال إلى تاريخ الزيارة التالية
                $visitDate->addDays($intervalDays);
            }
        }

        // إنشاء إشعارات للصيانة المستحقة
        $this->notificationService->checkUpcomingMaintenanceSchedules();

        return response()->json([
            'status' => 'success',
            'message' => "تم إنشاء {$createdCount} جدولة صيانة بنجاح",
        ]);
    }

    /**
     * الحصول على جدولة الصيانة للشهر الحالي
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMonthlySchedule(Request $request)
    {
        $month = $request->has('month') ? $request->month : Carbon::now()->month;
        $year = $request->has('year') ? $request->year : Carbon::now()->year;

        $startDate = Carbon::createFromDate($year, $month, 1)->startOfMonth();
        $endDate = $startDate->copy()->endOfMonth();

        $schedules = MaintenanceSchedule::with(['elevator', 'maintenanceContract'])
            ->whereBetween('scheduled_date', [$startDate, $endDate])
            ->orderBy('scheduled_date')
            ->get();

        // تنظيم البيانات حسب التاريخ
        $calendarData = [];
        foreach ($schedules as $schedule) {
            $date = $schedule->scheduled_date->format('Y-m-d');
            if (!isset($calendarData[$date])) {
                $calendarData[$date] = [];
            }
            $calendarData[$date][] = $schedule;
        }

        return response()->json([
            'status' => 'success',
            'data' => $calendarData,
            'month' => $month,
            'year' => $year,
            'start_date' => $startDate->format('Y-m-d'),
            'end_date' => $endDate->format('Y-m-d'),
        ]);
    }
}
