<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\MaintenanceLog;
use App\Models\MaintenanceSchedule;
use App\Models\Elevator;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class MaintenanceLogController extends Controller
{
    /**
     * عرض قائمة سجلات الصيانة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $query = MaintenanceLog::with(['elevator', 'technician', 'maintenanceSchedule']);

        // تصفية حسب المصعد
        if ($request->has('elevator_id')) {
            $query->where('elevator_id', $request->elevator_id);
        }

        // تصفية حسب الفني
        if ($request->has('technician_id')) {
            $query->where('technician_id', $request->technician_id);
        }

        // تصفية حسب جدولة الصيانة
        if ($request->has('maintenance_schedule_id')) {
            $query->where('maintenance_schedule_id', $request->maintenance_schedule_id);
        }

        // تصفية حسب الحالة
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // تصفية حسب النوع
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // تصفية حسب التاريخ
        if ($request->has('from_date') && $request->has('to_date')) {
            $query->whereBetween('maintenance_date', [$request->from_date, $request->to_date]);
        } elseif ($request->has('from_date')) {
            $query->where('maintenance_date', '>=', $request->from_date);
        } elseif ($request->has('to_date')) {
            $query->where('maintenance_date', '<=', $request->to_date);
        }

        // ترتيب النتائج
        $query->orderBy('maintenance_date', 'desc');

        // تقسيم الصفحات
        $perPage = $request->has('per_page') ? (int) $request->per_page : 15;
        $maintenanceLogs = $query->paginate($perPage);

        return response()->json([
            'status' => 'success',
            'data' => $maintenanceLogs,
        ]);
    }

    /**
     * إنشاء سجل صيانة جديد
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'elevator_id' => 'required|exists:elevators,id',
            'maintenance_schedule_id' => 'nullable|exists:maintenance_schedules,id',
            'maintenance_date' => 'required|date',
            'status' => 'required|in:completed,partial,failed',
            'issues_found' => 'nullable|string',
            'actions_taken' => 'nullable|string',
            'parts_replaced' => 'nullable|string',
            'next_maintenance_recommendation' => 'nullable|string',
            'signature' => 'nullable|string', // Base64 encoded signature
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        // التحقق من وجود المصعد
        $elevator = Elevator::findOrFail($request->elevator_id);

        // إنشاء سجل الصيانة
        $maintenanceLog = new MaintenanceLog();
        $maintenanceLog->elevator_id = $request->elevator_id;
        $maintenanceLog->maintenance_schedule_id = $request->maintenance_schedule_id;
        $maintenanceLog->maintenance_date = $request->maintenance_date;
        $maintenanceLog->technician_id = Auth::id();
        $maintenanceLog->status = $request->status;
        $maintenanceLog->issues_found = $request->issues_found;
        $maintenanceLog->actions_taken = $request->actions_taken;
        $maintenanceLog->parts_replaced = $request->parts_replaced;
        $maintenanceLog->next_maintenance_recommendation = $request->next_maintenance_recommendation;
        $maintenanceLog->company_id = Auth::user()->company_id;

        // حفظ التوقيع إذا تم تقديمه
        if ($request->has('signature') && !empty($request->signature)) {
            $signatureData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $request->signature));
            $signaturePath = 'signatures/' . uniqid() . '.png';
            Storage::disk('public')->put($signaturePath, $signatureData);
            $maintenanceLog->signature_path = $signaturePath;
        }

        $maintenanceLog->save();

        // تحديث حالة جدولة الصيانة إذا كانت موجودة
        if ($request->maintenance_schedule_id) {
            $schedule = MaintenanceSchedule::findOrFail($request->maintenance_schedule_id);
            $schedule->status = 'completed';
            $schedule->save();

            // تحديث الإشعارات المرتبطة بجدولة الصيانة
            Notification::where('notifiable_type', 'App\\Models\\MaintenanceSchedule')
                ->where('notifiable_id', $schedule->id)
                ->where('type', 'maintenance_due')
                ->update(['status' => 'read', 'read_at' => now()]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'تم إنشاء سجل الصيانة بنجاح',
            'data' => $maintenanceLog->load(['elevator', 'technician', 'maintenanceSchedule']),
        ], 201);
    }

    /**
     * عرض سجل صيانة محدد
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $maintenanceLog = MaintenanceLog::with(['elevator', 'technician', 'maintenanceSchedule'])->findOrFail($id);

        return response()->json([
            'status' => 'success',
            'data' => $maintenanceLog,
        ]);
    }

    /**
     * تحديث سجل صيانة محدد
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'nullable|in:completed,partial,failed',
            'issues_found' => 'nullable|string',
            'actions_taken' => 'nullable|string',
            'parts_replaced' => 'nullable|string',
            'next_maintenance_recommendation' => 'nullable|string',
            'signature' => 'nullable|string', // Base64 encoded signature
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $maintenanceLog = MaintenanceLog::findOrFail($id);

        // التحقق من الصلاحيات
        if ($maintenanceLog->technician_id !== Auth::id() && !Auth::user()->hasRole(['Company Admin', 'Branch Manager'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'غير مصرح لك بتحديث هذا السجل',
            ], 403);
        }

        // تحديث البيانات
        if ($request->has('status')) {
            $maintenanceLog->status = $request->status;
        }
        if ($request->has('issues_found')) {
            $maintenanceLog->issues_found = $request->issues_found;
        }
        if ($request->has('actions_taken')) {
            $maintenanceLog->actions_taken = $request->actions_taken;
        }
        if ($request->has('parts_replaced')) {
            $maintenanceLog->parts_replaced = $request->parts_replaced;
        }
        if ($request->has('next_maintenance_recommendation')) {
            $maintenanceLog->next_maintenance_recommendation = $request->next_maintenance_recommendation;
        }

        // حفظ التوقيع الجديد إذا تم تقديمه
        if ($request->has('signature') && !empty($request->signature)) {
            // حذف التوقيع القديم إذا كان موجودًا
            if ($maintenanceLog->signature_path) {
                Storage::disk('public')->delete($maintenanceLog->signature_path);
            }

            $signatureData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $request->signature));
            $signaturePath = 'signatures/' . uniqid() . '.png';
            Storage::disk('public')->put($signaturePath, $signatureData);
            $maintenanceLog->signature_path = $signaturePath;
        }

        $maintenanceLog->save();

        return response()->json([
            'status' => 'success',
            'message' => 'تم تحديث سجل الصيانة بنجاح',
            'data' => $maintenanceLog->load(['elevator', 'technician', 'maintenanceSchedule']),
        ]);
    }

    /**
     * حذف سجل صيانة محدد
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $maintenanceLog = MaintenanceLog::findOrFail($id);

        // التحقق من الصلاحيات
        if (!Auth::user()->hasRole(['Company Admin', 'Branch Manager'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'غير مصرح لك بحذف هذا السجل',
            ], 403);
        }

        // حذف التوقيع إذا كان موجودًا
        if ($maintenanceLog->signature_path) {
            Storage::disk('public')->delete($maintenanceLog->signature_path);
        }

        $maintenanceLog->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'تم حذف سجل الصيانة بنجاح',
        ]);
    }

    /**
     * إحصائيات سجلات الصيانة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function statistics(Request $request)
    {
        $query = MaintenanceLog::query();

        // تصفية حسب المصعد
        if ($request->has('elevator_id')) {
            $query->where('elevator_id', $request->elevator_id);
        }

        // تصفية حسب الفترة الزمنية
        if ($request->has('from_date') && $request->has('to_date')) {
            $query->whereBetween('maintenance_date', [$request->from_date, $request->to_date]);
        } elseif ($request->has('from_date')) {
            $query->where('maintenance_date', '>=', $request->from_date);
        } elseif ($request->has('to_date')) {
            $query->where('maintenance_date', '<=', $request->to_date);
        }

        // إحصائيات حسب الحالة
        $statusStats = $query->selectRaw('status, count(*) as count')
            ->groupBy('status')
            ->get();

        // إحصائيات حسب الفني
        $technicianStats = $query->selectRaw('technician_id, count(*) as count')
            ->groupBy('technician_id')
            ->with('technician:id,name')
            ->get();

        // إحصائيات حسب المصعد
        $elevatorStats = $query->selectRaw('elevator_id, count(*) as count')
            ->groupBy('elevator_id')
            ->with('elevator:id,serial_number,model')
            ->get();

        return response()->json([
            'status' => 'success',
            'data' => [
                'status_stats' => $statusStats,
                'technician_stats' => $technicianStats,
                'elevator_stats' => $elevatorStats,
                'total_count' => $query->count(),
            ],
        ]);
    }
}
