<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Services\MaintenanceFinancialService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class MaintenanceFinancialReportController extends Controller
{
    /**
     * خدمة التقارير المالية للصيانة.
     *
     * @var \App\Services\MaintenanceFinancialService
     */
    protected $maintenanceFinancialService;

    /**
     * إنشاء مثيل جديد للتحكم.
     *
     * @param  \App\Services\MaintenanceFinancialService  $maintenanceFinancialService
     * @return void
     */
    public function __construct(MaintenanceFinancialService $maintenanceFinancialService)
    {
        $this->maintenanceFinancialService = $maintenanceFinancialService;
    }

    /**
     * الحصول على التقرير المالي للصيانة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFinancialReport(Request $request)
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->can('view_financial_reports')) {
            return response()->json(['error' => 'غير مصرح لك بعرض التقارير المالية'], 403);
        }

        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // تحديد الشركة
        $companyId = auth()->user()->company_id;

        // تحديد الفترة الزمنية
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);

        // الحصول على التقرير
        $report = $this->maintenanceFinancialService->getMaintenanceFinancialReport(
            $companyId,
            $startDate,
            $endDate
        );

        return response()->json($report);
    }

    /**
     * الحصول على تقرير أداء الفنيين.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getTechnicianPerformanceReport(Request $request)
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->can('view_technician_reports')) {
            return response()->json(['error' => 'غير مصرح لك بعرض تقارير الفنيين'], 403);
        }

        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // تحديد الشركة
        $companyId = auth()->user()->company_id;

        // تحديد الفترة الزمنية
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);

        // الحصول على التقرير
        $report = $this->maintenanceFinancialService->getTechnicianPerformanceReport(
            $companyId,
            $startDate,
            $endDate
        );

        return response()->json($report);
    }

    /**
     * إنشاء فواتير تلقائية من جداول الصيانة المكتملة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function generateInvoicesFromMaintenance(Request $request)
    {
        // التحقق من الصلاحيات
        if (!auth()->user()->can('create_invoices')) {
            return response()->json(['error' => 'غير مصرح لك بإنشاء الفواتير'], 403);
        }

        // التحقق من البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // تحديد الفترة الزمنية
        $startDate = $request->has('start_date') ? Carbon::parse($request->start_date) : Carbon::now()->subDays(7);
        $endDate = $request->has('end_date') ? Carbon::parse($request->end_date) : Carbon::now();

        // إنشاء الفواتير
        $stats = $this->maintenanceFinancialService->generateInvoicesFromCompletedMaintenance(
            $startDate,
            $endDate
        );

        return response()->json([
            'message' => 'تم إنشاء الفواتير بنجاح',
            'stats' => $stats,
        ]);
    }
}
