<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\InvoiceInstallment;
use App\Models\Invoice;
use App\Models\Payment;
use App\Helpers\InstallmentHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class InvoiceInstallmentController extends Controller
{
    /**
     * عرض قائمة الأقساط.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = auth()->user()->company_id;

        $query = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })->with(['invoice', 'invoice.customer']);

        // تطبيق الفلاتر
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        if ($request->has('customer_id')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->where('customer_id', $request->customer_id);
            });
        }

        if ($request->has('due_date_from')) {
            $query->where('due_date', '>=', $request->due_date_from);
        }

        if ($request->has('due_date_to')) {
            $query->where('due_date', '<=', $request->due_date_to);
        }

        // الترتيب
        $query->orderBy('due_date', 'asc');

        // التصفح
        $perPage = $request->has('per_page') ? (int) $request->per_page : 15;
        $installments = $query->paginate($perPage);

        return response()->json([
            'status' => 'success',
            'data' => $installments,
        ]);
    }

    /**
     * إنشاء قسط جديد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_id' => 'required|exists:invoices,id',
            'amount' => 'required|numeric|min:0.01',
            'due_date' => 'required|date',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        // التحقق من أن الفاتورة تنتمي للشركة الحالية
        $invoice = Invoice::findOrFail($request->invoice_id);
        if ($invoice->company_id != auth()->user()->company_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'ليس لديك صلاحية الوصول لهذه الفاتورة',
            ], 403);
        }

        // إنشاء القسط
        $installment = new InvoiceInstallment();
        $installment->invoice_id = $request->invoice_id;
        $installment->amount = $request->amount;
        $installment->paid_amount = 0;
        $installment->due_date = $request->due_date;
        $installment->status = 'pending';
        $installment->notes = $request->notes;
        $installment->save();

        // تحديث حالة الفاتورة
        $invoice->payment_type = 'installments';
        if ($invoice->status != 'overdue') {
            $invoice->status = 'partially_paid';
        }
        $invoice->save();

        return response()->json([
            'status' => 'success',
            'message' => 'تم إنشاء القسط بنجاح',
            'data' => $installment->load('invoice'),
        ], 201);
    }

    /**
     * عرض تفاصيل قسط محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $installment = InvoiceInstallment::with([
            'invoice',
            'invoice.customer',
            'payments'
        ])->findOrFail($id);

        // التحقق من أن القسط ينتمي للشركة الحالية
        if ($installment->invoice->company_id != auth()->user()->company_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'ليس لديك صلاحية الوصول لهذا القسط',
            ], 403);
        }

        return response()->json([
            'status' => 'success',
            'data' => $installment,
        ]);
    }

    /**
     * تحديث قسط محدد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'due_date' => 'required|date',
            'status' => 'required|in:pending,paid,overdue,cancelled',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $installment = InvoiceInstallment::findOrFail($id);

        // التحقق من أن القسط ينتمي للشركة الحالية
        if ($installment->invoice->company_id != auth()->user()->company_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'ليس لديك صلاحية الوصول لهذا القسط',
            ], 403);
        }

        // تحديث القسط
        $installment->amount = $request->amount;
        $installment->due_date = $request->due_date;
        $installment->status = $request->status;
        $installment->notes = $request->notes;
        $installment->save();

        return response()->json([
            'status' => 'success',
            'message' => 'تم تحديث القسط بنجاح',
            'data' => $installment->load('invoice'),
        ]);
    }

    /**
     * حذف قسط محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $installment = InvoiceInstallment::findOrFail($id);

        // التحقق من أن القسط ينتمي للشركة الحالية
        if ($installment->invoice->company_id != auth()->user()->company_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'ليس لديك صلاحية الوصول لهذا القسط',
            ], 403);
        }

        // التحقق من عدم وجود مدفوعات مرتبطة بالقسط
        if ($installment->payments()->count() > 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'لا يمكن حذف القسط لوجود مدفوعات مرتبطة به',
            ], 400);
        }

        $installment->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'تم حذف القسط بنجاح',
        ]);
    }

    /**
     * تسجيل دفعة لقسط محدد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function recordPayment(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method_id' => 'required|exists:payment_methods,id',
            'reference_number' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $installment = InvoiceInstallment::findOrFail($id);

        // التحقق من أن القسط ينتمي للشركة الحالية
        if ($installment->invoice->company_id != auth()->user()->company_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'ليس لديك صلاحية الوصول لهذا القسط',
            ], 403);
        }

        // التحقق من أن المبلغ المدفوع لا يتجاوز المبلغ المتبقي
        $remainingAmount = $installment->amount - $installment->paid_amount;
        if ($request->amount > $remainingAmount) {
            return response()->json([
                'status' => 'error',
                'message' => 'المبلغ المدفوع يتجاوز المبلغ المتبقي للقسط',
            ], 400);
        }

        DB::beginTransaction();

        try {
            // إنشاء سجل الدفعة
            $payment = new Payment([
                'company_id' => auth()->user()->company_id,
                'invoice_id' => $installment->invoice_id,
                'payment_method_id' => $request->payment_method_id,
                'amount' => $request->amount,
                'payment_date' => $request->payment_date,
                'reference_number' => $request->reference_number,
                'notes' => $request->notes,
                'status' => 'completed',
                'created_by' => auth()->id(),
            ]);

            // ربط الدفعة بالقسط
            $installment->payments()->save($payment);

            // تحديث المبلغ المدفوع للقسط
            $installment->paid_amount += $request->amount;

            // تحديث حالة القسط
            if ($installment->paid_amount >= $installment->amount) {
                $installment->status = 'paid';
            } elseif ($installment->status === 'overdue') {
                $installment->status = 'partially_paid';
            }

            $installment->save();

            // تحديث المبالغ المدفوعة والمستحقة للفاتورة
            $invoice = $installment->invoice;
            $invoice->paid_amount += $request->amount;
            $invoice->due_amount -= $request->amount;

            // تحديث حالة الفاتورة
            if ($invoice->due_amount <= 0) {
                $invoice->status = 'paid';
            } else {
                $invoice->status = 'partially_paid';
            }

            $invoice->save();

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'تم تسجيل الدفعة بنجاح',
                'data' => [
                    'installment' => $installment->fresh(),
                    'payment' => $payment,
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status' => 'error',
                'message' => 'حدث خطأ أثناء تسجيل الدفعة',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * الحصول على الأقساط المستحقة قريباً.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function upcoming(Request $request)
    {
        $companyId = auth()->user()->company_id;
        $days = $request->has('days') ? (int) $request->days : 30;
        $limit = $request->has('limit') ? (int) $request->limit : 100;

        $installments = InstallmentHelper::getUpcomingInstallments($companyId, $days, $limit);

        return response()->json([
            'status' => 'success',
            'data' => $installments,
        ]);
    }

    /**
     * الحصول على الأقساط المتأخرة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function overdue(Request $request)
    {
        $companyId = auth()->user()->company_id;
        $limit = $request->has('limit') ? (int) $request->limit : 100;

        $installments = InstallmentHelper::getOverdueInstallments($companyId, $limit);

        return response()->json([
            'status' => 'success',
            'data' => $installments,
        ]);
    }

    /**
     * الحصول على إحصائيات الأقساط.
     *
     * @return \Illuminate\Http\Response
     */
    public function stats()
    {
        $companyId = auth()->user()->company_id;
        $stats = InstallmentHelper::getInstallmentStats($companyId);

        return response()->json([
            'status' => 'success',
            'data' => $stats,
        ]);
    }

    /**
     * إنشاء أقساط تلقائية للفاتورة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function createInstallmentsForInvoice(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_id' => 'required|exists:invoices,id',
            'number_of_installments' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'interval' => 'required|in:day,week,month',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        // التحقق من أن الفاتورة تنتمي للشركة الحالية
        $invoice = Invoice::findOrFail($request->invoice_id);
        if ($invoice->company_id != auth()->user()->company_id) {
            return response()->json([
                'status' => 'error',
                'message' => 'ليس لديك صلاحية الوصول لهذه الفاتورة',
            ], 403);
        }

        // إنشاء الأقساط
        $result = InstallmentHelper::createInstallmentsForInvoice(
            $invoice,
            $request->number_of_installments,
            $request->start_date,
            $request->interval,
            $request->notes
        );

        if ($result['success']) {
            return response()->json([
                'status' => 'success',
                'message' => $result['message'],
                'data' => $result['installments'],
            ]);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => $result['message'],
            ], 400);
        }
    }
}
